package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.38.2 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.38.2) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````Q!@``#0```"\D0````0`!30`(``)
M`"@`'``;``$``'`4C0``%(T``!2-``!P````<`````0````$````!@```#0`
M```T````-````"`!```@`0``!`````0````#````5`$``%0!``!4`0``&0``
M`!D````$`````0````$```````````````````"(C0``B(T```4```````$`
M`0```+"-``"PC0$`L(T!`)@"``"L`@``!@```````0`"````N(T``+B-`0"X
MC0$`&`$``!@!```&````!`````0```!P`0``<`$``'`!``!$````1`````0`
M```$````4>5T9```````````````````````````!@```!````!2Y71DL(T`
M`+"-`0"PC0$`4`(``%`"```$`````0```"]L:6(O;&0M;&EN=7@M87)M:&8N
M<V\N,P`````$````%`````,```!'3E4`5TNU%EQBW-:87&)YXQODB0.7G_D$
M````$`````$```!'3E4```````,````"`````````#L```!-````"`````@`
M```!D!`$)4`$@"B```6@AE"$@($``@LH!(PX@!)!B@``"`````!-````3P``
M`%````````````````````!2`````````%,```!4````50``````````````
M5@```%<`````````````````````````6````%D```!:`````````%L`````
M``````````````!<````70```%X```````````````````!?````8```````
M``!B`````````&,`````````9```````````````````````````````````
M`&4```````````````````!F`````````&<```````````````````!I````
M?JF<FUGOEWEK"4/6=E0\E"'_Y`X-.I<+P^4_+(D'<-/9<5@<K4OCP&L:GE#5
M9-1\F4_#FR_%F[8S8MOM:W^:?`&O-.B[XY)\LP&+;T+ZUX=#1=7L9Z_,9A6,
M)<RCX/M)S]Y'+>-'Q@'@IP(*;\7?'[[:5QO[H6Q@````````````````````
M```````,$P````````,`"P```````)`!```````#`!<`U@(`````````````
M$@```!D!`````````````!(```!.`@`````````````2````@`(`````````
M````$@```#4"`````````````!(```!C`0`````````````2````X@``````
M````````$@````H#`````````````!(```#X```````````````2`````0``
M````````````(````*D!`````````````!(```":```````````````2````
MPP``````````````$@```/L!`````````````!(````R`0`````````````2
M````<`(`````````````$@```!```````````````"````#M````````````
M```2````"@(`````````````$@```!X!`````````````!(```"D`0``````
M```````2````SP(`````````````$@```"P``````````````"````!Y`@``
M```````````2````0`$`````````````$@```((!`````````````!8```"*
M```````````````2````1@``````````````$@```+(``````````````!(`
M``"C```````````````2````)P(`````````````$@```.T!````````````
M`!(```!I`@`````````````2````[P,`````````````$@```",!````````
M`````!(```#>`@`````````````1````Y@$`````````````$@```/@"````
M`````````!$```"0`@`````````````2````"P$`````````````$@```!@"
M`````````````!(````N`@`````````````2````5P``````````````$@``
M`(@"`````````````!(```![`0`````````````2````O@(`````````````
M$@```-$!`````````````!(```#T`0`````````````2````(0(`````````
M````$@```'@"`````````````!(```"W`0`````````````2````Y@(`````
M````````$@```-T"`````````````"$```!.`0`````````````2````L`$`
M````````````$@```&\!`````````````!(```"U`@`````````````2````
MW`(`````````````$0```&("`````````````!$```!N```````````````1
M````1`(`````````````$@```-,``````````````!$```!4`@``````````
M```2`````@(`````````````$@```&,``````````````!(````\`@``````
M```````2````>P``````````````$@```+\!`````````````!(````A`P``
M```````````2````$0(`````````````$@```-\!`````````````!(```"P
M`@`````````````2````GP(`````````````$@```)4!`````````````"(`
M``#=`P``="\``"0````2``T`000``%R0`0``````$``8`!,$````D`$`````
M`!``%P`@!```2)`!```````0`!@`E00``.@B``"0`@``$@`-`!4$````D`$`
M`````"``%P`Y!```?"$``%0````2``T`H@0``$20`0`$````$0`7`%4$``!(
MD`$``````!``&``$!```^$H```0````1``\`TP,``'!)``!L`0``$@`-`(T#
M``"\-P``U`(``!(`#0"L!```/"T``'0````2``T`;P,``.0B```$````$@`-
M`!D$``#$&````````!(`#0#,`0``\!4``-0"```2``T`2@0``%R0`0``````
M$``8`&H$``!<D`$``````!``&`"C`P``G"P``*`````2``T`+@0``%A(```8
M`0``$@`-`'P$``!(D`$``````!``%P#'`P``J"4``+`!```2``T`800```1(
M``!4````$@`-`+0$``!8)P``1`4``!(`#0!O!```S#```/`&```2``T`@`,`
M`'@E```P````$@`-`+$#``#0(0``%`$``!(`#0#X`P``D#H``#P!```2``T`
M200``%R0`0``````$``8`(,$``"8+P``-`$``!(`#0``7U]G;6]N7W-T87)T
M7U\`7TE435]D97)E9VES=&5R5$U#;&]N951A8FQE`%])5$U?<F5G:7-T97)4
M34-L;VYE5&%B;&4`4&5R;%]A=&9O<FM?;&]C:P!097)L7VUG7W-E=`!097)L
M7VYE=UA3`%!,7V1O7W5N9'5M<`!097)L7W-Y<U]I;FET,P!B;V]T7T1Y;F%,
M;V%D97(`<&5R;%]R=6X`<&5R;%]C;VYS=')U8W0`4&5R;%]N97=84U]F;&%G
M<P!097)L7V=V7V9E=&-H<'8`4$Q?;65M;W)Y7W=R87``<&5R;%]P87)S90!P
M97)L7V%L;&]C`%!E<FQ?871F;W)K7W5N;&]C:P!P97)L7V1E<W1R=6-T`'!E
M<FQ?9G)E90!097)L7W-V7W-E='!V;@!097)L7W-V7W-E='!V`%!E<FQ?<WES
M7W1E<FT`4&5R;%]C<F]A:U]N;V-O;G1E>'0`4&5R;%]G971?<W8`4&5R;%]S
M869E<WES;6%L;&]C`%!,7V-U<G)E;G1?8V]N=&5X=`!?7V-X85]F:6YA;&EZ
M90!R96%D`'-T<F-H<@!G971P:60`;65M;6]V90!?7VQI8F-?<W1A<G1?;6%I
M;@!?7V9P<FEN=&9?8VAK`'-T<F-M<`!O<&5N-C0`=6YL:6YK`'-T<G1O:P!M
M96US970`<W1R;F-M<`!G971U:60`;65M8VAR`&-L;W-E9&ER`')M9&ER`'-T
M<FQE;@!S=')D=7``;65M8VUP`')E86QL;V,`<F5A9&1I<C8T`&%B;W)T`%]?
M<W!R:6YT9E]C:&L`<W1D97)R`&UE;6-P>0!S=')R8VAR`&QS=&%T-C0`;W!E
M;F1I<@!L<V5E:S8T`%]?=F9P<FEN=&9?8VAK`%]?97)R;F]?;&]C871I;VX`
M97AI=`!R96%D;&EN:P!?7W-T86-K7V-H:U]F86EL`&%C8V5S<P!M:V1I<@!?
M7V5N=FER;VX`7U]R96=I<W1E<E]A=&9O<FL`7U]S=&%C:U]C:&M?9W5A<F0`
M7U]A96%B:5]U;G=I;F1?8W!P7W!R,`!?7V%E86)I7W5N=VEN9%]C<'!?<'(Q
M`&QI8G!E<FPN<V\`;&EB8RYS;RXV`&QD+6QI;G5X+6%R;6AF+G-O+C,`;&EB
M9V-C7W,N<V\N,0!P87)?8W5R<F5N=%]E>&5C`'!A<E]B87-E;F%M90!84U])
M;G1E<FYA;'-?4$%27T)/3U0`<&%R7V5N=E]C;&5A;@!P87)?8W5R<F5N=%]E
M>&5C7W!R;V,`<&%R7V1I<FYA;64`<VAA7V9I;F%L`&=E=%]U<V5R;F%M95]F
M<F]M7V=E='!W=6ED`'!A<E]C;&5A;G5P`%])3U]S=&1I;E]U<V5D`%]?9&%T
M85]S=&%R=`!?7V)S<U]S=&%R=%]?`'-H85]U<&1A=&4`>'-?:6YI=`!?7V5N
M9%]?`%]?8G-S7V5N9%]?`%]?8G-S7W-T87)T`'-H85]I;FET`%]E;F0`<&%R
M7VUK=&UP9&ER`%]E9&%T80!P87)?<V5T=7!?;&EB<&%T:`!P87)?9FEN9'!R
M;V<`;7E?<&%R7W!L`'!A<E]D:64`<&%R7VEN:71?96YV`$=,24)#7S(N-`!'
M0T-?,RXU`$=,24)#7S(N,S0`1TQ)0D-?,BXS,P`O=7-R+VQI8B]P97)L-2\U
M+C,X+V-O<F5?<&5R;"]#3U)%``````````(``0`"``(``@`!``$``P`!``$`
M`@`!``$``@`!``(``0`!``(``@`"``(``0`$``$``0`!``$``0`!``(``@`"
M``(``0`"``(`!0`"``$``@`"``$``@`"``(``@`"``(`!``"``(``@`!``(`
M`0`"``(``@`!``(``0`"``(``0`"``$`!@`#``(``@`"``(``@`!``$``0`!
M``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$`
M`0`!``$``0````$``0!-`P``$````"`````4:6D-```%`,$$`````````0`!
M`&$#```0````(````%4F>0L```,`RP0````````!``,`0P,``!``````````
MM)&6!@``!@#3!```$````+.1E@8```0`W@0``!`````4:6D-```"`,$$````
M````L(T!`!<```"TC0$`%P```/R/`0`7````!)`!`!<````4D`$`%P```!B0
M`0`7````()`!`!<````DD`$`%P```"B0`0`7````+)`!`!<````PD`$`%P``
M`#B0`0`7````/)`!`!<```!$D`$`%P```,B/`0`5"P``S(\!`!4,``#0CP$`
M%1,``-2/`0`5&0``V(\!`!,<``#<CP$`%1T``."/`0`5'@``Y(\!`!4F``#H
MCP$`%2@``.R/`0`5/0``\(\!`!4^``#TCP$`%4```/B/`0`53```W(X!`!8#
M``#@C@$`%@0``.2.`0`6!0``Z(X!`!8&``#LC@$`%@<``/".`0`6"```](X!
M`!8)``#XC@$`%@P``/R.`0`6#0```(\!`!8.```$CP$`%@\```B/`0`6$```
M#(\!`!81```0CP$`%A(``!2/`0`6%```&(\!`!85```<CP$`%A8``""/`0`6
M%P``)(\!`!88```HCP$`%AH``"R/`0`6&P``,(\!`!8?```TCP$`%B```#B/
M`0`6(0``/(\!`!8B``!`CP$`%B,``$2/`0`6)```2(\!`!8E``!,CP$`%B<`
M`%"/`0`6*0``5(\!`!8J``!8CP$`%BL``%R/`0`6+```8(\!`!8M``!DCP$`
M%BX``&B/`0`6+P``;(\!`!8P``!PCP$`%C$``'2/`0`6,@``>(\!`!8S``!\
MCP$`%C0``("/`0`6-0``A(\!`!8V``"(CP$`%C@``(R/`0`6.0``D(\!`!8Z
M``"4CP$`%CL``)B/`0`6/P``G(\!`!9!``"@CP$`%D(``*2/`0`60P``J(\!
M`!9$``"LCP$`%D4``+"/`0`61@``M(\!`!9(``"XCP$`%DD``+R/`0`62@``
MP(\!`!9+``#$CP$`%DP```A`+>E\`0#K"("]Z`3@+>4$X)_E#N"/X`CPON6H
M>P$``,:/XA?*C.*H^[SE`,:/XA?*C.*@^[SE`,:/XA?*C.*8^[SE`,:/XA?*
MC.*0^[SE`,:/XA?*C.*(^[SE`,:/XA?*C.*`^[SE`,:/XA?*C.)X^[SE`,:/
MXA?*C.)P^[SE`,:/XA?*C.)H^[SE`,:/XA?*C.)@^[SE`,:/XA?*C.)8^[SE
M`,:/XA?*C.)0^[SE`,:/XA?*C.)(^[SE`,:/XA?*C.)`^[SE`,:/XA?*C.(X
M^[SE`,:/XA?*C.(P^[SE`,:/XA?*C.(H^[SE`,:/XA?*C.(@^[SE`,:/XA?*
MC.(8^[SE`,:/XA?*C.(0^[SE`,:/XA?*C.((^[SE`,:/XA?*C.(`^[SE`,:/
MXA?*C.+X^KSE`,:/XA?*C.+P^KSE`,:/XA?*C.+H^KSE`,:/XA?*C.+@^KSE
M`,:/XA?*C.+8^KSE`,:/XA?*C.+0^KSE`,:/XA?*C.+(^KSE`,:/XA?*C.+`
M^KSE`,:/XA?*C.*X^KSE`,:/XA?*C.*P^KSE`,:/XA?*C.*H^KSE`,:/XA?*
MC.*@^KSE`,:/XA?*C.*8^KSE`,:/XA?*C.*0^KSE`,:/XA?*C.*(^KSE`,:/
MXA?*C.*`^KSE`,:/XA?*C.)X^KSE`,:/XA?*C.)P^KSE`,:/XA?*C.)H^KSE
M`,:/XA?*C.)@^KSE`,:/XA?*C.)8^KSE`,:/XA?*C.)0^KSE`,:/XA?*C.)(
M^KSE`,:/XA?*C.)`^KSE`,:/XA?*C.(X^KSE`,:/XA?*C.(P^KSE`,:/XA?*
MC.(H^KSE`,:/XA?*C.(@^KSE`,:/XA?*C.(8^KSE`,:/XA?*C.(0^KSE`,:/
MXA?*C.((^KSE`,:/XA?*C.(`^KSE`,:/XA?*C.+X^;SE`,:/XA?*C.+P^;SE
M`,:/XA?*C.+H^;SE`,:/XA?*C.+@^;SE`,:/XA?*C.+8^;SE<$`MZ0'*H.,,
MP$W@R`^,Y63"G^4HT$WB8#*?Y0S`C^!<0I_E'`"-Y1P`C>(8$(WE&!"-XA0@
MC>44((WB!$"/X`,PG.<`,)/E)#"-Y0`PH./5___K+"*?Y2PRG^4"()3G#""-
MY0(0H.$#,)3G$#"-Y0,`H.$<#0#K$#*?Y0,PE.<`4-/E``!5XU(```IP+QWN
M_#&?Y1P`G>4#4(#B!`"`X@,PG^<'`7#C`R"2YZ8STN4#,(/CIC/"Y6,``(H`
M`:#AI/__Z\P1G^7,(9_E`#"@X0$0C^`80)WE`B"/X``0D>4$`(+E``"4Y0@0
M@^6L$9_E'""=Y0$0C^`,$(/EH!&?Y0$`4N,``(/E`1"/X`00@^54``#:!!"$
MX@S`@^("083@!."1Y`3@K.4!`%3A^___&@,0@N)L09_E``"@XP``C>4%(*#A
M!$"/X`$!@^=8$9_E#`"4Y0$0C^`)___K``!0XRD```I$49_E(&"-X@50C^`,
M`)7E2O__ZP!`H.$P`9_E!A"@X0``C^"N``#K``!0XR````H<,9_E`S"/X`P`
MD^7H_O_K(/__ZPPAG^7,,)_E`B"/X`,PDN<`()/E)#"=Y0(@,^``,*#C*@``
M&@0`H.$HT(WB<("]Z`'__^O<()_E``!0XP(@C^`,`(+E'P``"A/__^O(,)_E
M<"\=[@,PG^<#,)+G=E;#Y:#__^H,`)3EY/[_Z]+__^H(`)7E``!0XP$```J=
M"`#KV?__ZI0`G^4&$*#A``"/X((``.L``%#C"`"%Y=+__PKU___J>#"?Y7@`
MG^4``(_@`S"4YPPPC>4#$*#A-?__ZP00H..P___J`0"@XUC__^L;___KO'@!
M`!@!``"<>`$`^````!`!```@`0``0'D!`'QY`0!\>0$`Q#<``*PW```0>0$`
MA`D``.AX`0#4-```O'@!`"AW`0!P>`$`Y'<!`,`S```D`0``H#4```"PH.,`
MX*#C!!"=Y`T@H.$$("WE!``MY1R@G^48,(_B`Z"*X``PH.,$,"WE#`"?Y0``
MFN<J___KD/[_Z]!U`0`L`0``%#"?Y10@G^4#,(_@`B"3YP``4N,>_R\!EO[_
MZKAU`0#\````+`"?Y2PPG^4``(_@*""?Y0,PC^```%/A`B"/X![_+P$8,)_E
M`S"2YP``4^,>_R\!$_\OX0QW`0`$=P$`A'4!```!```X`)_E.#"?Y0``C^`T
M()_E`S"/X``P0^`"((_@HQ^@X4,1@>#!$+#A'O\O`1@PG^4#,)+G``!3XQ[_
M+P$3_R_AR'8!`,!V`0!`=0$`!`$``%@PG^58()_E`S"/X`(@C^``,-/E``!3
MXQ[_+Q$00"WI`<J@XPS`3>#X#XSE-#"?Y0,PDN<``%/C`@``"B@PG^4#`)_G
M]O[_Z\?__^L<,)_E`2"@XP,PC^``(,/E$("]Z'AV`0#\=`$`*`$``/QU`0`L
M=@$`S?__ZO!'+>D!RJ#C#,!-X.`/C.7(,)_E`(!0X@,PC^`K```*O""?Y0(P
MD^<`D)/E``!9XR@```H`4-CE`7"@X0``5>,]`%43`5"@$P!0H`,1```:`$"9
MY0``5.,,```*"6"@X00`H.$%(*#A"!"@X;/^_^L``%#C`@``&@4PU.<]`%/C
M"P``"@1`MN4``%3C\___&@0`H.'PA[WH"%"@X0$P]>4``%/C/0!3$_O__QH(
M4$7@YO__Z@E@1N`!4(7B!4"$X$9AH.$$`*#A`&"'Y?"'O>@(0*#A[?__Z@E`
MH.'K___J>'0!`!0!```00"WI`<J@XPS`3>"(#XSEC""?Y7#03>*(,)_E#1"@
MX0(@C^``0*#A`S"2YP`PD^5L,(WE`#"@XVO^_^L``%#C``"@$P0``!H0,)WE
M#SH#X@H*4^,!"5,3"@``"D@@G^5`,)_E`B"/X`,PDN<`()/E;#"=Y0(@,^``
M,*#C!P``&G#0C>(0@+WH`A"@XP0`H.$3_O_K$`]OX:`"H.'N___J1?[_ZXAS
M`0`8`0``1',!`/!`+>D!RJ#C#,!-X'`/C.5<(9_E?-!-XE@QG^4`4%#B`B"/
MX`,PDN<`,)/E=#"-Y0`PH.,"```*`##5Y0``4^,*```:,"&?Y2@QG^4"((_@
M`S"2YP`@D^5T,)WE`B`SX``PH.-!```:?-"-XO"`O>C"_?_K`$!0XO'__PH$
M`*#A0O[_ZP``4.,J```*$S#0Y1-@@.(N`%/C`@``&@$PUN4``%/C]/__"A,P
MT.4N`%/C)0``"@4`H.'L_?_K`'"@X08`H.'I_?_K``"'X`(`@.(*_O_KI#"?
MY0!PH.$!$*#C`S"/X``@X.,$8(WE`%"-Y2G^_^L($(WB!P"@X<_]_^L!`'#C
M!P"@X0,```H8,)WE#SH#X@$)4^,2```*UOW_ZP<`H.&\_?_K!`"@X1?^_^L`
M`%#CU/__&@0`H.'C_?_K!0"@X?G]_^N]___J`3#6Y2X`4^/6__\:`C#6Y0``
M4^/%__\*TO__ZJ3__^OK___JY?W_Z]AR`0`8`0``K'(!`-0N``#P02WI`<J@
MXPS`3>```(SE(``,Y0%PH.$`$9_E`#&?Y0':3>(!$(_@"-!-X@!@H.$!"HWB
M!`"`X@2`C>(#,)'G_R\`XP@0H.$`,)/E`#"`Y0`PH./,`)_E``"/X.?]_^L`
M0%#B!P``V@A0C>(!,$3B`\"%X`0@7.4``%+C%P``&@``4^,4```:G#"?Y7`/
M'>X'(*#A!A"@X0,PG^<#`)#G</W_ZX0@G^4!&HWB<#"?Y000@>("((_@`S"2
MYP`@D^4`,)'E`B`SX``PH.,2```:`=J-X@C0C>+P@;WH`T"@X00@H.$O$*#C
M"`"@X=O]_^L``%#CX___"C0`G^5PSQWN!#"@X0@@H.$&$*#A``"?YP``G.>`
M_?_KX?__ZIG]_^M0<0$`&`$``.`M``#D<0$`P'`!`'!Q`0#P1RWI`<J@XPS`
M3>#8#XSE%"*?Y0C03>(0,I_E`4"@X0(@C^``4*#A!(*?Y0,PDN<(@(_@`#"3
MY00PC>4`,*#C`##1Y3T`4^,!0($"!`"@X5?]_^L-$*#A`&"@X04`H.'1_O_K
M`)!0XD````I0_?_K``!6X3<``)JX,9_E`'"=Y0.`F.<'<:#A`##5Y0``4^,]
M`%,3`3"@$P`PH`,%```*!3"@X0$@\^4``%+C/0!2$_O__QH%,$/@`@"&X@!@
MF.4#`(#@7OW_ZP``4.,'`(;G%0``"@`PF.4%`*#A!Q"3YP$PT.0`,,'E``!3
MXST`4Q,&```*`2"!X@$PT.0"$*#A`3#"Y#T`4^,``%,3^?__&@$@H.$],*#C
M`##!Y0$PU.0!,.+E``!3X_O__QH((9_E^#"?Y0(@C^`#,)+G`""3Y00PG>4"
M(#/@`#"@XS4``!H(T(WB\(>]Z`&02>(!,-3D`3#IY0``4^/[__\:[O__ZL`P
MG^4#@)CG`'"8Y0`PE^4``%/C)```"@<PH.$$(+/E"1"@X0&0B>(``%+C^O__
M&@,0@>(!$:#AD#"?Y0,PC^``()/E``!2XP\``!H!`*#A`2"@XP`@@^4=_?_K
M`*!0XM7__PH'$*#A"7&@X0<@H.$I_?_K`*"(Y0"0C>4!D(GB`#"@XPDQBN>B
M___J!P"@X3_]_^L`H*#A``!0XP``B.7%__\*"7&@X?+__^H#D*#A"!"@X]__
M_^H(_?_K''`!`!@!```,<`$`%`$```AO`0`,<`$`$$`MZ0'*H.,,P$W@Z`^,
MY7`0G^5P3QWN;""?Y1#03>)H`)_E`B"/X&0PG^4!$)_G`S"/X``@DN<!`)3G
M5!"?Y0P@C>4!$(_@&?W_ZT@0G^5(,)_E`,"@XT0@G^4#,(_@`1"?YP(@C^`!
M`)3G-!"?Y03`C>4!$(_@`#"-Y;7\_^L0T(WB$("]Z-1N`0#4;0$`#`$``$1K
M``"(*@``H&X!`'0M``"`%@``="H``#@`G^4(T$WB-""?Y7#/'>XP$)_E`B"/
MX"PPG^4``)_G`S"/X`$@DN<@$)_E``"<YP$0C^`$((WE"-"-XO'\_^HX;@$`
M.&T!``P!``"H:@``\"D``'!`+>D!RJ#C#,!-X```C.4!RDSB``",Y2@`#.7<
M()_E`MI-XM@PG^48T$WB`B"/X`(:C>(4$('B#$"-X@%JC>(#,)+G$&"&X@`P
MD^4`,('E`#"@X\3\_^NH$)_E`2`!XZ0PG^4!$(_@\`#-X0,PC^`!$*#C!`"@
MX<?\_^L$`*#A`2J@XP80H.&]_/_K`$!0XA@``+H!`(3BF/S_ZP!04.(4```*
M!""@X080H.%U_/_K`#"@XP0PQ>=0()_E`AJ-XCPPG^44$('B`B"/X`,PDN<`
M()/E`#"1Y0(@,^``,*#C`P``&@4`H.$"VHWB&-"-XG"`O>B$_/_K`%"@X^W_
M_^K,;`$`&`$``*@I``"4*0``,&P!`+G__^KP3RWI`<J@XPS`3>```(SEJ``,
MY40BG^5"W4WB0#*?Y0303>("((_@`&"@X30"G^5!S8WB$("-XCS`C.(#,)+G
M`4"@X0``C^`($*#A`#"3Y0`PC.4`,*#CO/W_ZR\0H.,`4*#A!@"@X0W\_^L`
M`%#C$@``"O`!G^4&$*#A``"/X,?^_^L&`*#AX"&?Y4$=C>+,,9_E/!"!X@(@
MC^`#,)+G`""3Y0`PD>4"(#/@`#"@XVD``!I"W8WB!-"-XO"/O>@$`*#A/OS_
MZZ01G^4!$(_@3?S_ZP!`4.+E__\*E)&?Y8"@C>(`,-3E!'!*X@F0C^```%/C
M.P``"@``5>,$```*!!"@X04`H.%R_/_K``!0XRL```H$`*#A#OS_ZP$P0.(#
M,(3@`+"@X0,`5.$`,(0P`!"@,P$`A#(#```Z"```Z@,`4.$`$,/E`@``"@$@
M<^4O`%+C^?__"@0`H.']^__K`+"@X08`H.'Z^__K"P"`X`$`@.(!"E#CO?__
M*O@`G^4),*#A`2J@XP$0H.,``(_@`$"-Y4$`C>D'`*#A./S_ZP@0H.$'`*#A
MWOO_ZP``4.,#```:8#`:Y0\Z`^(""5/C%0``"K00G^4``*#C`1"/X`W\_^L`
M0%#BI?__"@`PU.4``%/CP___&@``5>,6```*`##5Y2X`4^,#```*@$"?Y0&P
MH.,$0(_@U?__Z@$PU>4``%/CZO__"O?__^H!$*#C!P"@X;S[_^L``%#CY/__
M&E``G^4'$*#A``"/X%C^_^L'`*#AWOO_ZX[__^HX0)_E`;"@XP1`C^#!___J
MY/O_Z[QK`0`8`0``N"@``(PH``!(:P$`3"@``#`H``"8)P``3"<``/0F``#0
M)@``I"8``!!`+>D!RJ#C#,!-X/@/C.4O$*#C`$"@X8S[_^L``%#C`4"`$@0`
MH.$00+WHP/O_ZO!`+>D!RJ#C#,!-X```C.4@``SE>"&?Y0':3>)T,9_E#-!-
MX@(@C^`!&HWB!!"!X@!04.(#,)+G`#"3Y0`P@>4`,*#C,0``"@`PU>4``%/C
M+@``"H_[_^L!((#B`$"@X0$*4N,Z``"*!'"-X@40H.$'`*#AC?O_ZP$P1.(#
M,(?@`P!7X0D``"H$`(?@!2"-X@`0H.,"``#J`@!0X0`0P.4"```*`3!PY2\`
M4^/Y__\*+Q"@XP<`H.%8^__K``!0XR4```H'`%#A`#"@@P`PP(4'`*#ABOO_
MZ[P@G^4!&HWBL#"?Y000@>("((_@`S"2YP`@D^4`,)'E`B`SX``PH.,B```:
M`=J-X@S0C>+P@+WHB""?Y0$:C>)X,)_E!!"!X@(@C^`#,)+G`""3Y0`PD>4"
M(#/@`#"@XQ0``!I@`)_E``"/X`':C>(,T(WB\$"]Z&O[_^H``*#CWO__ZD0@
MG^4!&HWB+#"?Y000@>("((_@`S"2YP`@D^4`,)'E`B`SX``PH.,@`)\%``"/
M`.S__PIG^__K_&@!`!@!```X:`$``&@!`/@D``"T9P$`L"0``'!`+>D!RJ#C
M#,!-X.@/C.7$))_E"-!-XL`TG^4-0*#A`B"/X+ADG^6X5)_E!F"/X`,PDN<%
M4(_@`#"3Y00PC>4`,*#C!!"@X08`H.&C_/_K``!0XPX```J,-)_E``"=Y0,P
ME><`$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___
M&NO__^I49)_E!F"/X`00H.$&`*#AC?S_ZP``4.,.```*-#2?Y0``G>4#,)7G
M`!"3Y0`Q@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK
M___J`&2?Y09@C^`$$*#A!@"@X7?\_^L``%#C#@``"MPSG^4``)WE`S"5YP`0
MD^4`,8'@!""3Y0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___
MZJQCG^4&8(_@!!"@X08`H.%A_/_K``!0XPX```J$,Y_E``"=Y0,PE><`$)/E
M`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^I8
M8Y_E!F"/X`00H.$&`*#A2_S_ZP``4.,.```*+#.?Y0``G>4#,)7G`!"3Y0`Q
M@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J!&.?
MY09@C^`$$*#A!@"@X37\_^L``%#C#@``"M0RG^4``)WE`S"5YP`0D^4`,8'@
M!""3Y0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZK!BG^4&
M8(_@!!"@X08`H.$?_/_K``!0XPX```I\,I_E``"=Y0,PE><`$)/E`#&!X`0@
MD^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^I<8I_E!F"/
MX`00H.$&`*#A"?S_ZP``4.,.```*)#*?Y0``G>4#,)7G`!"3Y0`Q@>`$()/E
M`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J"&*?Y09@C^`$
M$*#A!@"@X?/[_^L``%#C#@``"LPQG^4``)WE`S"5YP`0D^4`,8'@!""3Y0`A
M@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZK1AG^4&8(_@!!"@
MX08`H.'=^__K``!0XPX```IT,9_E``"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G
M``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^I@89_E!F"/X`00H.$&
M`*#AQ_O_ZP``4.,.```*'#&?Y0``G>4#,)7G`!"3Y0`Q@>`$()/E`"&!YP``
M4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J#`&?Y000H.$``(_@LOO_
MZP`04.("```*^`"?Y0``C^#"_/_K\`"?Y000H.$``(_@J?O_ZP`04.("```*
MW`"?Y0``C^"Y_/_KU`"?Y000H.$``(_@H/O_ZP`04.(2```*P`"?Y0``C^"P
M_/_KN""?Y60PG^4"((_@`S"2YP`@D^4$,)WE`B`SX``PH.,0```:F!"?Y9@`
MG^4!$(_@``"/X`C0C>)P0+WHH/S_ZH0`G^4$$*#A``"/X(?[_^L`$%#BZ?__
M"G``G^4``(_@E_S_Z^7__^HI^O_K4&<!`!@!``",)```/&<!`!0!```L)```
MW",``)`C``!`(P``^"(``#PB``!4(@``""(``+PA``#H(```%"$``.@@```$
M(0``!"$````A```D(```^&(!`-P@``!`(```K"```#`@```$X"WE`<J@XPS`
M3>#P#XSE>""?Y0S03>)T,)_E#1"@X0(@C^!L`)_E``"/X`,PDN<`,)/E!#"-
MY0`PH.-6^__K``!0XP4```H``-#E``!0XS``4!,!`*`#``"@$P$`(.(T()_E
M*#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@XP$``!H,T(WB!/"=Y.OY_^L,
M8@$`&`$``)`?``#$80$`#P`MZ0'*H.,$X"WE#,!-X/`/C.5($)_E#-!-XD0@
MG^44,(WB`1"/X#P`G^4`,(WE.,"?Y0``C^`"()'G`1"@XP`@DN4$((WE`""@
MXQ`@G>4,P)#G``"<Y;OY_^O_`*#C"OK_ZVAA`0`8`0``6&$!`!P!``#P1RWI
M`,"@XP'HH.,,P$W@#N!,X`'*3.(``(SE#@!<X?O__QHX``[E`=A-XAC03>)T
MP9_E`A"@XP'HC>(,P(_@%.".X@`0C>4`(*#C7!&?Y0`PH.,`@*#A`1"<YP`0
MD>4`$([E`!"@XZGY_^L!`%#B`'#!XL<OH.$","#@`C!3X',P_^8","/@`C!3
MX`(@PN`#8%#@`G#'X```5^,X``"Z")"@XP&00.,,4(WB`!"@XP8@H.$',*#A
M"`"@X0`0C>63^?_K`0!QXP$`<`,R```*"2"@X040H.$(`*#A6?G_ZP$`<.,I
M```*!P!0XQ\``)H(`$#B`$"5X!P``"JPH)_E"J"/X`(``.H!0$3B!0!4X18`
M`#H((*#C"A"@X00`H.$A^?_K``!0X_;__QH%0$3@!@"4X,0?I^!X()_E`<B-
MXF@PG^44P(SB`B"/X`,PDN<`()/E`#"<Y0(@,^``,*#C$```&@'8C>(8T(WB
M\(>]Z`%H5N(`<,?B`0!WXP$(=@/)__\:``#@XP`0H.'I___J)`"?Y0``C^![
M___K'`"?Y0``C^!X___K7_G_Z]Q@`0`8`0``4&$!`-!?`0"@'0``A!T``!!`
M+>D!RJ#C#,!-X/@/C.45^?_K-?G_ZP``4.,``)`5$("]Z/!`+>D!RJ#C#,!-
MX-`/C.7\()_E'-!-XO@PG^40$(WB`B"/X/!@G^4`4*#A!F"/X`,PDN<&`*#A
M`#"3Y10PC>4`,*#CE?K_ZP!`4.("```*`##4Y0``4^,.```:O""?Y;`PG^4"
M((_@`S"2YP`@D^44,)WE`B`SX``PH.,C```:G`"?Y040H.$``(_@'-"-XO!`
MO>B6^__J!0"@X0'Y_^L`<*#A!`"@X?[X_^L``(?@`@"`XA_Y_^MH,)_E:""?
MY0$0H.,#,(_@"$"-Y0(@C^``4(WE!""-Y0`@X.,`0*#A._G_ZT0@G^4H,)_E
M`B"/X`,PDN<`()/E%#"=Y0(@,^``,*#C!!"@`08`H`'>__\*"_G_ZQ!?`0`8
M`0``+!T``,Q>`0#0'```J!L``)P;``!$7@$`\$\MZ0'*H.,,P$W@Z`Z,Y50F
MG^7TT$WB4#:?Y8A@C>("((_@2,:?Y0"@H.&@0(WB#,"/X`3@H.$#,)+G&'",
MXI10C>(LMI_E`#"3Y>PPC>4`,*#C##",XC#`C.(@@(WB"["/X`<`D^@'`(;H
M#P"WZ`\`KN@#`)?H`P".Z`<`G.@'`(7H\`6?Y0@0H.$``(_@-?K_ZP!P4.("
M```*`##7Y0``4^,7`0`:F?C_Z[GX_^L``%#C>@``"@!PD.4``%?C=P``"@<`
MH.&I^/_K@`"@X0$`@.+*^/_K`##7Y0!0H.$<`(WE``!3XPL```J,E9_E"9"/
MX`4`H.$`,(WE`1"@XPDPH.$`(.#C`E"%XN/X_^L!,/?E``!3X_7__QI@!9_E
M``"/X`(``.H$`+3E``!0XPT```H($*#A"_K_ZP!04.+X__\*`##5Y0``4^/U
M__\*/_K_ZP``4./R__\*!0"@X9[X_^L`0%#B#0``&@1`EN0``%3C"@``"@`P
MU.4``%/CSP``"@0`H.$Q^O_K``!0X_7__PH$`*#AD/C_ZP!`4.+Q__\*!`"@
MX7'X_^L<4)WE`&"@X04`H.%M^/_K`&"&X`%KAN($8(;B!@"@X8SX_^NP-)_E
M`'"@X0$0H.,#,(_@"#"-Y:`TG^4`(.#C#%"-Y0,PC^`$,(WED#2?Y0!`C>4#
M,(_@I?C_ZP<=H.,'`*#A$OC_ZP$`<.,#```:NOC_ZP`0D.41`%'CJP``&@@0
MH.$'`*#A@?C_ZP!04.(M```*3!2?Y0`PFN5())_E`0";YP(@C^``<(WE`1"@
MXP``D.5M^/_K`&"@XRPDG^7\,Y_E`B"/X`,PDN<`()/E[#"=Y0(@,^``,*#C
M\P``&@8`H.'TT(WB\(^]Z%#X_^L`<%#BB0``&@0`E>0``%#C"0``"@@0H.&L
M^?_K`#!0XOC__PH`,-/E``!3X_+__QH$`)7D``!0X_7__QK`<Y_E#0"@XT?X
M_^M3,*#C`%"@X0=PC^`<`(WE?/__ZC`PG>4/.@/B`0E3X\W__QHX0)WE`/C_
MZP``5.')__\:,#"=Y5,PZ.<'#5/CQ?__&@8`H.$T^/_K`&"@X6@#G^4($*#A
M``"/X(GY_^L`$%#B?```"@``FN6Q^__K`)!0XG@```H)`*#A`_C_ZP!`H.$#
M`%#C#```FC`3G^4$`$#B``")X`$0C^!;^/_K``!0XP4``!H$`%3CH@``"@1`
MB>`%,%3E+P!3XYX```H``Y_E"!"@X0``C^!M^?_K``!0XP,```H`,-#E,`!3
MXP``4Q,$```:`!"@XPD`H.'U]__K`$!0XED``!K($I_ER`*?Y0$0C^#$HI_E
M``"/X,"2G^5P^O_KO(*?Y1KX_^NX,I_E"`"-Y0$0H.,#,(_@##"-Y:@RG^4&
M`*#A`'"-Y0`@X.,#,(_@!#"-Y90RG^4*H(_@"9"/X`B`C^`#,(_@%?C_ZPX`
M`.HN^/_K`#"0Y1$`4^,0```:`_C_ZP%0A>(`P*#A"C"@X00@H.$!$*#C!@"@
MX1"0C>4,4(WE@!&-Z`7X_^L'':#C!@"@X7+W_^L`0*#A`0!PX^K__PH'`*#A
MG??_ZR`"G^4&$*#A``"/X$'Z_^L&`*#A@O[_ZVC__^H#0*#A-?__ZJCW_^N`
M`*#A`0"`XOW^_^JP(9_E`#":Y0(`F^<$$(WE`1"@XP!PC>78(9_E``"0Y0(@
MC^#%]__K5O__ZF_^_^O$(9_E6#&?Y0(@C^`#,)+G`""3Y>PPG>4"(#/@`#"@
MXTH``!H'`*#A]-"-XO!/O>BG]__J`)":Y83__^KE_?_K`("@X0"04>*A__^Z
M"B!0X@"@H.,`,,GB!`"@X;BPC>(`H(WEH/?_ZP8@H.,+$*#A!`"@X6GW_^L&
M`%#C-```&K@@G>5#/*#C03-$XP,`4N&/__\:M"#;X4@U!.,#`%+AB___&C(@
M6.($`*#A`##)XL!0C>(`H(WEB_?_ZP0`H.$H(*#C!1"@X53W_^LH`%#C'P``
M&O0PG^4!$*#C\""?Y08`H.$#,(_@!#"-Y>0PG^4"((_@"%"-Y0P@C>4#,(_@
M`'"-Y0`@X./HH,WEH/?_ZY___^H$0)KE``!4XUW__PH$`*#A3_?_ZP,`4.-9
M__^:I!"?Y00`0.(``(3@`1"/X*CW_^L``%#C!)"@`5'__^IK]__KA`"?Y0``
MC^"`_?_KW%T!`!@!```$7P$`G%T!`(P:``!,&P``M!H``'0:``!0&0``1!H`
M`!P!``!$&@``C%L!`$@9``"L&0``=!D``-P7```<&```F!<``/@8``"<&0``
M!!<``,09```8%P``W!@``&@6``!<%P``Z%@!`%`5``#8%P``!!<``+`6``"<
M%@``\$`MZ0'*H.,,P$W@X`^,Y5P2G^4,T$WB6"*?Y7!/'>X!$(_@4#*?Y5`"
MG^4-8*#A`B"1YP``C^`&$*#A`""2Y00@C>4`(*#C`S"?YP-0E.=D,-7E93#%
MY8?X_^L``%#C`@``"@`PT.4Q`%/C9@``"LG[_^L(,I_E`S"?YP-0E.<`$I_E
M!0"@X0,PH.,!(*#C`1"/X-/V_^L`4%#B#```"N0QG^4,()7E`S"?YP`0DN78
M(9_E`P"4YP(@C^#/]O_K##"5Y0`0D^4(,)'E`043XUD``!JX$9_E`S"@XP$@
MH.,!$)_G`0"4YZ@1G^4!$(_@O/;_ZP!04.(+```*F`&?Y080H.$``(_@6_C_
MZP`@4.)!```*A#&?Y0P0E>4#,)_G`!"1Y0,`E.>T]O_K<#&?Y0$@H.-L$9_E
M;'&?Y0$0C^`#,)_G!W"/X`-0E.<%`*#AE?;_ZU0AG^4`$*#A!0"@X0(@C^"E
M]O_K1#&?Y40!G^4`(*#C!A"@X0``C^`#,)_G`S"4YV4@P^4[^/_K`$"@X0``
M4.,(`(?E$@``"@<=H.,$`*#A<?;_ZP$@@.(`,*#A$0!0XP$`4A,E``"*^""?
MY;`PG^4"((_@`S"2YP`@D^4$,)WE`B`SX``PH.,;```:#-"-XO"`O>@$`)?E
MQ_W_ZP!`H.$``%#C"`"'Y>;__QJX`)_E``"/X-GV_^L!,-#E``!3XYG__PJ4
M___JH#"?Y0P@E>4#,(_@``"2Y00PD^4`$)/EU_C_Z[O__^J$,)_E`S"?YP,`
ME.>K]O_KH/__ZK+V_^MP$)_E!""@X6P`G^4!$(_@``"/X`00D>4`$)'EO_;_
MZ^Q6`0`8`0``S%<!`$@4``"<5P$`P!4``&Q7`0"<%0``-%<!`'`5```P$P``
M^%8!`-16`0`D%0``1%<!`!@5``"85@$`L!(``$A5`0!\%```9%8!`-15`0`H
M5@$`0!0``#!`+>D!RJ#C#,!-X.@/C.4,(9_E`$"@X0@QG^4,T$WB`B"/X``!
MG^4-$*#A``"/X`,PDN<`,)/E!#"-Y0`PH./8]__K`#!0X@H```H`,-/E,`!3
MXP``4Q,!,*`#`#"@$P$P(^(``%3C`#"@`P$P`Q(``%/C"@``&K`@G^6D,)_E
M`B"/X`,PDN<`()/E!#"=Y0(@,^``,*#C(```&@S0C>(P@+WH`##4Y0``4^/Q
M__\*!`"@X9?Z_^LO$*#C`%"@X1SV_^L``%#C`5"`$@4`H.%1]O_K6!"?Y00@
MH.,!$(_@@/;_ZP``4./B__\:1""?Y3`PG^4"((_@`S"2YP`@D^4$,)WE`B`S
MX``PH.,#```:!`"@X0S0C>(P0+WH!_C_ZDGV_^L85`$`&`$``)@1``"X4P$`
ML!$``$13`0``````\$\MZ0'*H.,,P$W@@`Z,Y1!,G^57WTWB#.R?Y1`PC>($
M0(_@`Q"@X1P@@.)<P(#B#N"4YP#@GN54X8WE`."@XXT*8O3@`/#S`@!<X=T*
M0?3Z__\:!DO=[0@[W>T1'HWB"AO=[0```.JR$6+R""O3[30@@^(`"]/M"#"#
MX@$`4^&/5V+TL`%"\[4!0/.T`4#SLT%C\K$Q8?*@&&#R,`#A\[`!8?(."\/M
M[O__&B!`D.F9.0?C@CI%XPQ`D.40P)WE`&"0Y000+N`0<)#E`\",X`40`>!E
MD:#A!\",X`00(>`4()WEYLV,X`'`C.`)$"[@`R""X`80`>!F@:#A!"""X!B@
MG>4.$"'@`A"!X`@@*>`#H(K@#"`"X`Z@BN`)("+@"B""X!R@G>7L'8'@;,&@
MX0.@BN`)H(K@#)`HX`&0">#A+8+@")`IX&$1H.$*D(G@(*"=Y>*=B>`#H(K@
M"*"*X`&`+.`"@`C@8B&@X0R`*.`"L"'@"H"(X"2@G>4)L`O@Z8V(X`.@BN`,
MH(K@:<&@X0&0*^`,L"+@"I")X"B@G>4(L`O@Z)V)X`.@BN`!H(K@:!&@X0*`
M*^`!L"S@"H"(X"R@G>4)L`O@Z8V(X`.@BN`"H(K@:2&@X0R0*^`"L"'@"I")
MX#"@G>4(L`O@Z)V)X`.@BN`,H(K@:,&@X0&`*^`,L"+@"H"(X#2@G>4)L`O@
MZ8V(X`.@BN`!H(K@:1&@X0*0*^`!L"S@"I")X#B@G>4(L`O@Z)V)X`.@BN`"
MH(K@:"&@X0R`*^`"L"'@"H"(X#R@G>4)L`O@Z8V(X`.@BN`,H(K@:<&@X0&0
M*^`,L"+@"I")X$"@G>4(L`O@Z)V)X`.@BN`!H(K@:!&@X0*`*^`!L"S@"H"(
MX$2@G>4)L`O@Z8V(X`.@BN`"H(K@:2&@X0R0*^`"L"'@"I")X$B@G>4(L`O@
MZ)V)X`.@BN`,H(K@:,&@X0&`*^`,L"+@"H"(X$R@G>4)L`O@Z8V(X`.@BN`!
MH(K@:1&@X0*0*^`!L"S@"I")X%"@G>4(L`O@Z)V)X`.@BN`"H(K@:"&@X0R`
M*^`*@(C@5*"=Y>F-B.`#H(K@#,"*X`*@(>`)H`K@:9&@X0&@*N`,H(K@6,"=
MY>BMBN`#P(S@`<",X`D0(N`($`'@:(&@X0(0(>`,$('@7,"=Y>H=@>`#,(S@
M",`IX`K`#.`",(/@"<`LX&`@G>4#P(S@H3L.X]D^1N-JH:#A`R""X.'-C.`)
MD(+@9""=Y0,@@N`(((+@"H`HX`&`*.`(@(G@89&@X6@0G>7LC8C@`Q"!X`H0
M@>`)H"K@#*`JX&S!H.$*((+@Z*V"X&P@G>4#((+@"2""X`R0*>`(D"G@:(&@
MX0D0@>#JG8'@<!"=Y0,0@>`,$('@",`LX`K`+.!JH:#A#"""X.G-@N!T()WE
M`R""X`@@@N`*@"C@"8`HX&F1H.$($('@[(V!X'@0G>4#$('@"A"!X`F@*N`,
MH"K@;,&@X0H@@N#HK8+@?""=Y0,@@N`)((+@#)`IX`B0*>!H@:#A"1"!X.J=
M@>"`$)WE`Q"!X`P0@>`(P"S@"L`LX&JAH.$,((+@Z<V"X(0@G>4#((+@""""
MX`J`*.`)@"C@:9&@X0@0@>#LC8'@B!"=Y0,0@>`*$('@":`JX`R@*N!LP:#A
M"B""X.BM@N",()WE`R""X&JQH.$)((+@#)`IX`B0*>!H@:#A"1"!X.J=@>"0
M$)WE`Q"!X`P0@>`(P"S@"L`LX`P@@N#IK8+@E""=Y0,@@N`(((+@"X`HX`G`
M*.!ID:#A#!"!X)C`G>7JC8'@"1`KX`/`C.`*$"'@"\",X)RPG>5JH:#A`2""
MX`.PB^#H'8+@";"+X`J0*>`(("G@:(&@X0+`C."@()WEX9V,X`,@@N`*((+@
M"*`JX`'`*N"DH)WE81&@X0S`B^`#H(K@Z<V,X`B@BN`!@"C@"8`HX&F1H.$(
M((+@L+"=Y>R-@N"H()WE`R""X`$@@N`)$"'@#!`AX&S!H.$!H(K@#!`IX.BM
MBN`($"'@`2""X&B!H.'J'8+@K""=Y0,@@N`(,"S@"C`CX`D@@N`#((+@W#P+
MXQL_2.-JD:#A`["+X`F@B.$,L(O@M,"=Y>$M@N`!H`K@`\",X&$1H.$(P(S@
M"8`(X`B@BN$!@(GA"Z"*X`*`".`!L`G@XJV*X`N`B.&\L)WE#("(X+C`G>5B
M(:#A`["+X`/`C.`!L(O@"9",X`+`@>$*P`S@`A`!X`'`C.'JC8C@"<",X,"0
MG>5JH:#AZ,V,X`H0@N$#D(G@`I")X`@0`>`*(`+@:(&@X0(0@>$((`K@"["!
MX`@0BN$,$`'@[+V+X`(0@>%LP:#A"1"!X,20G>4,((CAZQV!X`.0B>`+(`+@
M"I")X,B@G>5KL:#A`Z"*X`B@BN`,@`C@""""X6&!H.$)D(+@"R",X>&=B>`!
M(`+@S!"=Y0,0@>`,$('@"\`,X`S`@N$(((OA"L",X`D@`N`(H`O@Z<V,X`H@
M@N'0H)WE:9&@X0$0@N`#H(K@"2"(X0N@BN#4L)WE#"`"X.P=@>`#L(O@;,&@
MX0BPB^`)@`C@""""X6&!H.$*H(+@#"")X>&MBN`!(`+@V!"=Y0,0@>`)$('@
M#)`)X`F0@N$(((SA"Y")X`H@`N`(L`S@ZIV)X`L@@N'@L)WE`1""X-P@G>5J
MH:#A`["+X`,@@N`(L(O@#,""X`H@B.$)(`+@"H`(X.D=@>`(@(+A:9&@X0R`
MB.`)((KA8<&@X>&-B.`!(`+@Y!"=Y0,0@>`*$('@":`*X`J@@N$,((GA"Z"*
MX`@@`N`,L`G@Z*V*X`L@@N'LL)WE`1""X.@@G>5H@:#A`["+X`,@@N`,L(O@
M"9""X`@@C.$*(`+@",`,X.H=@>`,((+A:J&@X0F0@N`*((CA8<&@X>&=B>`!
M(`+@\!"=Y0,0@>`($('@"H`(X`B`@N$,((KA"X"(X`D@`N`,L`K@Z8V(X`L@
M@N'XL)WE`2""X/00G>5ID:#A`["+X`,0@>`,L(O@"J"!X`D0C.$($`'@"<`,
MX`S`@>'H+8+@"L",X/R@G>5H@:#AXLV,X`@0B>$#,(K@"3"#X`(0`>`(D`G@
M8B&@X0*@B.$)$('A"Q"!X`R@"N`"L`C@[!V!X&R1H.$+H(KA`,&=Y0.@BN#6
M,0SC8CI,XP/`C.#AK8K@"(",X`3!G>5AL:#A`\",X`+`C.`)("+@`2`BX`@1
MG>4"@(C@"R`IX.J-B.`*("+@`Q"!X&JAH.$)$('@`L",X`J0*^#HS8S@")`I
MX`PAG>4)$('@:(&@X>R=@>`0$9WE`R""X`,0@>`+((+@"A"!X`B@*N`,H"K@
M;,&@X0H@@N#IK8+@%"&=Y0,@@N`(((+@#(`HX`F`*.!ID:#A"!"!X.J-@>`8
M$9WE`Q"!X`P0@>`)P"S@"L`LX&JAH.$,((+@Z,V"X!PAG>4#((+@"2""X`J0
M*>`(D"G@:(&@X0D0@>#LG8'@(!&=Y0,0@>`*$('@"*`JX`R@*N!LP:#A"B""
MX.FM@N`D(9WE`R""X`@@@N`,@"C@"8`HX&F1H.$($('@ZHV!X"@1G>4#$('@
M#!"!X`G`+.`*P"S@:J&@X0P@@N#HS8+@+"&=Y0,@@N`)((+@"I`IX`B0*>!H
M@:#A"1"!X.R=@>`P$9WE`Q"!X`H0@>`(H"K@#*`JX&S!H.$*((+@Z:V"X#0A
MG>4#((+@""""X`R`*.`)@"C@:9&@X0@0@>#JC8'@.!&=Y0,0@>`,$('@"<`L
MX`K`+.!JH:#A#"""X.C-@N`\(9WE`R""X`D@@N`*D"G@")`IX&B!H.$)$('@
M0)&=Y>P=@>`#D(G@"I")X`B@*N`,H"K@;,&@X0H@@N!$H9WEX2V"X`.@BN`(
MH(K@#(`HX`&`*.!A$:#A")")X$B!G>7BG8G@`X"(X`R`B.`!P"S@`L`LX&(A
MH.$,P(K@`G"'X.G-C.`0<(#E;.&.X`C@C>5,X9WE`S".X`8P@^`!,(/@`A`A
MX`D0(>!ID:#A`1"(X`D@(N#L'8'@#"`BX`(P@^`)0(3@X3V#X`Q`C>4`,(WE
M`5"%X#P@G^4$4(WEWPIM]`(@C^`H,)_ECPI`]`,PDN<`()/E5#&=Y0(@,^``
M,*#C`0``&E??C>+PC[WH.//_Z]A2`0`8`0```$<!`!!`+>D!RJ#C#,!-X/@/
MC.5@`*#C+//_ZP<+W^T(*]_M"3O?[0`@H.,$"\#M&""`Y5P@@.7?*D#T$("]
MZ/#ATL,``````2-%9XFKS>_^W+J8=E0R$/!'+>D!RJ#C#,!-X.`/C.44,)#E
M`F"@X8(AH.$`4*#A`C"3X!@@D.44,(#E`4"@X5P`D.4!(((BIBZ"X!R`A>(`
M`%#C&""%Y2```!H_`%;C+@``VD"01N(I8Z#A`7"&X@=SA.`$$*#A"#"@X4!`
MA.(`X)'E$!"!X@S`$>40,(/B"``1Y00@$>4$`%'A$.`#Y0S``^4(``/E!"`#
MY?/__QH!0*#A!0"@X;+\_^L'`%3AZ___&@9C2>`&(*#A!Q"@X0@`H.'-\O_K
M7&"%Y?"'O>A`<&#B``"(X`8`5^$&<*"A!R"@X<7R_^M<,)7E`S"'X%PPA>5`
M`%/C\(>]&`4`H.$'8$;@!T"$X)K\_^O.___J!'"@X>?__^IP0"WI`<J@XP*+
M+>T,P$W@Z`^,Y0%`H.$4$)'E`%"@X1Q@A.(Q/[_F%!"4Y1@`E.71$>7G,"^_
MYA@K0^P!(('B?S#@XS@`4N,"`(;@`3#&YT$``-I`(&+B`!"@XW?R_^L$`*#A
M??S_ZRPPA.)0`,#R#PI&]`\*0_0\,(3B#PI#]$PPA.(/!T/T5#"$X@0`H.$/
MAP/T<?S_ZP,PU.4`,,7E!`"@X;(PU.$!,,7E`#"4Y2,TH.$",,7E`#"4Y0,P
MQ>4',-3E!##%Y;8PU.$%,,7E!#"4Y2,TH.$&,,7E!#"4Y0<PQ>4+,-3E"##%
MY;HPU.$),,7E"#"4Y2,TH.$*,,7E"#"4Y0LPQ>4/,-3E###%Y;XPU.$-,,7E
M##"4Y2,TH.$.,,7E##"4Y0\PQ>43,-3E$##%Y;(QU.$1,,7E$#"4Y2,TH.$2
M,,7E$#"4Y1,PQ>4"B[WL<$"]Z$?R_^HX(&+B`!"@XS7R_^O&___J"#"?Y0,P
MC^``,)/EC?+_ZAQ%`0`(0"WI"("]Z`$``@`]````0`````L```!`````$```
M`$`````F````0````#4```!``````P```$`````&````0````!0```!`````
M#@```$`````-````0````"4```!`````$@```$`````H````0`````(```!`
M````&````$`````)````0`````H```!````````````````E<R\E<P```"]P
M<F]C+W-E;&8O97AE``!$>6YA3&]A9&5R.CIB;V]T7T1Y;F%,;V%D97(`26YT
M97)N86QS.CI005(Z.D)/3U0`````+W!R;V,O)6DO)7,`97AE`"X```!005)?
M5$5-4`````!005)?4%)/1TY!344`````.@```"5S)7,E<P``+P```%!%4DPU
M3$E"`````%!%4DQ,24(`4$523#5/4%0`````4$523$E/``!005)?24Y)5$E!
M3$E:140`4$%27U-005=.140`4$%27T-,14%.````4$%27T1%0E5'````4$%2
M7T-!0TA%````4$%27T=,3T)!3%]$14)51P````!005)?1TQ/0D%,7U1-4$1)
M4@```%!!4E]435!$25(``%!!4E]'3$]"04Q?5$5-4`!005)?1TQ/0D%,7T-,
M14%.`````#$```!L<V5E:R!F86EL960`````<F5A9"!F86EL960`3$1?3$E"
M4D%265]0051(`%-94U1%30``)3`R>``````E<R5S)7,E<P````!P87(M````
M`"5S.B!C<F5A=&EO;B!O9B!P<FEV871E('-U8F1I<F5C=&]R>2`E<R!F86EL
M960@*&5R<FYO/25I*0H``"5S.B!P<FEV871E('-U8F1I<F5C=&]R>2`E<R!I
M<R!U;G-A9F4@*'!L96%S92!R96UO=F4@:70@86YD(')E=')Y('EO=7(@;W!E
M<F%T:6]N*0H``%!!5$@`````<&%R;``````N<&%R`````'-H;W)T(')E860`
M`"5S)7-C86-H92TE<R5S```E<R5S=&5M<"TE=25S````)7,E<W1E;7`M)74M
M)74E<P`````8````<&5R;``````P````4$%2.CI086-K97(Z.E9%4E-)3TX`
M````,2XP-C,```!5;F%B;&4@=&\@8W)E871E(&-A8VAE(&1I<F5C=&]R>0``
M```E<SH@8W)E871I;VX@;V8@<')I=F%T92!C86-H92!S=6)D:7)E8W1O<GD@
M)7,@9F%I;&5D("AE<G)N;STE:2D*`````"UE```M+0``55-%4@````!54T52
M3D%-10````!414U01$E2`%1-4``O=&UP`````&5V86P@)V5X96,@+W5S<B]B
M:6XO<&5R;"`@+5,@)#`@)'LQ*R(D0")])PII9B`P.R`C(&YO="!R=6YN:6YG
M('5N9&5R('-O;64@<VAE;&P*<&%C:V%G92!?7W!A<E]P;#L*;7D@*"1005)?
M34%'24,L("1&24Q%7V]F9G-E=%]S:7IE+"`D8V%C:&5?;F%M95]S:7IE*3L*
M0D5'24X@>PHD4$%27TU!1TE#(#T@(EQN4$%2+G!M7&XB.PHD1DE,15]O9F9S
M971?<VEZ92`](#0[("`@(R!P86-K*").(BD*)&-A8VAE7VYA;65?<VEZ92`]
M(#0P.PHD4$M:25!?34%'24,@/2`B4$M<,#`S7#`P-"(["GT*<W5B(&9I;F1?
M<&%R7VUA9VEC"GL*;7D@*"1F:"D@/2!`7SL*;7D@)&-H=6YK7W-I>F4@/2`V
M-"`J(#$P,C0["FUY("1B=68["FUY("1S:7IE(#T@+7,@)&9H.PIM>2`D<&]S
M(#T@*"1S:7IE+3$I("T@*"1S:7IE+3$I("4@)&-H=6YK7W-I>F4[("`@("`@
M(R!.3U1%.B`D<&]S(&ES(&$@;75L=&EP;&4@;V8@)&-H=6YK7W-I>F4*=VAI
M;&4@*"1P;W,@/CT@,"D@>PIS965K("1F:"P@)'!O<RP@,#L*<F5A9"`D9F@L
M("1B=68L("1C:'5N:U]S:7IE("L@;&5N9W1H*"1005)?34%'24,I.PII9B`H
M*&UY("1I(#T@<FEN9&5X*"1B=68L("1005)?34%'24,I*2`^/2`P*2!["G)E
M='5R;B`D<&]S("L@)&D["GT*)'!O<R`M/2`D8VAU;FM?<VEZ93L*?0IR971U
M<FX@+3$["GT*;7D@*"1P87)?=&5M<"P@)'!R;V=N86UE+"!`=&UP9FEL92P@
M)4UO9'5L94-A8VAE*3L*14Y$('L@:68@*"1%3E9[4$%27T-,14%.?2D@>PIR
M97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL93HZ0F%S96YA;64["G)E
M<75I<F4@1FEL93HZ4W!E8SL*;7D@)'1O<&1I<B`]($9I;&4Z.D)A<V5N86UE
M.CID:7)N86UE*"1P87)?=&5M<"D["F]U=',H<7%;4F5M;W9I;F<@9FEL97,@
M:6X@(B1P87)?=&5M<")=*3L*1FEL93HZ1FEN9#HZ9FEN9&1E<'1H*'-U8B![
M("@@+60@*2`_(')M9&ER(#H@=6YL:6YK('TL("1P87)?=&5M<"D["G)M9&ER
M("1P87)?=&5M<#L*:68@*"UD("1P87)?=&5M<"`F)B`D7D\@;F4@)TU35VEN
M,S(G*2!["FUY("1T;7`@/2!N97<@1FEL93HZ5&5M<"@*5$5-4$Q!5$4@/3X@
M)W1M<%A86%A8)RP*1$E2(#T^($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1T
M;W!D:7(I+`I3549&25@@/3X@)RYC;60G+`I53DQ)3DL@/3X@,"P**3L*;7D@
M)&9I;&5N86UE(#T@)'1M<"T^9FEL96YA;64["G!R:6YT("1T;7`@/#PB+BXN
M(CL*>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@73L@9&\*<FT@+7)F("<D<&%R
M7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M<"<@73L@=&AE;@IB<F5A:PIF
M:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*9&]N90IR;2`G)&9I;&5N86UE
M)PHN+BX*8VQO<V4@)'1M<#L*8VAM;V0@,#<P,"P@)&9I;&5N86UE.PIM>2`D
M8VUD(#T@(B1F:6QE;F%M92`^+V1E=B]N=6QL(#(^)C$@)B(["G-Y<W1E;2@D
M8VUD*3L*;W5T<RAQ<5M3<&%W;F5D(&)A8VMG<F]U;F0@<')O8V5S<R!T;R!P
M97)F;W)M(&-L96%N=7`Z("1F:6QE;F%M95TI.PI]"GT@?0I"14=)3B!["DEN
M=&5R;F%L<SHZ4$%2.CI"3T]4*"D@:68@9&5F:6YE9"`F26YT97)N86QS.CI0
M05(Z.D)/3U0["F5V86P@>PI?<&%R7VEN:71?96YV*"D["FUY("1Q=6EE="`]
M("$D14Y6>U!!4E]$14)51WT["FUY("5S>7,@/2`H"G!A=&A?<V5P("`@(#T^
M("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G.B<I+`I?97AE("`@("`@("`]
M/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y9W=I;BDO(#\@)RYE>&4G(#H@
M)R<I+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^("]>35-7:6Y\3U,R+R`_("=<
M7"<@.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE*"D["F]U=',H<7%;7"1P<F]G
M;F%M92`]("(D<')O9VYA;64B72D["E]S971?<&%R7W1E;7`H*3L*;W5T<RAQ
M<5M<)$5.5GM005)?5$5-4'T@/2`B)$5.5GM005)?5$5-4'TB72D["FUY("@D
M<W1A<G1?<&]S+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N*3L*34%'24,Z('L*
M;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@>WT["G5N;&5S<R`H;W!E;B!?
M1D@L("<\.G)A=R<L("1P<F]G;F%M92D@>PIO=71S*'%Q6T-A;B=T(')E860@
M9G)O;2!F:6QE("(D<')O9VYA;64B72D[("`C(&1O;B=T('5S92`D(2!H97)E
M(&%S(&ET(')E<75I<F5S($5R<FYO+G!M"FQA<W0@34%'24,["GT*;7D@)&UA
M9VEC7W!O<R`](&9I;F1?<&%R7VUA9VEC*"I?1D@I.PII9B`H)&UA9VEC7W!O
M<R`\(#`I('L*;W5T<RAQ<5M#86XG="!F:6YD(&UA9VEC('-T<FEN9R`B)%!!
M4E]-04=)0R(@:6X@9FEL92`B)'!R;V=N86UE(ETI.PIL87-T($U!1TE#.PI]
M"F]U=',H(D9O=6YD(%!!4B!M86=I8R!A="!P;W-I=&EO;B`D;6%G:6-?<&]S
M(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@
M+2!L96YG=&@H(EPP0T%#2$4B*2P@,#L*<F5A9"!?1D@L("1B=68L(&QE;F=T
M:"@B7#!#04-(12(I.PII9B`H)&)U9B!N92`B7#!#04-(12(@)B8@)&)U9B!N
M92`B7#!#3$5!3B(I('L*;W5T<R@B3F\@8V%C:&4@;6%R:V5R(&9O=6YD(BD[
M"FQA<W0@34%'24,["GT*;W5T<RAQ<5M#86-H92!M87)K97(@(B1B=68B(&9O
M=6YD72D["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I
M>F4L(#`["G)E860@7T9(+"`D8G5F+"`D1DE,15]O9F9S971?<VEZ93L*;7D@
M)&]F9G-E="`]('5N<&%C:R@B3B(L("1B=68I.PIO=71S*")/9F9S970@9G)O
M;2!S=&%R="!O9B!&24Q%<R!I<R`D;V9F<V5T(BD["G-E96L@7T9(+"`D;6%G
M:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@+2`D;V9F<V5T+"`P.PHD<W1A
M<G1?;V9?1DE,15]S96-T:6]N(#T@=&5L;"!?1D@["FUY("5R97%U:7)E7VQI
M<W0["G)E860@7T9(+"`D8G5F+"`T.R`@("`@("`@("`@("`@("`@("`@("`@
M("`@(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B"G=H:6QE("@D8G5F(&5Q(")&
M24Q%(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A9"!?1D@L("1B=68L('5N
M<&%C:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA;64@/2`D8G5F.PIO=71S*'%Q
M6U5N<&%C:VEN9R!&24Q%("(D9G5L;&YA;64B+BXN72D["FUY("1C<F,@/2`H
M("1F=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX?2DO?'P@*2`_("0Q(#H@=6YD
M968["FUY("@D8F%S96YA;64L("1E>'0I(#T@*"1B=68@/7X@;7PH/SHN*B\I
M/R@N*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B=68L(#0["G)E860@7T9(+"`D
M8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@*&1E9FEN960H)&5X="D@86YD
M("1E>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI)"]I*2!["FUY("1F:6QE;F%M
M92`](%]S879E7V%S*"(D8W)C)&5X="(L("1B=68L(#`W-34I.PHD4$%2.CI(
M96%V>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@/2`D9FEL96YA;64["B1005(Z
M.DAE879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE?2`]("1F=6QL;F%M93L*?0IE
M;'-E('L*)$UO9'5L94-A8VAE>R1F=6QL;F%M97T@/2!["F)U9B`]/B`D8G5F
M+`IC<F,@/3X@)&-R8RP*;F%M92`]/B`D9G5L;&YA;64L"GT["B1R97%U:7)E
M7VQI<W1[)&9U;&QN86UE?2LK.PI]"G)E860@7T9(+"`D8G5F+"`T.PI]"FQO
M8V%L($!)3D,@/2`H<W5B('L*;7D@*"1S96QF+"`D;6]D=6QE*2`]($!?.PIR
M971U<FX@:68@<F5F("1M;V1U;&4@;W(@(21M;V1U;&4["FUY("1I;F9O(#T@
M)$UO9'5L94-A8VAE>R1M;V1U;&5].PIR971U<FX@=6YL97-S("1I;F9O.PID
M96QE=&4@)')E<75I<F5?;&ES='LD;6]D=6QE?3L*)$E.0WLD;6]D=6QE?2`]
M("(O;&]A9&5R+R1I;F9O+R1M;V1U;&4B.PII9B`H)$5.5GM005)?0TQ%04Y]
M(&%N9"!D969I;F5D*"9)3SHZ1FEL93HZ;F5W*2D@>PIM>2`D9F@@/2!)3SHZ
M1FEL92T^;F5W7W1M<&9I;&4@;W(@9&EE(")#86XG="!C<F5A=&4@=&5M<"!F
M:6QE.B`D(2(["B1F:"T^8FEN;6]D92@I.PHD9F@M/G!R:6YT*"1I;F9O+3Y[
M8G5F?2D["B1F:"T^<V5E:R@P+"`P*3L*<F5T=7)N("1F:#L*?0IE;'-E('L*
M;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1I;F9O+3Y[8W)C?2YP;2(L("1I
M;F9O+3Y[8G5F?2D["B1I;F9O+3Y[9FEL97T@/2`D9FEL96YA;64["F]P96X@
M;7D@)&9H+"`G/#IR87<G+"`D9FEL96YA;64@;W(@9&EE('%Q6T-A;B=T(')E
M860@(B1F:6QE;F%M92(Z("0A73L*<F5T=7)N("1F:#L*?0ID:64@(D)O;W1S
M=')A<'!I;F<@9F%I;&5D.B!C86XG="!F:6YD(&UO9'5L92`D;6]D=6QE(2([
M"GTL($!)3D,I.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E(%!!4CHZ2&5A
M=GD["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I<F4@17AP;W)T97(Z.DAE
M879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["G)E<75I<F4@
M24\Z.D9I;&4["G=H:6QE("AM>2`D9FEL96YA;64@/2`H<V]R="!K97ES("5R
M97%U:7)E7VQI<W0I6S!=*2!["G5N;&5S<R`H)$E.0WLD9FEL96YA;65](&]R
M("1F:6QE;F%M92`]?B`O0E-$4$%.+RD@>PII9B`H)&9I;&5N86UE(#U^("]<
M+G!M8S\D+VDI('L*<F5Q=6ER92`D9FEL96YA;64["GT*96QS92!["F1O("1F
M:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE(#U^("]S:71E8W5S=&]M:7IE7"YP
M;"0O.PI]"GT*9&5L971E("1R97%U:7)E7VQI<W1[)&9I;&5N86UE?3L*?0II
M9B`H)&)U9B!N92`D4$M:25!?34%'24,I('L*;W5T<RAQ<5M.;R!Z:7`@9F]U
M;F0@869T97(@1DE,12!S96-T:6]N(&EN(&9I;&4@(B1P<F]G;F%M92)=*3L*
M;&%S="!-04=)0R`["GT*)'-T87)T7W!O<R`]("AT96QL(%]&2"D@+2`T.R`@
M("`@("`@("`@("`@("`C('-T87)T(&]F('II<`I]"FUY($!P87)?87)G<SL*
M;7D@*"1O=70L("1B=6YD;&4L("1L;V=F:"P@)&-A8VAE7VYA;64I.PID96QE
M=&4@)$5.5GM005)?05!07U)%55-%?3L@(R!S86YI=&EZ92`H4D554T4@;6%Y
M(&)E(&$@<V5C=7)I='D@<')O8FQE;2D*:68@*"$D<W1A<G1?<&]S(&]R("@D
M05)'5ELP72!E<2`G+2UP87(M;W!T:6]N<R<@)B8@<VAI9G0I*2!["FUY("5D
M:7-T7V-M9"`]('%W*`IP("`@8FQI8E]T;U]P87(*:2`@(&EN<W1A;&Q?<&%R
M"G4@("!U;FEN<W1A;&Q?<&%R"G,@("!S:6=N7W!A<@IV("`@=F5R:69Y7W!A
M<@HI.PII9B`H0$%21U8@86YD("1!4D=66S!=(&5Q("<M+7)E=7-E)RD@>PIS
M:&EF="!`05)'5CL*)$5.5GM005)?05!07U)%55-%?2`]('-H:69T($!!4D=6
M.PI]"F5L<V4@>R`C(&YO<FUA;"!P87)L(&)E:&%V:6]U<@IM>2!`861D7W1O
M7VEN8SL*=VAI;&4@*$!!4D=6*2!["B1!4D=66S!=(#U^("]>+2A;04E-3T),
M8G%P:75S5'9=*2@N*BDO(&]R(&QA<W0["FEF("@D,2!E<2`G22<I('L*<'5S
M:"!`861D7W1O7VEN8RP@)#(["GT*96QS:68@*"0Q(&5Q("=-)RD@>PIE=F%L
M(")U<V4@)#(B.PI]"F5L<VEF("@D,2!E<2`G02<I('L*=6YS:&EF="!`<&%R
M7V%R9W,L("0R.PI]"F5L<VEF("@D,2!E<2`G3R<I('L*)&]U="`]("0R.PI]
M"F5L<VEF("@D,2!E<2`G8B<I('L*)&)U;F1L92`]("=S:71E)SL*?0IE;'-I
M9B`H)#$@97$@)T(G*2!["B1B=6YD;&4@/2`G86QL)SL*?0IE;'-I9B`H)#$@
M97$@)W$G*2!["B1Q=6EE="`](#$["GT*96QS:68@*"0Q(&5Q("=,)RD@>PIO
M<&5N("1L;V=F:"P@(CX^(BP@)#(@;W(@9&EE('%Q6T-A;B=T(&]P96X@;&]G
M(&9I;&4@(B0R(CH@)"%=.PI]"F5L<VEF("@D,2!E<2`G5"<I('L*)&-A8VAE
M7VYA;64@/2`D,CL*?0IS:&EF="A`05)'5BD["FEF("AM>2`D8VUD(#T@)&1I
M<W1?8VUD>R0Q?2D@>PID96QE=&4@)$5.5GLG4$%27U1%35`G?3L*:6YI=%]I
M;F,H*3L*<F5Q=6ER92!005(Z.D1I<W0["B9[(E!!4CHZ1&ES=#HZ)&-M9")]
M*"D@=6YL97-S($!!4D=6.PHF>R)005(Z.D1I<W0Z.B1C;60B?2@D7RD@9F]R
M($!!4D=6.PIE>&ET.PI]"GT*=6YS:&EF="!`24Y#+"!`861D7W1O7VEN8SL*
M?0I]"FEF("@D;W5T*2!["GL*<F5Q=6ER92!)3SHZ1FEL93L*<F5Q=6ER92!!
M<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T.CI32$$["GT*;7D@)'!A<B`]
M('-H:69T*$!!4D=6*3L*;7D@)'II<#L*:68@*&1E9FEN960@)'!A<BD@>PIO
M<&5N(&UY("1F:"P@)SPZ<F%W)RP@)'!A<B!O<B!D:64@<7%;0V%N)W0@9FEN
M9"!P87(@9FEL92`B)'!A<B(Z("0A73L*8FQE<W,H)&9H+"`G24\Z.D9I;&4G
M*3L*)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH("1Z:7`M/G)E861&<F]M
M1FEL94AA;F1L92@D9F@L("1P87(I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TLH
M*2`I"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H:79E("(D<&%R
M(ET["GT*;7D@)6UE=&%?<&%R(#T@9&\@>PII9B`H)'II<"!A;F0@;7D@)&UE
M=&$@/2`D>FEP+3YC;VYT96YT<R@G345402YY;6PG*2D@>PHD;65T82`]?B!S
M+RXJ7G!A<CHD+R]M<SL*)&UE=&$@/7X@<R]>7%,N*B\O;7,["B1M971A(#U^
M("]>("`H6UXZ72LI.B`H+BLI)"]M9SL*?0I].PII9B`H9&5F:6YE9"`D<&%R
M*2!["F]P96X@;7D@)'!H+"`G/#IR87<G+"`D<&%R(&]R(&1I92!Q<5M#86XG
M="!R96%D('!A<B!F:6QE("(D<&%R(CH@)"%=.PIM>2`D8G5F.PIR96%D("1P
M:"P@)&)U9BP@-#L*9&EE('%Q6R(D<&%R(B!I<R!N;W0@82!P87(@9FEL95T@
M=6YL97-S("1B=68@97$@)%!+6DE07TU!1TE#.PIC;&]S92`D<&@["GT*0W)E
M871E4&%T:"@D;W5T*2`["FUY("1F:"`]($E/.CI&:6QE+3YN97<H"B1O=70L
M"DE/.CI&:6QE.CI/7T-214%4*"D@?"!)3SHZ1FEL93HZ3U]21%=2*"D@?"!)
M3SHZ1FEL93HZ3U]44E5.0R@I+`HP-S<W+`HI(&]R(&1I92!Q<5M#86XG="!C
M<F5A=&4@9FEL92`B)&]U="(Z("0A73L*)&9H+3YB:6YM;V1E*"D["G-E96L@
M7T9(+"`P+"`P.PIM>2`D;&]A9&5R.PII9B`H9&5F:6YE9"`D<W1A<G1?;V9?
M1DE,15]S96-T:6]N*2!["G)E860@7T9(+"`D;&]A9&5R+"`D<W1A<G1?;V9?
M1DE,15]S96-T:6]N.PI](&5L<V4@>PIL;V-A;"`D+R`]('5N9&5F.PHD;&]A
M9&5R(#T@/%]&2#X["GT*:68@*"$D14Y6>U!!4E]615)"051)37T@86YD("1L
M;V%D97(@/7X@+UXH/SHC(7Q<0')E;2DO*2!["G)E<75I<F4@4$%2.CI&:6QT
M97(Z.E!O9%-T<FEP.PI005(Z.D9I;'1E<CHZ4&]D4W1R:7`M/F%P<&QY*%PD
M;&]A9&5R+"`D,"D["GT*)&9H+3YP<FEN="@D;&]A9&5R*0IO<B!D:64@<7%;
M17)R;W(@=W)I=&EN9R!L;V%D97(@=&\@(B1O=70B.B`D(5T["FEF("@D8G5N
M9&QE*2!["G)E<75I<F4@4$%2.CI(96%V>3L*4$%2.CI(96%V>3HZ7VEN:71?
M9'EN86QO861E<B@I.PII;FET7VEN8R@I.PIR97%U:7)E7VUO9'5L97,H*3L*
M;7D@0&EN8R`](&=R97`@>R`A+T)31%!!3B\@?0IG<F5P('L**"1B=6YD;&4@
M;F4@)W-I=&4G*2!O<@HH)%\@;F4@)$-O;F9I9SHZ0V]N9FEG>V%R8VAL:6)E
M>'!](&%N9`HD7R!N92`D0V]N9FEG.CI#;VYF:6=[<')I=FQI8F5X<'TI.PI]
M($!)3D,["G-\+RM\+WQG+"!S?"\D?'P@9F]R96%C:"!`:6YC.PIM>2`E9FEL
M97,["B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,;V%D97(Z.F1L7W-H87)E9%]O
M8FIE8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI8E]E>'0@/2`D0V]N9FEG.CI#
M;VYF:6=[;&EB7V5X='T[("`@("`@("`@(R!86%@@;&EB7V5X="!V<R!D;&5X
M="`_"FUY("5W<FET=&5N.PIF;W)E86-H(&UY("1K97D@*'-O<G0@:V5Y<R`E
M9FEL97,I('L*;7D@*"1F:6QE+"`D;F%M92D["FEF("AD969I;F5D*&UY("1F
M8R`]("1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&ME>7TI*2!["B@D9FEL92P@
M)&YA;64I(#T@*"1K97DL("1F8RD["GT*96QS92!["F9O<F5A8V@@;7D@)&1I
M<B`H0&EN8RD@>PII9B`H)&ME>2`]?B!M?%Y<421D:7)<12\H+BHI)'QI*2![
M"B@D9FEL92P@)&YA;64I(#T@*"1K97DL("0Q*3L*;&%S=#L*?0II9B`H)&ME
M>2`]?B!M?%XO;&]A9&5R+UM>+UTK+R@N*BDD?"D@>PII9B`H;7D@)')E9B`]
M("1-;V1U;&5#86-H97LD,7TI('L**"1F:6QE+"`D;F%M92D@/2`H)')E9BP@
M)#$I.PIL87-T.PI]"FEF("@M9B`B)&1I<B\D,2(I('L**"1F:6QE+"`D;F%M
M92D@/2`H(B1D:7(O)#$B+"`D,2D["FQA<W0["GT*?0I]"GT*;F5X="!U;FQE
M<W,@9&5F:6YE9"`D;F%M93L*;F5X="!I9B`D=W)I='1E;GLD;F%M97TK*SL*
M;F5X="!I9B`A<F5F*"1F:6QE*2!A;F0@)&9I;&4@/7X@+UPN7%$D;&EB7V5X
M=%Q%)"]I.PIO=71S*'-P<FEN=&8H<7%;4&%C:VEN9R!&24Q%("(E<R(N+BY=
M+"!R968@)&9I;&4@/R`D9FEL92T^>VYA;65](#H@)&9I;&4I*3L*;7D@)&-O
M;G1E;G0["FEF("AR968H)&9I;&4I*2!["B1C;VYT96YT(#T@)&9I;&4M/GMB
M=69].PI]"F5L<V4@>PIL;V-A;"`D+R`]('5N9&5F.PIO<&5N(&UY("1T:"P@
M)SPZ<F%W)RP@)&9I;&4@;W(@9&EE('%Q6T-A;B=T(')E860@(B1F:6QE(CH@
M)"%=.PHD8V]N=&5N="`](#PD=&@^.PIC;&]S92`D=&@["E!!4CHZ1FEL=&5R
M.CI0;V13=')I<"T^87!P;'DH7"1C;VYT96YT+"`B/&5M8F5D9&5D/B\D;F%M
M92(I"FEF("$D14Y6>U!!4E]615)"051)37T@86YD("1N86UE(#U^("]<+B@_
M.G!M?&EX?&%L*20O:3L*4$%2.CI&:6QT97(Z.E!A=&-H0V]N=&5N="T^;F5W
M+3YA<'!L>2A<)&-O;G1E;G0L("1F:6QE+"`D;F%M92D["GT*)&9H+3YP<FEN
M="@B1DE,12(L"G!A8VLH)TXG+"!L96YG=&@H)&YA;64I("L@.2DL"G-P<FEN
M=&8H(B4P.'@O)7,B+"!!<F-H:79E.CI::7`Z.F-O;7!U=&5#4D,S,B@D8V]N
M=&5N="DL("1N86UE*2P*<&%C:R@G3B<L(&QE;F=T:"@D8V]N=&5N="DI+`HD
M8V]N=&5N="D*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@96UB961D960@1DE,
M12!T;R`B)&]U="(Z("0A73L*;W5T<RAQ<5M7<FET=&5N(&%S("(D;F%M92)=
M*3L*?0I]"FEF("@D>FEP*2!["B1Z:7`M/G=R:71E5&]&:6QE2&%N9&QE*"1F
M:"D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2PIO<B!D:64@<7%;17)R;W(@=W)I
M=&EN9R!Z:7`@<&%R="!O9B`B)&]U=")=.PI]"FEF("@D;65T85]P87)[8VQE
M86Y]*2!["B1F:"T^<')I;G0H(EPP0TQ%04XB*3L*?0IE;'-E('L*:68@*"%D
M969I;F5D("1C86-H95]N86UE*2!["FUY("1C='@@/2!$:6=E<W0Z.E-(02T^
M;F5W*#$I.PIS965K("1F:"P@,"P@,#L*)&-T>"T^861D9FEL92@D9F@I.PIS
M965K("1F:"P@,"P@,CL*)&-A8VAE7VYA;64@/2`D8W1X+3YH97AD:6=E<W0[
M"GT*)&-A8VAE7VYA;64@/2!S=6)S='(@)&-A8VAE7VYA;64L(#`L("1C86-H
M95]N86UE7W-I>F4["B1C86-H95]N86UE("X](")<,"(@>"`H)&-A8VAE7VYA
M;65?<VEZ92`M(&QE;F=T:"`D8V%C:&5?;F%M92D["B1F:"T^<')I;G0H)&-A
M8VAE7VYA;64L"B)<,$-!0TA%(BD["GT*;7D@)&]F9G-E="`]("1F:"T^=&5L
M;"`M(&QE;F=T:"@D;&]A9&5R*3L*)&9H+3YP<FEN="AP86-K*"=.)RP@)&]F
M9G-E="DL"B1005)?34%'24,I.PHD9F@M/F-L;W-E(&]R(&1I92!Q<5M%<G)O
M<B!W<FET:6YG('1R86EL97(@;V8@(B1O=70B.B`D(5T["F-H;6]D(#`W-34L
M("1O=70["F5X:70["GT*>PIL87-T('5N;&5S<R!D969I;F5D("1S=&%R=%]P
M;W,["E]F:7A?<')O9VYA;64H*3L*<F5Q=6ER92!005(["E!!4CHZ2&5A=GDZ
M.E]I;FET7V1Y;F%L;V%D97(H*3L*>PIR97%U:7)E($9I;&4Z.D9I;F0["G)E
M<75I<F4@07)C:&EV93HZ6FEP.PI]"FUY("1F:"`]($E/.CI&:6QE+3YN97<[
M("`@("`@("`@("`@("`@("`@("`@("`@("`@("`C($%R8VAI=F4Z.EII<"!O
M<&5R871E<R!O;B!A;B!)3SHZ2&%N9&QE"B1F:"T^9F1O<&5N*&9I;&5N;RA?
M1D@I+"`G<B<I(&]R(&1I92!Q<5MF9&]P96XH*2!F86EL960Z("0A73L*07)C
M:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H+7,@7T9(*3L*;7D@)'II<"`]($%R
M8VAI=F4Z.EII<"T^;F5W.PHH)'II<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F
M:"P@)'!R;V=N86UE*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"DI"F]R(&1I
M92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H:79E("(D<')O9VYA;64B73L*
M07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H-C0@*B`Q,#(T*3L*<'5S:"!`
M4$%2.CI,:6)#86-H92P@)'II<#L*)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE
M?2`]("1Z:7`["F]U=',H(D5X=')A8W1I;F<@>FEP+BXN(BD["FEF("AD969I
M;F5D("1%3E9[4$%27U1%35!]*2![(",@<VAO=6QD(&)E('-E="!A="!T:&ES
M('!O:6YT(0IF;W)E86-H(&UY("1M96UB97(@*"`D>FEP+3YM96UB97)S("D@
M>PIN97AT(&EF("1M96UB97(M/FES1&ER96-T;W)Y.PIM>2`D;65M8F5R7VYA
M;64@/2`D;65M8F5R+3YF:6QE3F%M93L*;F5X="!U;FQE<W,@)&UE;6)E<E]N
M86UE(#U^(&U["EX*+S]S:&QI8B\**#\Z)$-O;F9I9SHZ0V]N9FEG>V%R8VAN
M86UE?2\I/PHH6UXO72LI"B0*?7@["FUY("1E>'1R86-T7VYA;64@/2`D,3L*
M;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M/F-A=&9I;&4H)$5.5GM005)?
M5$5-4'TL("1E>'1R86-T7VYA;64I.PII9B`H+68@)&1E<W1?;F%M92`F)B`M
M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E9%-I>F4H*2D@>PIO=71S*'%Q
M6U-K:7!P:6YG("(D;65M8F5R7VYA;64B('-I;F-E(&ET(&%L<F5A9'D@97AI
M<W1S(&%T("(D9&5S=%]N86UE(ETI.PI](&5L<V4@>PIO=71S*'%Q6T5X=')A
M8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D97-T7VYA;64B+BXN72D["B@D
M;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H)&1E<W1?;F%M92D@/3T@07)C
M:&EV93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@<7%;17)R;W(@97AT<F%C=&EN
M9R!Z:7`@;65M8F5R('1O("(D9&5S=%]N86UE(ET["F-H;6]D*#`U-34L("1D
M97-T7VYA;64I(&EF("1>3R!E<2`B:'!U>"(["GT*?0I]"GT*=6YL97-S("@D
M4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2!["F1I92`\/"`B+B(@=6YL97-S
M($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP87(@72!;("U)9&ER(%T@6R`M
M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O9W)A;2YP;"!="B0P(%L@+4)\
M+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*)$5.5GM005)?4%)/1TY!345]
M(#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`05)'5BD["GT*<W5B($-R96%T
M95!A=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E<75I<F4@1FEL93HZ0F%S96YA
M;64["FUY("@D8F%S96YA;64L("1P871H+"`D97AT*2`]($9I;&4Z.D)A<V5N
M86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN*B<I*3L*<F5Q=6ER92!&:6QE
M.CI0871H.PI&:6QE.CI0871H.CIM:W!A=&@H)'!A=&@I('5N;&5S<R@M92`D
M<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E<G)O<@I]"G-U8B!R97%U:7)E
M7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q=6ER92!$>6YA3&]A9&5R.PIR
M97%U:7)E(&EN=&5G97(["G)E<75I<F4@<W1R:6-T.PIR97%U:7)E('=A<FYI
M;F=S.PIR97%U:7)E('9A<G,["G)E<75I<F4@0V%R<#L*<F5Q=6ER92!#87)P
M.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q=6ER92!%>'!O<G1E<CHZ2&5A
M=GD["G)E<75I<F4@17AP;W)T97(["G)E<75I<F4@1F-N=&P["G)E<75I<F4@
M1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3<&5C.PIR97%U:7)E(%A33&]A
M9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER92!)3SHZ2&%N9&QE.PIR97%U
M:7)E($E/.CI&:6QE.PIR97%U:7)E($-O;7!R97-S.CI:;&EB.PIR97%U:7)E
M($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z.E-(03L*<F5Q=6ER92!0
M05(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER92!005(Z.D1I<W0["G)E
M<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PIR97%U:7)E(%!!4CHZ1FEL
M=&5R.CI0871C:$-O;G1E;G0["G)E<75I<F4@871T<FEB=71E<SL*979A;"![
M(')E<75I<F4@0W=D('T["F5V86P@>R!R97%U:7)E(%=I;C,R('T["F5V86P@
M>R!R97%U:7)E(%-C86QA<CHZ571I;"!].PIE=F%L('L@<F5Q=6ER92!!<F-H
M:79E.CI5;GII<#HZ0G5R<W0@?3L*979A;"![(')E<75I<F4@5&EE.CI(87-H
M.CI.86UE9$-A<'1U<F4@?3L*979A;"![(')E<75I<F4@4&5R;$E/.R!R97%U
M:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@>R!R97%U:7)E('5T9C@@?3L*
M?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD969I;F5D("1%3E9[4$%27U1%
M35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK*2\I('L*)'!A<E]T96UP
M(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY("1P871H("@**&UA<"`D14Y6
M>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2(%1%35!$25(@5$5-4"!435`@
M*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI('L*;F5X="!U;FQE<W,@9&5F
M:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD("UW("1P871H.PIM>2`D=7-E
M<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P=W5I9"D@/2!G971P=W5I9"@D
M/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E9FEN960H)E=I;C,R.CI,;V=I
M;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN,S(Z.DQO9VEN3F%M93L*?0IE
M;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S97)N86UE(#T@)'!W=6ED.PI]
M"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U5315).04U%?2!\?"`D14Y6>U53
M15)]('Q\("=365-414TG.PI]"FUY("1S=&UP9&ER(#T@(B1P871H)'-Y<WM?
M9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L("1U<V5R;F%M92D["FUK9&ER("1S
M=&UP9&ER+"`P-S4U.PIM>2`D8V%C:&5?9&ER.PII9B`H)$5.5GM005)?0TQ%
M04Y]*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["GT*96QS92!["F]P96X@
M;7D@)&9H+"`B/#IR87<B+"`D<')O9VYA;64@;W(@9&EE('%Q6T-A;B=T(')E
M860@(B1P<F]G;F%M92(Z("0A73L*:68@*"AM>2`D;6%G:6-?<&]S(#T@9FEN
M9%]P87)?;6%G:6,H)&9H*2D@/CT@,"D@>PIS965K("1F:"P@)&UA9VEC7W!O
M<PHM("1&24Q%7V]F9G-E=%]S:7IE"BT@;&5N9W1H*")<,$-!0TA%(BDL(#`[
M"FUY("1B=68["G)E860@)&9H+"`D8G5F+"!L96YG=&@H(EPP0T%#2$4B*3L*
M:68@*"1B=68@97$@(EPP0T%#2$4B*2!["G-E96L@)&9H+"`D;6%G:6-?<&]S
M"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG=&@H(EPP0T%#2$4B*0HM("1C
M86-H95]N86UE7W-I>F4L(#`["G)E860@)&9H+"`D8G5F+"`D8V%C:&5?;F%M
M95]S:7IE.PHD8G5F(#U^(',O7#`O+V<["B1C86-H95]D:7(@/2`B8V%C:&4M
M)&)U9B(["GT*?0IC;&]S92`D9F@["GT*:68@*"$D8V%C:&5?9&ER*2!["B1C
M86-H95]D:7(@/2`B=&5M<"TD)"(["B1%3E9[4$%27T-,14%.?2`](#$["GT*
M)'-T;7!D:7(@+CT@(B1S>7-[7V1E;&EM?21C86-H95]D:7(B.PIM:V1I<B`D
M<W1M<&1I<BP@,#<U-3L*)$5.5GM005)?5$5-4'T@/2`D<W1M<&1I<CL*;&%S
M=#L*?0HD<&%R7W1E;7`@/2`D,2!I9B`D14Y6>U!!4E]414U0?2!A;F0@)$5.
M5GM005)?5$5-4'T@/7X@+R@N*RDO.PI]"G-U8B!?<V%V95]A<R!["FUY("@D
M;F%M92P@)&-O;G1E;G1S+"`D;6]D92D@/2!`7SL*;7D@)&9U;&QN86UE(#T@
M(B1P87)?=&5M<"\D;F%M92(["G5N;&5S<R`H+64@)&9U;&QN86UE*2!["FUY
M("1T96UP;F%M92`]("(D9G5L;&YA;64N)"0B.PIM>2`D9F@["BAO<&5N("1F
M:"P@)SXZ<F%W)RP@)'1E;7!N86UE*0HF)B`H<')I;G0@)&9H("1C;VYT96YT
M<RD*)B8@*&-L;W-E("1F:"D*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@(B1T
M96UP;F%M92(Z("0A73L*8VAM;V0@)&UO9&4L("1T96UP;F%M92!I9B!D969I
M;F5D("1M;V1E.PIR96YA;64H)'1E;7!N86UE+"`D9G5L;&YA;64I(&]R('5N
M;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D9G5L;&YA;64["GT*<W5B(%]S
M971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D14Y6>U!!4E]04D]'3D%-17T@
M86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O*"XK*2\I('L*)'!R;V=N86UE
M(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*:68@*"1%3E9[4$%27U1%35!]
M(&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[4$%27U1%35!]*2`^/2`P*2![
M"B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA;64L(')I;F1E>"@D<')O9VYA
M;64L("1S>7-[7V1E;&EM?2D@*R`Q*3L*?0II9B`H(21%3E9[4$%27U!23T=.
M04U%?2!O<B!I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@/CT@,"D@
M>PII9B`H;W!E;B!M>2`D9F@L("<\)RP@)'!R;V=N86UE*2!["G)E='5R;B!I
M9B`M<R`D9F@["GT*:68@*"US("(D<')O9VYA;64D<WES>U]E>&5](BD@>PHD
M<')O9VYA;64@+CT@)'-Y<WM?97AE?3L*<F5T=7)N.PI]"GT*9F]R96%C:"!M
M>2`D9&ER("AS<&QI="`O7%$D<WES>W!A=&A?<V5P?5Q%+RP@)$5.5GM0051(
M?2D@>PIN97AT(&EF(&5X:7-T<R`D14Y6>U!!4E]414U0?2!A;F0@)&1I<B!E
M<2`D14Y6>U!!4E]414U0?3L*)&1I<B`]?B!S+UQ1)'-Y<WM?9&5L:6U]7$4D
M+R\["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64D
M<WES>U]E>&5](BDL(&QA<W0I"FEF("US("(D9&ER)'-Y<WM?9&5L:6U])'!R
M;V=N86UE)'-Y<WM?97AE?2(["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D
M96QI;7TD<')O9VYA;64B*2P@;&%S="D*:68@+7,@(B1D:7(D<WES>U]D96QI
M;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?<')O9VYA;64@>PHD,"`]("1P
M<F]G;F%M92!\?#T@)$5.5GM005)?4%)/1TY!345].PII9B`H:6YD97@H)'!R
M;V=N86UE+"`D<WES>U]D96QI;7TI(#P@,"D@>PHD<')O9VYA;64@/2`B+B1S
M>7-[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@)'!W9"`]("AD969I;F5D("9#
M=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I"CH@*"AD969I;F5D("97:6XS
M,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H*2`Z(&!P=V1@*3L*8VAO;7`H
M)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<+EPN/UQ1)'-Y<WM?9&5L:6U]
M7$4I+R1P=V0D<WES>U]D96QI;7TO.PHD14Y6>U!!4E]04D]'3D%-17T@/2`D
M<')O9VYA;64["GT*<W5B(%]P87)?:6YI=%]E;G8@>PII9B`H("1%3E9[4$%2
M7TE.251)04Q)6D5$?2LK(#T](#$@*2!["G)E='5R;CL*?2!E;'-E('L*)$5.
M5GM005)?24Y)5$E!3$E:141](#T@,CL*?0IF;W(@*'%W*"!34$%73D5$(%1%
M35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/1TY!344@*2`I('L*9&5L971E("1%
M3E9[(E!!4E\D7R)].PI]"F9O<B`H<7<O(%1-4$1)4B!414U0($-,14%.($1%
M0E5'("\I('L*)$5.5GLB4$%27R1?(GT@/2`D14Y6>R)005)?1TQ/0D%,7R1?
M(GT@:68@97AI<W1S("1%3E9[(E!!4E]'3$]"04Q?)%\B?3L*?0II9B`H)$5.
M5GM005)?5$5-4'TI('L*9&5L971E("1%3E9[4$%27T-,14%.?3L*?0I]"G-U
M8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*:68@*"1L;V=F:"D@>PIP<FEN
M="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP<FEN="`B0%]<;B(["GT*?0IS
M=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I9SL*<'5S:"!`24Y#+"!G<F5P
M(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF:6=[)%]]+"!Q=R@*87)C:&QI
M8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP('-I=&5L:6)E>'`*=F5N9&]R
M87)C:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP86-K86=E(&UA:6X["G)E<75I
M<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2.CIF:6YD7W!A<CL*4$%2+3YI
M;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q6W!A<BYP;#H@0V%N)W0@;W!E;B!P
M97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@<W5C:"!F:6QE(&]R(&1I<F5C
M=&]R>5QN70IU;FQE<W,@+64@)'!R;V=N86UE.PID;R`D<')O9VYA;64["D-/
M4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+7T58251?7"@H7&0K*5PI+RD[
M"F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/4B`]("1`(&EF("1`.PI]"D-/
M4D4Z.F5X:70H)#$I(&EF("@D.CI?7T524D]2(#U^+UY?5$M?15A)5%]<*"A<
M9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@:68@)#HZ7U]%4E)/4CL*,3L*7U]%
M3D1?7PH`<&5R;'AS:2YC`````$-!0TA%```]````0`````L```!`````$```
M`$`````F````0````#4```!``````P```$`````&````0````!0```!`````
M#@```$`````-````0````"4```!`````$@```$`````H````0`````(```!`
M````&````$`````)````0`````H```!```````````````"!L@&!L+"L!P``
M``"$`@&!L`^Q``````"%L@&!L+"N?P````#<B/]_L*H)@*B+_W\!````K([_
M?["K'H`HD/]_P/__?UB1_W\!````E)/_?["H`X`XE/]_`0```/"?_W^L__]_
M7*#_?[#__W\8HO]_L+"H@#2B_W\!````8*/_?["O/(!(JO]_L*L"@!2M_W\!
M````````````````````````````````````````````````````````````
M```T&@``P!D```$````X`P```0```$,#```!````30,```$```!A`P``#P``
M`.D$```,````#!,```T```#P2@``&0```+"-`0`;````!````!H```"TC0$`
M'`````0```#U_O]OM`$```4```#X"0``!@```$@#```*````#`4```L````0
M````%0`````````#````T(X!``(```#8`0``%````!$````7````-!$``!$`
M``!<$```$@```-@````3````"````!@`````````^___;P$```C^__]OW`\`
M`/___V\#````\/__;P0/``#Z__]O#@``````````````````````````````
M``````````````````````````"XC0$````````````8$P``&!,``!@3```8
M$P``&!,``!@3```8$P``&!,``!@3```8$P``&!,``!@3```8$P``&!,``!@3
M```8$P``&!,``!@3```8$P``&!,``!@3```8$P``&!,``!@3```8$P``&!,`
M`!@3```8$P``&!,``!@3```8$P``&!,``!@3```8$P``&!,``!@3```8$P``
M&!,``!@3```8$P``&!,``!@3```8$P``&!,``!@3```8$P``&!,``!@3```8
M$P``&!,``!@3```8$P``&!,``!@3```8$P``&!,``!@3```8$P``&!,`````
M````````````````````````````````````````````````````````````
M``````#P%0````````20`0`*4$%2+G!M"@````#43@``Z$L```````"H3```
MK$P``,A.``#P2P``T$X```````"T3@``O$X```````#<3@``1T-#.B`H1TY5
M*2`Q,BXQ+C``1T-#.B`H8W)O<W-T;V]L+4Y'(#$N,C4N,"XR.%]B.#AD,S,X
M*2`Q,BXQ+C``030```!A96%B:0`!*@````4W+4$`!@H'00@!"0(*`PP!$@03
M`10!%0$7`Q@!&0$:`AP!(@$`+G-H<W1R=&%B`"YI;G1E<G``+FYO=&4N9VYU
M+F)U:6QD+6ED`"YN;W1E+D%"22UT86<`+F=N=2YH87-H`"YD>6YS>6T`+F1Y
M;G-T<@`N9VYU+G9E<G-I;VX`+F=N=2YV97)S:6]N7W(`+G)E;"YD>6X`+G)E
M;"YP;'0`+FEN:70`+G1E>'0`+F9I;FD`+G)O9&%T80`N05)-+F5X=&%B`"Y!
M4DTN97AI9'@`+F5H7V9R86UE`"YI;FET7V%R<F%Y`"YF:6YI7V%R<F%Y`"YD
M>6YA;6EC`"YG;W0`+F1A=&$`+F)S<P`N8V]M;65N=``N05)-+F%T=')I8G5T
M97,```````````````````````````````````````````````````````L`
M```!`````@```%0!``!4`0``&0```````````````0`````````3````!P``
M``(```!P`0``<`$``"0```````````````0`````````)@````<````"````
ME`$``)0!```@```````````````$`````````#0```#V__]O`@```+0!``"T
M`0``E`$```4`````````!`````0````^````"P````(```!(`P``2`,``+`&
M```&`````P````0````0````1@````,````"````^`D``/@)```,!0``````
M```````!`````````$X```#___]O`@````0/```$#P``U@````4`````````
M`@````(```!;````_O__;P(```#<#P``W`\``(`````&`````P````0`````
M````:@````D````"````7!```%P0``#8````!0`````````$````"````',`
M```)````0@```#01```T$0``V`$```4````6````!`````@```!\`````0``
M``8````,$P``#!,```P```````````````0`````````=P````$````&````
M&!,``!@3``#8`@`````````````$````!````((````!````!@```/`5``#P
M%0```#4`````````````"`````````"(`````0````8```#P2@``\$H```@`
M``````````````0`````````C@````$````"````^$H``/A*``#X00``````
M```````$`````````)8````!`````@```/",``#PC```)```````````````
M!`````````"A`````0``<((````4C0``%(T``'`````-``````````0`````
M````K`````$````"````A(T``(2-```$```````````````$`````````+8`
M```.`````P```+"-`0"PC0``!```````````````!`````0```#"````#P``
M``,```"TC0$`M(T```0```````````````0````$````S@````8````#````
MN(T!`+B-```8`0``!@`````````$````"````-<````!`````P```-".`0#0
MC@``,`$`````````````!`````0```#<`````0````,`````D`$``)```$@`
M``````````````0`````````X@````@````#````2)`!`$B0```4````````
M```````$`````````.<````!````,`````````!(D```/P``````````````
M`0````$```#P`````P``<```````````AY```#4```````````````$`````
K`````0````,``````````````+R0`````0`````````````!````````````
