#include "../pfmlib_amd64_priv.h"
#include "../pfmlib_priv.h"
/*
 * Copyright (C) 2024 Advanced Micro Devices, Inc. All rights reserved.
 *               Contributed by Swarup Sahoo
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * This file is part of libpfm, a performance monitoring support library for
 * applications on Linux.
 *
 * PMU: amd64_fam1ah_zen5 (AMD64 Fam1Ah Zen5)
 */

static const amd64_umask_t amd64_fam1ah_zen5_retired_x87_fp_ops[]={
  { .uname  = "ADD_SUB_OPS",
    .udesc  = "Number of add/subtract ops",
    .ucode  = 0x01,
  },
  { .uname  = "MUL_OPS",
    .udesc  = "Number of multiply ops",
    .ucode  = 0x2,
  },
  { .uname  = "DIV_SQRT_OPS",
    .udesc  = "Number of divide and square root ops",
    .ucode  = 0x4,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_retired_sse_avx_flops[]={
  { .uname  = "ADD_SUB_FLOPS",
    .udesc  = "Addition/subtraction FLOPS",
    .ucode  = 0x1,
    .grpid  = 0,
  },
  { .uname  = "MULT_FLOPS",
    .udesc  = "Multiplication FLOPS",
    .ucode  = 0x2,
    .grpid  = 0,
  },
  { .uname  = "DIV_FLOPS",
    .udesc  = "Division/Square-root FLOPS",
    .ucode  = 0x4,
    .grpid  = 0,
  },
  { .uname  = "MAC_FLOPS",
    .udesc  = "Multiply-Accumulate flops. Each MAC operation is counted as 2 FLOPS. This event does not include bfloat MAC operations",
    .ucode  = 0x8,
    .grpid  = 0,
  },
  { .uname  = "ALL_TYPE",
    .udesc  = "All types",
    .ucode  = 0x0,
    .uflags = AMD64_FL_DFL | AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "FLOP_TYPE_BFLOAT16",
    .udesc  = "B Float 16",
    .ucode  = 0x20,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "FLOP_TYPE_SCALAR_SINGLE",
    .udesc  = "Scalar single",
    .ucode  = 0x40,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "FLOP_TYPE_PACKED_SINGLE",
    .udesc  = "Packed single",
    .ucode  = 0x60,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "FLOP_TYPE_SCALAR_DOUBLE",
    .udesc  = "Scalar double",
    .ucode  = 0x80,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "FLOP_TYPE_PACKED_DOUBLE",
    .udesc  = "Packed double",
    .ucode  = 0xa0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "ANY",
    .udesc  = "Double precision add/subtract flops",
    .ucode  = 0x0f,
    .uflags = AMD64_FL_DFL | AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_retired_serializing_ops[]={
  { .uname  = "X87_CTRL_RET",
    .udesc  = "x87 control word mispredict traps due to mispredictions in RC or PC, or changes in Exception Mask bits",
    .ucode  = 0x1,
  },
  { .uname  = "X87_BOT_RET",
    .udesc  = "x87 bottom-executing ops retired",
    .ucode  = 0x2,
  },
  { .uname  = "SSE_CTRL_RET",
    .udesc  = "SSE/AVX control word mispredict traps",
    .ucode  = 0x4,
  },
  { .uname  = "SSE_BOT_RET",
    .udesc  = "SSE/AVX bottom-executing ops retired",
    .ucode  = 0x8,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_retired_fp_ops_by_width[]={
  { .uname  = "X87_UOPS_RETIRED",
    .udesc  = "X87 uops retired",
    .ucode  = 0x1,
  },
  { .uname  = "MMX_UOPS_RETIRED",
    .udesc  = "MMX uops retired",
    .ucode  = 0x2,
  },
  { .uname  = "SCALAR_UOPS_RETIRED",
    .udesc  = "Scalar uops retired",
    .ucode  = 0x4,
  },
  { .uname  = "PACK128_UOPS_RETIRED",
    .udesc  = "Packed 128-bit uops retired",
    .ucode  = 0x8,
  },
  { .uname  = "PACK256_UOPS_RETIRED",
    .udesc  = "Packed 256-bit uops retired",
    .ucode  = 0x10,
  },
  { .uname  = "PACK512_UOPS_RETIRED",
    .udesc  = "Packed 512-bit uops retired",
    .ucode  = 0x20,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_retired_fp_ops_by_type[]={
  { .uname  = "SCALAR_NONE",
    .udesc  = "Do not count scalar ops",
    .ucode  = 0x0,
    .uflags = AMD64_FL_NCOMBO | AMD64_FL_DFL,
    .grpid  = 0,
  },
  { .uname  = "SCALAR_ADD",
    .udesc  = "Number of scalar ADD ops retired",
    .ucode  = 0x01,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "SCALAR_SUB",
    .udesc  = "Number of scalar SUB ops retired",
    .ucode  = 0x02,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "SCALAR_MUL",
    .udesc  = "Number of scalar MUL ops retired",
    .ucode  = 0x03,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "SCALAR_MAC",
    .udesc  = "Number of scalar MAC ops retired",
    .ucode  = 0x04,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "SCALAR_DIV",
    .udesc  = "Number of scalar DIV ops retired",
    .ucode  = 0x05,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "SCALAR_SQRT",
    .udesc  = "Number of scalar SQRT ops retired",
    .ucode  = 0x06,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "SCALAR_CMP",
    .udesc  = "Number of scalar CMP ops retired",
    .ucode  = 0x07,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "SCALAR_CVT",
    .udesc  = "Number of scalar CVT ops retired",
    .ucode  = 0x08,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "SCALAR_BLEND",
    .udesc  = "Number of scalar BLEND ops retired",
    .ucode  = 0x09,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "SCALAR_MOVE",
    .udesc  = "Number of scalar MOV ops retired",
    .ucode  = 0x0a,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "SCALAR_SHUFFLE",
    .udesc  = "Number of scalar SHUFFLE ops retired",
    .ucode  = 0x0b,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "SCALAR_BFLOAT",
    .udesc  = "Number of scalar BFloat ops retired",
    .ucode  = 0x0c,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "SCALAR_LOGICAL",
    .udesc  = "Number of scalar LOGICAL ops retired",
    .ucode  = 0x0d,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "SCALAR_OTHER",
    .udesc  = "Number of other scalar ops retired",
    .ucode  = 0x0e,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "SCALAR_ALL",
    .udesc  = "Number of any scalar ops retired",
    .ucode  = 0x0f,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "VECTOR_NONE",
    .udesc  = "Do not count vector ops",
    .ucode  = 0x0,
    .uflags = AMD64_FL_NCOMBO | AMD64_FL_DFL,
    .grpid  = 1,
  },
  { .uname  = "VECTOR_ADD",
    .udesc  = "Number of vector ADD ops retired",
    .ucode  = 0x10,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "VECTOR_SUB",
    .udesc  = "Number of vector SUB ops retired",
    .ucode  = 0x20,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "VECTOR_MUL",
    .udesc  = "Number of vector MUL ops retired",
    .ucode  = 0x30,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "VECTOR_MAC",
    .udesc  = "Number of vector MAC ops retired",
    .ucode  = 0x40,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "VECTOR_DIV",
    .udesc  = "Number of DIV ops retired",
    .ucode  = 0x50,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "VECTOR_SQRT",
    .udesc  = "Number of vector SQRT ops retired",
    .ucode  = 0x60,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "VECTOR_CMP",
    .udesc  = "Number of vector CMP ops retired",
    .ucode  = 0x70,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "VECTOR_CVT",
    .udesc  = "Number of vector CVT ops retired",
    .ucode  = 0x80,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "VECTOR_BLEND",
    .udesc  = "Number of vector BLEND ops retired",
    .ucode  = 0x90,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "VECTOR_MOVE",
    .udesc  = "Number of vector MOV ops retired",
    .ucode  = 0xa0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "VECTOR_SHUFFLE",
    .udesc  = "Number of vector SHUFFLE ops retired",
    .ucode  = 0xb0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "VECTOR_BFLOAT",
    .udesc  = "Number of vector BFloat ops retired",
    .ucode  = 0xc0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "VECTOR_LOGICAL",
    .udesc  = "Number of vector LOGICAL ops retired",
    .ucode  = 0xd0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "VECTOR_OTHER",
    .udesc  = "Number of other vector ops retired",
    .ucode  = 0xe0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "VECTOR_ALL",
    .udesc  = "Number of scalar ADD and vector ALL ops retired",
    .ucode  = 0xf0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_retired_int_ops[]={
  { .uname  = "MMX_NONE",
    .udesc  = "Do count MMX ops",
    .ucode  = 0x0,
    .uflags = AMD64_FL_NCOMBO | AMD64_FL_DFL,
    .grpid  = 0,
  },
  { .uname  = "MMX_ADD",
    .udesc  = "Number of MMX ADD ops retired",
    .ucode  = 0x1,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "MMX_SUB",
    .udesc  = "Number of MMX SUB ops retired",
    .ucode  = 0x2,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "MMX_MUL",
    .udesc  = "Number of MMX MUL ops retired",
    .ucode  = 0x3,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "MMX_MAC",
    .udesc  = "Number of MMX MAC ops retired",
    .ucode  = 0x4,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "MMX_AES",
    .udesc  = "Number of MMX AES ops retired",
    .ucode  = 0x5,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "MMX_SHA",
    .udesc  = "Number of MMX SHA ops retired",
    .ucode  = 0x6,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "MMX_CMP",
    .udesc  = "Number of MMX CMP ops retired",
    .ucode  = 0x7,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "MMX_CVT",
    .udesc  = "Number of MMX CVT or PACK ops retired",
    .ucode  = 0x8,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "MMX_SHIFT",
    .udesc  = "Number of MMX SHIFT ops retired",
    .ucode  = 0x9,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "MMX_MOV",
    .udesc  = "Number of MMX MOV ops retired",
    .ucode  = 0xa,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "MMX_SHUFFLE",
    .udesc  = "Number of MMX SHUFFLE ops retired",
    .ucode  = 0xb,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "MMX_VNNI",
    .udesc  = "Number of MMX VNNI ops retired",
    .ucode  = 0xc,
    .uflags = AMD64_FL_NCOMBO,
  },
  { .uname  = "MMX_LOGICAL",
    .udesc  = "Number of MMX LOGICAL ops retired",
    .ucode  = 0xd,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "MMX_OTHER",
    .udesc  = "Number of other MMX ops retired",
    .ucode  = 0xe,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "MMX_ALL",
    .udesc  = "Any MMX ops retired",
    .ucode  = 0xf,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "SSE_AVX_NONE",
    .udesc  = "Do not count SSE/AVX ops",
    .ucode  = 0x0,
    .uflags = AMD64_FL_NCOMBO | AMD64_FL_DFL,
    .grpid  = 1,
  },
  { .uname  = "SSE_AVX_ADD",
    .udesc  = "Number of SSE/AVX ADD ops retired",
    .ucode  = 0x10,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "SSE_AVX_SUB",
    .udesc  = "Number of SSE/AVX SUB ops retired",
    .ucode  = 0x20,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "SSE_AVX_MUL",
    .udesc  = "Number of SSE/AVX MUL ops retired",
    .ucode  = 0x30,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "SSE_AVX_MAC",
    .udesc  = "Number of SSE/AVX MAC ops retired",
    .ucode  = 0x40,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "SSE_AVX_AES",
    .udesc  = "Number of SSE/AVX AES ops retired",
    .ucode  = 0x50,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "SSE_AVX_SHA",
    .udesc  = "Number of SSE/AVX SHA ops retired",
    .ucode  = 0x60,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "SSE_AVX_CMP",
    .udesc  = "Number of SSE/AVX CMP ops retired",
    .ucode  = 0x70,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "SSE_AVX_CVT",
    .udesc  = "Number of SSE/AVX CVT or PACK ops retired",
    .ucode  = 0x80,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "SSE_AVX_SHIFT",
    .udesc  = "Number of SSE/AVX SHIFT ops retired",
    .ucode  = 0x90,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "SSE_AVX_MOV",
    .udesc  = "Number of SSE/AVX MOV ops retired",
    .ucode  = 0xa0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "SSE_AVX_SHUFFLE",
    .udesc  = "Number of SSE/AVX SHUFFLE ops retired",
    .ucode  = 0xb0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "SSE_AVX_VNNI",
    .udesc  = "Number of SSE/AVX VNNI ops retired",
    .ucode  = 0xc0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "SSE_AVX_LOGICAL",
    .udesc  = "Number of SSE/AVX LOGICAL ops retired",
    .ucode  = 0xd0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "SSE_AVX_OTHER",
    .udesc  = "Number of other SSE/AVX ops retired",
    .ucode  = 0xe0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "SSE_AVX_ALL",
    .udesc  = "Any SSE/AVX ops retired",
    .ucode  = 0xf0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_packed_fp_ops_retired[]={
  { .uname  = "FP128_NONE",
    .udesc  = "Do not count any 128-bit ops",
    .ucode  = 0x0,
    .uflags = AMD64_FL_NCOMBO | AMD64_FL_DFL,
    .grpid  = 0,
  },
  { .uname  = "FP128_ADD",
    .udesc  = "Number of floating-point 128-bit ADD ops retired",
    .ucode  = 0x01,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "FP128_SUB",
    .udesc  = "Number of floating-point 128-bit SUB ops retired",
    .ucode  = 0x02,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "FP128_MUL",
    .udesc  = "Number of floating-point 128-bit MUL ops retired",
    .ucode  = 0x03,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "FP128_MAC",
    .udesc  = "Number of floating-point 128-bit MAC ops retired",
    .ucode  = 0x04,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "FP128_DIV",
    .udesc  = "Number of floating-point 128-bit DIV ops retired",
    .ucode  = 0x05,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "FP128_SQRT",
    .udesc  = "Number of floating-point 128-bit SQRT ops retired",
    .ucode  = 0x06,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "FP128_CMP",
    .udesc  = "Number of floating-point 128-bit CMP ops retired",
    .ucode  = 0x07,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "FP128_CVT",
    .udesc  = "Number of floating-point 128-bit CVT ops retired",
    .ucode  = 0x08,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "FP128_BLEND",
    .udesc  = "Number of floating-point 128-bit 256-bit BLEND ops retired",
    .ucode  = 0x09,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "FP128_MOV",
    .udesc  = "Number of floating-point 128-bit MOV ops retired",
    .ucode  = 0x0a,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "FP128_SHUFFLE",
    .udesc  = "Number of floating-point 128-bit SHUFFLE ops retired",
    .ucode  = 0x0b,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "FP128_BFLOAT",
    .udesc  = "Number of floating-point 128-bit BFloat ops retired",
    .ucode  = 0x0c,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "FP128_LOGICAL",
    .udesc  = "Number of floating-point 128-bit LOGICAL ops retired",
    .ucode  = 0x0d,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "FP128_OTHER",
    .udesc  = "Number of other floating-point 128-bit ops retired",
    .ucode  = 0x0e,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "FP128_ALL",
    .udesc  = "Number of any  floating-point 128-bit ops retired",
    .ucode  = 0x0f,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "FP256_NONE",
    .udesc  = "Do not count any 256-bit ops",
    .ucode  = 0x0,
    .uflags = AMD64_FL_NCOMBO | AMD64_FL_DFL,
    .grpid  = 1,
  },
  { .uname  = "FP256_ADD",
    .udesc  = "Number of floating-point 256-bit ADD ops retired",
    .ucode  = 0x10,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "FP256_SUB",
    .udesc  = "Number of floating-point 256-bit SUB ops retired",
    .ucode  = 0x20,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "FP256_MUL",
    .udesc  = "Number of floating-point 256-bit MUL ops retired",
    .ucode  = 0x30,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "FP256_MAC",
    .udesc  = "Number of floating-point 256-bit MAC ops retired",
    .ucode  = 0x40,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "FP256_DIV",
    .udesc  = "Number of floating-point 256-bit DIV ops retired",
    .ucode  = 0x50,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "FP256_SQRT",
    .udesc  = "Number of floating-point 256-bit SQRT ops retired",
    .ucode  = 0x60,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "FP256_CMP",
    .udesc  = "Number of floating-point 256-bit CMP ops retired",
    .ucode  = 0x70,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "FP256_CVT",
    .udesc  = "Number of floating-point 256-bit CVT ops retired",
    .ucode  = 0x80,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "FP256_BLEND",
    .udesc  = "Number of floating-point 256-bit BLEND ops retired",
    .ucode  = 0x90,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "FP256_MOV",
    .udesc  = "Number of floating-point 256-bit MOV ops retired",
    .ucode  = 0xa0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "FP256_SHUFFLE",
    .udesc  = "Number of floating-point 256-bit SHUFFLE ops retired",
    .ucode  = 0xb0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "FP256_BFLOAT",
    .udesc  = "Number of floating-point 256-bit BFloat ops retired",
    .ucode  = 0xc0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "FP256_LOGICAL",
    .udesc  = "Number of floating-point 256-bit LOGICAL ops retired",
    .ucode  = 0xd0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "FP256_OTHER",
    .udesc  = "Number of other floating-point 256-bit ops retired",
    .ucode  = 0xe0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "FP256_ALL",
    .udesc  = "Any floating-point 256-bit ops retired",
    .ucode  = 0xf0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_packed_int_ops_retired[]={
  { .uname  = "INT128_NONE",
    .udesc  = "Do not count 128-bit ops",
    .ucode  = 0x0,
    .uflags = AMD64_FL_NCOMBO | AMD64_FL_DFL,
    .grpid  = 0,
  },
  { .uname  = "INT128_ADD",
    .udesc  = "Number of integer 128-bit ADD ops retired",
    .ucode  = 0x01,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "INT128_SUB",
    .udesc  = "Number of integer 128-bit SUB ops retired",
    .ucode  = 0x02,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "INT128_MUL",
    .udesc  = "Number of integer 128-bit MUL ops retired",
    .ucode  = 0x03,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "INT128_MAC",
    .udesc  = "Number of integer 256-bit MAC ops retired",
    .ucode  = 0x04,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "INT128_AES",
    .udesc  = "Number of integer 128-bit AES ops retired",
    .ucode  = 0x05,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "INT128_SHA",
    .udesc  = "Number of integer 128-bit SHA ops retired",
    .ucode  = 0x06,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "INT128_CMP",
    .udesc  = "Number of integer 128-bit CMP ops retired",
    .ucode  = 0x07,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "INT128_CVT",
    .udesc  = "Number of integer 128-bit CVT or PACK ops retired",
    .ucode  = 0x08,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "INT128_SHIFT",
    .udesc  = "Number of integer 128-bit SHIFT ops retired",
    .ucode  = 0x09,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "INT128_MOV",
    .udesc  = "Number of integer 128-bit MOV ops retired",
    .ucode  = 0x0a,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "INT128_SHUFFLE",
    .udesc  = "Number of integer 128-bit SHUFFLE ops retired",
    .ucode  = 0x0b,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "INT128_VNNI",
    .udesc  = "Number of integer 128-bit VNNI ops retired",
    .ucode  = 0x0c,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "INT128_LOGICAL",
    .udesc  = "Number of integer 128-bit LOGICAL ops retired",
    .ucode  = 0x0d,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "INT128_OTHER",
    .udesc  = "Number of other integer 128-bit ops retired",
    .ucode  = 0x0e,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "INT128_ALL",
    .udesc  = "Any integer 128-bit ops retired",
    .ucode  = 0x0f,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 0,
  },
  { .uname  = "INT256_NONE",
    .udesc  = "Do not count 256-bit ops",
    .ucode  = 0x0,
    .uflags = AMD64_FL_NCOMBO | AMD64_FL_DFL,
    .grpid  = 1,
  },
  { .uname  = "INT256_ADD",
    .udesc  = "Number of integer 256-bit ADD ops retired",
    .ucode  = 0x10,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "INT256_SUB",
    .udesc  = "Number of integer 256-bit SHIFT ops retired",
    .ucode  = 0x20,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "INT256_MUL",
    .udesc  = "Number of integer 256-bit MUL ops retired",
    .ucode  = 0x30,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "INT256_MAC",
    .udesc  = "Number of integer 256-bit MAC ops retired",
    .ucode  = 0x40,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "INT256_AES",
    .udesc  = "Number of integer 256-bit AES ops retired",
    .ucode  = 0x50,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "INT256_SHA",
    .udesc  = "Number of integer 256-bit SHA ops retired",
    .ucode  = 0x60,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "INT256_CMP",
    .udesc  = "Number of integer 256-bit CMP ops retired",
    .ucode  = 0x70,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "INT256_CVT",
    .udesc  = "Number of integer 256-bit CVT or PACK ops retired",
    .ucode  = 0x80,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "INT256_SHIFT",
    .udesc  = "Number of integer 256-bit SHIFT ops retired",
    .ucode  = 0x90,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "INT256_MOV",
    .udesc  = "Number of integer 256-bit MOV ops retired",
    .ucode  = 0xa0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "INT256_SHUFFLE",
    .udesc  = "Number of integer 256-bit SHUFFLE ops retired",
    .ucode  = 0xb0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "INT256_VNNI",
    .udesc  = "Number of integer 256-bit VNNI ops retired",
    .ucode  = 0xc0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "INT256_LOGICAL",
    .udesc  = "Number of integer 256-bit LOGICAL ops retired",
    .ucode  = 0xd0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "INT256_OTHER",
    .udesc  = "Number of other integer 256-bit ops retired",
    .ucode  = 0xe0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
  { .uname  = "INT256_ALL",
    .udesc  = "Any integer 256-bit ops retired",
    .ucode  = 0xf0,
    .uflags = AMD64_FL_NCOMBO,
    .grpid  = 1,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_fp_dispatch_faults[]={
  { .uname  = "X87_FILL_FAULT",
    .udesc  = "x87 fill faults",
    .ucode  = 0x1,
  },
  { .uname  = "XMM_FILL_FAULT",
    .udesc  = "XMM fill faults",
    .ucode  = 0x2,
  },
  { .uname  = "YMM_FILL_FAULT",
    .udesc  = "YMM fill faults",
    .ucode  = 0x4,
  },
  { .uname  = "YMM_SPILL_FAULT",
    .udesc  = "YMM spill faults",
    .ucode  = 0x8,
  },
  { .uname  = "ANY",
    .udesc  = "Any FP dispatch faults",
    .ucode  = 0xf,
    .uflags = AMD64_FL_DFL | AMD64_FL_NCOMBO,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_bad_status_2[]={
  { .uname  = "STLI_OTHER",
    .udesc  = "Store-to-load conflicts. A load was unable to complete due to a non-forwardable conflict with an older store",
    .ucode  = 0x2,
    .uflags = AMD64_FL_DFL,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_retired_lock_instructions[]={
  { .uname  = "BUS_LOCK",
    .udesc  = "Number of non-cacheable or cacheline-misaligned locks",
    .ucode  = 0x1,
  },
  { .uname  = "ANY",
    .udesc  = "Number of all locks",
    .ucode  = 0x1f,
    .uflags = AMD64_FL_NCOMBO | AMD64_FL_DFL,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_ls_dispatch[]={
  { .uname  = "LD_ST_DISPATCH",
    .udesc  = "Dispatched op that performs a load from and store to the same memory address",
    .ucode  = 0x4,
  },
  { .uname  = "STORE_DISPATCH",
    .udesc  = "Store ops dispatched",
    .ucode  = 0x2,
  },
  { .uname  = "LD_DISPATCH",
    .udesc  = "Load ops dispatched",
    .ucode  = 0x1,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_store_commit_cancels_2[]={
  { .uname  = "OLDER_ST_VISIBLE_CANCEL",
    .udesc  = "Older SCB being waited on to become globally visible was unable to become globally visible",
    .ucode  = 0x1,
    .uflags = AMD64_FL_DFL,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_mab_allocation_by_type[]={
  { .uname  = "LS",
    .udesc  = "Load store allocations",
    .ucode  = 0x7,
    .uflags = AMD64_FL_NCOMBO,
  },
  { .uname  = "HW_PF",
    .udesc  = "Hardware prefetcher allocations",
    .ucode  = 0x8,
    .uflags = AMD64_FL_NCOMBO,
  },
  { .uname  = "ALL",
    .udesc  = "All allocations",
    .ucode  = 0xf,
    .uflags = AMD64_FL_NCOMBO | AMD64_FL_DFL,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_demand_data_fills_from_system[]={
  { .uname  = "LCL_L2",
    .udesc  = "Fill from local L2 to the core",
    .ucode  = 0x1,
  },
  { .uname  = "LOCAL_CCX",
    .udesc  = "Fill from L3 or different L2 in same CCX",
    .ucode  = 0x2,
  },
  { .uname  = "NEAR_FAR_CACHE_NEAR",
    .udesc  = "Fill from cache of different CCX in same node",
    .ucode  = 0x4,
  },
  { .uname  = "DRAM_IO_NEAR",
    .udesc  = "Fill from DRAM or IO connected to same node",
    .ucode  = 0x8,
  },
  { .uname  = "NEAR_FAR_CACHE_FAR",
    .udesc  = "Fill from CCX cache in different node",
    .ucode  = 0x10,
  },
  { .uname  = "DRAM_IO_FAR",
    .udesc  = "Fill from DRAM or IO connected from a different node (same socket or remote)",
    .ucode  = 0x40,
  },
  { .uname  = "ALT_MEM_NEAR_FAR",
    .udesc  = "Fill from Extension Memory",
    .ucode  = 0x80,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_l1_dtlb_miss[]={
  { .uname  = "TLB_RELOAD_1G_L2_MISS",
    .udesc  = "Data TLB reload to a 1GB page that missed in the L2 TLB",
    .ucode  = 0x80,
  },
  { .uname  = "TLB_RELOAD_2M_L2_MISS",
    .udesc  = "Data TLB reload to a 2MB page that missed in the L2 TLB",
    .ucode  = 0x40,
  },
  { .uname  = "TLB_RELOAD_COALESCED_PAGE_MISS",
    .udesc  = "Data TLB reload to a coalesced page that also missed in the L2 TLB",
    .ucode  = 0x20,
  },
  { .uname  = "TLB_RELOAD_4K_L2_MISS",
    .udesc  = "Data TLB reload to a 4KB page that missed in the L2 TLB",
    .ucode  = 0x10,
  },
  { .uname  = "TLB_RELOAD_1G_L2_HIT",
    .udesc  = "Data TLB reload to a 1GB page that hit in the L2 TLB",
    .ucode  = 0x8,
  },
  { .uname  = "TLB_RELOAD_2M_L2_HIT",
    .udesc  = "Data TLB reload to a 2MB page that hit in the L2 TLB",
    .ucode  = 0x4,
  },
  { .uname  = "TLB_RELOAD_COALESCED_PAGE_HIT",
    .udesc  = "Data TLB reload to a coalesced page that hit in the L2 TLB",
    .ucode  = 0x2,
  },
  { .uname  = "TLB_RELOAD_4K_L2_HIT",
    .udesc  = "Data TLB reload to a 4KB page that hit in the L2 TLB",
    .ucode  = 0x1,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_misaligned_loads[]={
  { .uname  = "MA4K",
    .udesc  = "The number of 4KB misaligned (page crossing) loads",
    .ucode  = 0x2,
  },
  { .uname  = "MA64",
    .udesc  = "The number of 64B misaligned (cacheline crossing) loads",
    .ucode  = 0x1,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_prefetch_instructions_dispatched[]={
  { .uname  = "PREFETCH_T0_T1_T2",
    .udesc  = "Number of prefetcht0, perfetcht1, prefetcht2 instructions dispatched",
    .ucode  = 0x1,
  },
  { .uname  = "PREFETCHW",
    .udesc  = "Number of prefetchtw instructions dispatched",
    .ucode  = 0x2,
  },
  { .uname  = "PREFETCHNTA",
    .udesc  = "Number of prefetchtnta instructions dispatched",
    .ucode  = 0x4,
  },
  { .uname  = "ANY",
    .udesc  = "Any prefetch",
    .ucode  = 0x7,
    .uflags = AMD64_FL_DFL | AMD64_FL_NCOMBO,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_wcb_close[]={
  { .uname  = "FULL_LINE_64B",
    .udesc  = "All 64 bytes of the WCB entry have been written",
    .ucode  = 0x1,
    .uflags = AMD64_FL_DFL,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_ineffective_software_prefetch[]={
  { .uname  = "MAB_MCH_CNT",
    .udesc  = "Software prefetch instructions saw a match on an already allocated miss request buffer",
    .ucode  = 0x2,
  },
  { .uname  = "DATA_PIPE_SW_PF_DC_HIT",
    .udesc  = "Software Prefetch instruction saw a DC hit",
    .ucode  = 0x1,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_tlb_flushes[]={
  { .uname  = "TLB_FLUSHES",
    .udesc  = "All TLB Flushes",
    .ucode  = 0xff,
    .uflags = AMD64_FL_DFL,
  }
};

static const amd64_umask_t amd64_fam1ah_zen5_p0_freq_cycles_not_in_halt[]={
  { .uname  = "P0_FREQ_CYCLES",
    .udesc  = "Counts at P0 frequency (same as MPERF) when CPU is not in halted state",
    .ucode  = 0x1,
    .uflags = AMD64_FL_DFL,
  }
};

static const amd64_umask_t amd64_fam1ah_zen5_l1_itlb_miss_l2_itlb_miss[]={
  { .uname  = "COALESCED4K",
    .udesc  = "Number of instruction fetches to a >4K coalesced page",
    .ucode  = 0x8,
  },
  { .uname  = "IF1G",
    .udesc  = "Number of instruction fetches to a 1GB page",
    .ucode  = 0x4,
  },
  { .uname  = "IF2M",
    .udesc  = "Number of instruction fetches to a 2MB page",
    .ucode  = 0x2,
  },
  { .uname  = "IF4K",
    .udesc  = "Number of instruction fetches to a 4KB page",
    .ucode  = 0x1,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_itlb_fetch_hit[]={
  { .uname  = "IF1G",
    .udesc  = "L1 instruction fetch TLB hit a 1GB page size",
    .ucode  = 0x4,
  },
  { .uname  = "IF2M",
    .udesc  = "L1 instruction fetch TLB hit a 2MB page size",
    .ucode  = 0x2,
  },
  { .uname  = "IF4K",
    .udesc  = "L1 instruction fetch TLB hit a 4KB or 16KB page size",
    .ucode  = 0x1,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_bp_redirects[]={
  { .uname  = "EX_REDIR",
    .udesc  = "Mispredict redirect from EX (execution-time)",
    .ucode  = 0x2,
  },
  { .uname  = "RESYNC",
    .udesc  = "Resync redirect from RT (retire-time)",
    .ucode  = 0x1,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_fetch_ibs_events[]={
  { .uname  = "SAMPLE_VAL",
    .udesc  = "Counts the number of valid fetch IBS samples that were collected. Each valid sample also created an IBS interrupt",
    .ucode  = 0x10,
  },
  { .uname  = "SAMPLE_FILTERED",
    .udesc  = "Counts the number of fetch IBS tagged fetches that were discarded due to IBS filtering. When a tagged fetch is discarded the fetch IBS facility will automatically tag a new fetch",
    .ucode  = 0x8,
  },
  { .uname  = "SAMPLE_DISCARDED",
    .udesc  = "Counts when the fetch IBS facility discards an IBS tagged fetch for reasons other than IBS filtering. When a tagged fetch is discarded the fetch IBS facility will automatically tag a new fetch",
    .ucode  = 0x4,
  },
  { .uname  = "FETCH_TAGGED",
    .udesc  = "Counts the number of fetches tagged for fetch IBS. Not all tagged fetches create an IBS interrupt and valid fetch sample",
    .ucode  = 0x2,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_ic_tag_hit_miss[]={
  { .uname  = "IC_HIT",
    .udesc  = "Instruction cache hit",
    .ucode  = 0x7,
    .uflags = AMD64_FL_NCOMBO,
  },
  { .uname  = "IC_MISS",
    .udesc  = "Instruction cache miss",
    .ucode  = 0x18,
    .uflags = AMD64_FL_NCOMBO,
  },
  { .uname  = "ALL_IC_ACCESS",
    .udesc  = "All instruction cache accesses",
    .ucode  = 0x1f,
    .uflags = AMD64_FL_NCOMBO | AMD64_FL_DFL,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_op_cache_hit_miss[]={
  { .uname  = "OC_HIT",
    .udesc  = "Op cache hit",
    .ucode  = 0x3,
    .uflags = AMD64_FL_NCOMBO,
  },
  { .uname  = "OC_MISS",
    .udesc  = "Op cache miss",
    .ucode  = 0x4,
    .uflags = AMD64_FL_NCOMBO,
  },
  { .uname  = "ALL_OC_ACCESS",
    .udesc  = "All op cache accesses",
    .ucode  = 0x7,
    .uflags = AMD64_FL_NCOMBO | AMD64_FL_DFL,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_ops_source_dispatched_from_decoder[]={
  { .uname  = "DECODER",
    .udesc  = "Number of ops dispatched from x86 decoder",
    .ucode  = 0x1,
  },
  { .uname  = "OPCACHE",
    .udesc  = "Number of ops dispatched from Op Cache",
    .ucode  = 0x2,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_ops_type_dispatched_from_decoder[]={
  { .uname  = "FP_DISPATCH",
    .udesc  = "Any FP dispatch",
    .ucode  = 0x04,
    .uflags = AMD64_FL_NCOMBO,
  },
  { .uname  = "INTEGER_DISPATCH",
    .udesc  = "Any Integer dispatch",
    .ucode  = 0x08,
    .uflags = AMD64_FL_NCOMBO,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_dispatch_resource_stall_cycles_1[]={
  { .uname  = "INT_PHY_REG_FILE_RSRC_STALL",
    .udesc  = "Number of cycles stalled due to integer physical register file resource stalls",
    .ucode  = 0x1,
  },
  { .uname  = "LOAD_QUEUE_RSRC_STALL",
    .udesc  = "Number of cycles stalled due to load queue resource stalls",
    .ucode  = 0x2,
  },
  { .uname  = "STORE_QUEUE_RSRC_STALL",
    .udesc  = "Number of cycles stalled due to store queue resource stalls",
    .ucode  = 0x4,
  },
  { .uname  = "TAKEN_BRANCH_BUFFER_RSRC_STALL",
    .udesc  = "Number of cycles stalled due to taken branch buffer resource stalls",
    .ucode  = 0x10,
  },
  { .uname  = "FP_SCHEDULER_RSRC_STALL",
    .udesc  = "Number of cycles stalled due to floating-point scheduler resource stalls",
    .ucode  = 0x40,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_dispatch_resource_stall_cycles_2[]={
  { .uname  = "AL_TOKEN_STALL",
    .udesc  = "Number of cycles stalled due to insufficient tokens available for ALU",
    .ucode  = 0x1,
  },
  { .uname  = "AG_TOKEN_STALL",
    .udesc  = "Number of cycles stalled due to insufficient tokens available for Agen",
    .ucode  = 0x2,
  },
  { .uname  = "EX_FLUSH_STALL",
    .udesc  = "Number of cycles stalled due to integer execution flush recovery pending",
    .ucode  = 0x4,
  },
  { .uname  = "RETIRE_TOKEN_STALL",
    .udesc  = "Number of cycles stalled due to insufficient tokens available for Retire Queue",
    .ucode  = 0x20,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_dispatch_stalls_1[]={
  { .uname  = "FE_NO_OPS",
    .udesc  = "Counts dispatch slots left empty because the front-end did not supply ops",
    .ucode  = 0x01,
    .uflags = AMD64_FL_NCOMBO,
  },
  { .uname  = "BE_STALLS",
    .udesc  = "Counts ops unable to dispatch due to back-end stalls",
    .ucode  = 0x1e,
    .uflags = AMD64_FL_NCOMBO,
  },
  { .uname  = "SMT_CONTENTION",
    .udesc  = "Counts ops unable to dispatch because the dispatch cycle was granted to the other SMT thread",
    .ucode  = 0x60,
    .uflags = AMD64_FL_NCOMBO,
  }
};

static const amd64_umask_t amd64_fam1ah_zen5_dispatch_stalls_2[]={
  { .uname  = "FE_NO_OPS",
    .udesc  = "Counts cycles dispatch is stalled due to the lack of dispatch resources",
    .ucode  = 0x30,
    .uflags = AMD64_FL_DFL,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_retired_mmx_fp_instructions[]={
  { .uname  = "SSE_INSTR",
    .udesc  = "Number of SSE instructions (SSE, SSE2, SSE3, SSSE3, SSE4A, SSE41, SSE42, AVX)",
    .ucode  = 0x4,
  },
  { .uname  = "MMX_INSTR",
    .udesc  = "Number of MMX instructions",
    .ucode  = 0x2,
  },
  { .uname  = "X87_INSTR",
    .udesc  = "Number of x87 instructions",
    .ucode  = 0x1,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_cycles_no_retire[]={
  { .uname  = "EMPTY",
    .udesc  = "Number of cycles when there were no valid ops in the retire queue. This may be caused by front-end bottlenecks or pipeline redirects",
    .ucode  = 0x1,
    .uflags = AMD64_FL_NCOMBO,
  },
  { .uname  = "NOT_COMPLETE_LOAD_AND_ALU",
    .udesc  = "Number of cycles when the oldest retire slot did not have its completion bits set",
    .ucode  = 0x2,
    .uflags = AMD64_FL_NCOMBO,
  },
  { .uname  = "OTHER",
    .udesc  = "Number of cycles where ops could have retired but were stopped from retirement for other reasons: retire breaks, traps, faults, etc",
    .ucode  = 0x8,
    .uflags = AMD64_FL_NCOMBO,
  },
  { .uname  = "THREAD_NOT_SELECTED",
    .udesc  = "Number of cycles where ops could have retired but did not because thread arbitration did not select the thread for retire",
    .ucode  = 0x10,
    .uflags = AMD64_FL_NCOMBO,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_tagged_ibs_ops[]={
  { .uname  = "IBS_COUNT_ROLLOVER",
    .udesc  = "Number of times a op could not be tagged by IBS because of a previous tagged op that has not yet signaled interrupt",
    .ucode  = 0x4,
  },
  { .uname  = "IBS_TAGGED_OPS_RET",
    .udesc  = "Number of ops tagged by IBS that retired",
    .ucode  = 0x2,
  },
  { .uname  = "IBS_TAGGED_OPS",
    .udesc  = "Number of ops tagged by IBS",
    .ucode  = 0x1,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_requests_to_l2_group1[]={
  { .uname  = "RD_BLK_L",
    .udesc  = "Number of data cache reads (including software and hardware prefetches)",
    .ucode  = 0x80,
  },
  { .uname  = "RD_BLK_X",
    .udesc  = "Number of data cache stores",
    .ucode  = 0x40,
  },
  { .uname  = "LS_RD_BLK_C_S",
    .udesc  = "Number of data cache shared reads",
    .ucode  = 0x20,
  },
  { .uname  = "CACHEABLE_IC_READ",
    .udesc  = "Number of instruction cache reads",
    .ucode  = 0x10,
  },
  { .uname  = "PREFETCH_L2_CMD",
    .udesc  = "TBD",
    .ucode  = 0x4,
  },
  { .uname  = "L2_HW_PF",
    .udesc  = "Number of prefetches accepted by L2 pipeline, hit or miss",
    .ucode  = 0x2,
  },
  { .uname  = "MISC",
    .udesc  = "Count various non-cacheable requests: non-cached data read, non-cached instruction reads, self-modifying code checks",
    .ucode  = 0x1,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_requests_to_l2_group2[]={
  { .uname  = "LS_RD_SIZED",
    .udesc  = "LS sized read, coherent non-cacheable",
    .ucode  = 0x40,
  },
  { .uname  = "LS_RD_SIZED_NC",
    .udesc  = "LS sized read, non-coherent, non-cacheable",
    .ucode  = 0x20,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_ls_to_l2_wbc_requests[]={
  { .uname  = "WCB_CLOSE",
    .udesc  = "Write combining buffer close",
    .ucode  = 0x20,
    .uflags = AMD64_FL_DFL,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_core_to_l2_cacheable_request_access_status[]={
  { .uname  = "LS_RD_BLK_C_S",
    .udesc  = "Number of data cache shared read hitting in the L2",
    .ucode  = 0x80,
  },
  { .uname  = "LS_RD_BLK_L_HIT_X",
    .udesc  = "Number of data cache reads hitting in the L2",
    .ucode  = 0x40,
  },
  { .uname  = "LS_RD_BLK_L_HIT_S",
    .udesc  = "Number of data cache reads hitting a non-modifiable line in the L2",
    .ucode  = 0x20,
  },
  { .uname  = "LS_RD_BLK_X",
    .udesc  = "Number of data cache store requests hitting in the L2",
    .ucode  = 0x10,
  },
  { .uname  = "LS_RD_BLK_C",
    .udesc  = "Number of data cache requests missing in the L2",
    .ucode  = 0x8,
  },
  { .uname  = "IC_FILL_HIT_X",
    .udesc  = "Number of instruction cache fill requests hitting a modifiable line in the L2",
    .ucode  = 0x4,
  },
  { .uname  = "IC_FILL_HIT_S",
    .udesc  = "Number of instruction cache fill requests hitting a non-modifiable line in the L2",
    .ucode  = 0x2,
  },
  { .uname  = "IC_FILL_MISS",
    .udesc  = "Number of instruction cache fill requests missing in the L2",
    .ucode  = 0x1,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_l2_prefetch_hit_l2[]={
  { .uname  = "L2_HW_PF",
    .udesc  = "Counts requests generated from L2 hardware prefetchers",
    .ucode  = 0x1f,
    .uflags = AMD64_FL_NCOMBO,
  },
  { .uname  = "L1_DC_HW_PF",
    .udesc  = "Counts requests generated from L1 DC hardware prefetchers",
    .ucode  = 0xe0,
    .uflags = AMD64_FL_NCOMBO,
  },
  { .uname  = "L1_DC_L2_HW_PF",
    .udesc  = "Counts requests generated from L1 DC and L2 hardware prefetchers",
    .ucode  = 0xff,
    .uflags = AMD64_FL_NCOMBO,
  },
};

static const amd64_umask_t amd64_fam1ah_zen5_l2_fill_resp_src[]={
  { .uname  = "LOCAL_CCX",
    .udesc  = "Fill from L3 or different L2 in same CCX",
    .ucode  = 0x2,
  },
  { .uname  = "NEAR_CACHE",
    .udesc  = "Fill from cache of different CCX in same node",
    .ucode  = 0x4,
  },
  { .uname  = "DRAM_IO_NEAR",
    .udesc  = "Fill from DRAM or IO connected to same node",
    .ucode  = 0x8,
  },
  { .uname  = "FAR_CACHE",
    .udesc  = "Fill from CCX cache in different node",
    .ucode  = 0x10,
  },
  { .uname  = "DRAM_IO_FAR",
    .udesc  = "Fill from DRAM or IO connected from a different node (same socket or remote)",
    .ucode  = 0x40,
  },
  { .uname  = "ALT_MEM",
    .udesc  = "Fill from Extension Memory",
    .ucode  = 0x80,
  },
};

static const amd64_entry_t amd64_fam1ah_zen5_pe[]={
  { .name   = "RETIRED_X87_FP_OPS",
    .desc   = "Number of X87 floating-point ops retired",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x2,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_retired_x87_fp_ops),
    .umasks = amd64_fam1ah_zen5_retired_x87_fp_ops,
  },
  { .name   = "RETIRED_SSE_AVX_FLOPS",
    .desc   = "This is a retire-based event. The number of retired SSE/AVX FLOPS. The number of events logged per cycle can vary from 0 to 64. This event can count above 15 and therefore requires the MergeEvent",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x3,
    .flags   = 0,
    .ngrp    = 2,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_retired_sse_avx_flops),
    .umasks = amd64_fam1ah_zen5_retired_sse_avx_flops,
  },
  { .name   = "RETIRED_FP_OPS_BY_WIDTH",
    .desc   = "The number of retired floating-point ops by width",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x8,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_retired_fp_ops_by_width),
    .umasks = amd64_fam1ah_zen5_retired_fp_ops_by_width,
  },
  { .name   = "RETIRED_FP_OPS_BY_TYPE",
    .desc   = "The number of retired floating-point ops by type",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xa,
    .flags   = 0,
    .ngrp    = 2,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_retired_fp_ops_by_type),
    .umasks = amd64_fam1ah_zen5_retired_fp_ops_by_type,
  },
  { .name   = "RETIRED_INT_OPS",
    .desc   = "The number of retired integer ops (SSE/AVX)",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xb,
    .flags   = 0,
    .ngrp    = 2,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_retired_int_ops),
    .umasks = amd64_fam1ah_zen5_retired_int_ops,
  },
  { .name   = "PACKED_FP_OPS_RETIRED",
    .desc   = "The number of packed floating-point operations",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xc,
    .flags   = 0,
    .ngrp    = 2,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_packed_fp_ops_retired),
    .umasks = amd64_fam1ah_zen5_packed_fp_ops_retired,
  },
  { .name   = "PACKED_INT_OPS_RETIRED",
    .desc   = "The number of packed integer operations",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xd,
    .flags   = 0,
    .ngrp    = 2,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_packed_int_ops_retired),
    .umasks = amd64_fam1ah_zen5_packed_int_ops_retired,
  },
  { .name   = "FP_DISPATCH_FAULTS",
    .desc   = "Floating-point dispatch faults",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xe,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_fp_dispatch_faults),
    .umasks = amd64_fam1ah_zen5_fp_dispatch_faults,
  },
  { .name   = "BAD_STATUS_2",
    .desc   = "TBD",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x24,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_bad_status_2),
    .umasks = amd64_fam1ah_zen5_bad_status_2,
  },
  { .name   = "RETIRED_LOCK_INSTRUCTIONS",
    .desc   = "Counts the number of retired locked instructions",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x25,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_retired_lock_instructions),
    .umasks = amd64_fam1ah_zen5_retired_lock_instructions,
  },
  { .name   = "RETIRED_CLFLUSH_INSTRUCTIONS",
    .desc   = "Counts the number of retired non-speculative clflush instructions",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x26,
    .flags   = 0,
  },
  { .name   = "RETIRED_CPUID_INSTRUCTIONS",
    .desc   = "Counts the number of retired cpuid instructions",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x27,
    .flags   = 0,
  },
  { .name   = "LS_DISPATCH",
    .desc   = "Counts the number of operations dispatched to the LS unit. Unit Masks ADDed",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x29,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_ls_dispatch),
    .umasks = amd64_fam1ah_zen5_ls_dispatch,
  },
  { .name   = "SMI_RECEIVED",
    .desc   = "Counts the number system management interrupts (SMI) received",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x2b,
    .flags   = 0,
  },
  { .name   = "INTERRUPT_TAKEN",
    .desc   = "Counts the number of interrupts taken",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x2c,
    .flags   = 0,
  },
  { .name   = "STORE_TO_LOAD_FORWARD",
    .desc   = "Number of STore to Load Forward hits",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x35,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "STORE_GLOBALLY_VISIBLE_CANCELS_2",
    .desc   = "Count reasons why a store coalescing buffer (SCB) commit is canceled",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x37,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_store_commit_cancels_2),
    .umasks = amd64_fam1ah_zen5_store_commit_cancels_2,
  },
  { .name   = "MAB_ALLOCATION_BY_TYPE",
    .desc   = "Counts when a LS pipe allocates a MAB entry",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x41,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_mab_allocation_by_type),
    .umasks = amd64_fam1ah_zen5_mab_allocation_by_type,
  },
  { .name   = "DEMAND_DATA_CACHE_FILLS_FROM_SYSTEM",
    .desc   = "Demand Data Cache fills by data source",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x43,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_demand_data_fills_from_system),
    .umasks = amd64_fam1ah_zen5_demand_data_fills_from_system, /* shared */
  },
  { .name   = "ANY_DATA_CACHE_FILLS_FROM_SYSTEM",
    .desc   = "Any Data Cache fills by data source",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x44,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_demand_data_fills_from_system),
    .umasks = amd64_fam1ah_zen5_demand_data_fills_from_system, /* shared */
  },
  { .name   = "L1_DTLB_MISS",
    .desc   = "L1 Data TLB misses",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x45,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_l1_dtlb_miss),
    .umasks = amd64_fam1ah_zen5_l1_dtlb_miss,
  },
  { .name   = "MISALIGNED_LOADS",
    .desc   = "Misaligned loads retired",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x47,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_misaligned_loads),
    .umasks = amd64_fam1ah_zen5_misaligned_loads,
  },
  { .name   = "PREFETCH_INSTRUCTIONS_DISPATCHED",
    .desc   = "Software Prefetch Instructions Dispatched. This is a speculative event",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x4b,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_prefetch_instructions_dispatched),
    .umasks = amd64_fam1ah_zen5_prefetch_instructions_dispatched,
  },
  { .name   = "WRITE_COMBINING_BUFFER_CLOSE",
    .desc   = "Counts events that cause a Write Combining Buffer (WCB) entry to close",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x50,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_wcb_close),
    .umasks = amd64_fam1ah_zen5_wcb_close,
  },
  { .name   = "INEFFECTIVE_SOFTWARE_PREFETCH",
    .desc   = "Number of software prefetches that did not fetch data outside of the processor core",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x52,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_ineffective_software_prefetch),
    .umasks = amd64_fam1ah_zen5_ineffective_software_prefetch,
  },
  { .name   = "SOFTWARE_PREFETCH_DATA_CACHE_FILLS",
    .desc   = "Number of software prefetches fills by data source",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x59,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_demand_data_fills_from_system),
    .umasks = amd64_fam1ah_zen5_demand_data_fills_from_system, /* shared */
  },
  { .name   = "HARDWARE_PREFETCH_DATA_CACHE_FILLS",
    .desc   = "Number of hardware prefetches fills by data source",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x5a,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_demand_data_fills_from_system),
    .umasks = amd64_fam1ah_zen5_demand_data_fills_from_system, /* shared */
  },
  { .name   = "ALLOC_MAB_COUNT",
    .desc   = "Counts the in-flight L1 data cache misses each cycle",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x5f,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "CYCLES_NOT_IN_HALT",
    .desc   = "Number of core cycles not in halted state",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x76,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "TLB_FLUSHES",
    .desc   = "Number of TLB flushes",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x78,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_tlb_flushes),
    .umasks = amd64_fam1ah_zen5_tlb_flushes,
  },
  { .name   = "P0_FREQ_CYCLES_NOT_IN_HALT",
    .desc   = "Number of core cycles not in halted state by P-level",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x120,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_p0_freq_cycles_not_in_halt),
    .umasks = amd64_fam1ah_zen5_p0_freq_cycles_not_in_halt,
  },
  { .name   = "INSTRUCTION_CACHE_REFILLS_FROM_L2",
    .desc   = "Number of 64-byte instruction cachelines that was fulfilled by the L2 cache",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x82,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "INSTRUCTION_CACHE_REFILLS_FROM_SYSTEM",
    .desc   = "Number of 64-byte instruction cachelines fulfilled from system memory or another cache",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x83,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "L1_ITLB_MISS_L2_ITLB_HIT",
    .desc   = "Number of instruction fetches that miss in the L1 ITLB but hit in the L2 ITLB",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x84,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "L1_ITLB_MISS_L2_ITLB_MISS",
    .desc   = "The number of valid fills into the ITLB originating from the LS Page-Table Walker. Tablewalk requests are issued for L1-ITLB and L2-ITLB misses",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x85,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_l1_itlb_miss_l2_itlb_miss),
    .umasks = amd64_fam1ah_zen5_l1_itlb_miss_l2_itlb_miss,
  },
  { .name   = "BP_CORRECTION",
    .desc   = "Number of times the Branch Predictor (BP) flushed its own pipeline due to internal conditions such as a second level prediction structure. Does not count the number of bubbles caused by these internal flushes",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x8b,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "DYNAMIC_INDIRECT_PREDICTIONS",
    .desc   = "Number of times a branch used the indirect predictor to make a prediction",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x8e,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "DECODER_OVERRIDE_BRANCH_PRED",
    .desc   = "Number of decoder overrides of existing branch prediction",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x91,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "L1_ITLB_FETCH_HIT",
    .desc   = "Instruction fetches that hit in the L1 ITLB",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x94,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_itlb_fetch_hit),
    .umasks = amd64_fam1ah_zen5_itlb_fetch_hit,
  },
  { .name   = "BP_REDIRECTS",
    .desc   = "Counts redirects of the branch predictor",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x9f,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_bp_redirects),
    .umasks = amd64_fam1ah_zen5_bp_redirects,
  },
  { .name   = "FETCH_IBS_EVENTS",
    .desc   = "Counts significant Fetch Instruction Based Sampling (IBS) state transitions",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x188,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_fetch_ibs_events),
    .umasks = amd64_fam1ah_zen5_fetch_ibs_events,
  },
  { .name   = "IC_TAG_HIT_MISS",
    .desc   = "Counts various IC tag related hit and miss events",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x18e,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_ic_tag_hit_miss),
    .umasks = amd64_fam1ah_zen5_ic_tag_hit_miss,
  },
  { .name   = "OP_CACHE_HIT_MISS",
    .desc   = "Counts op cache micro-tag hit/miss events",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x28f,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_op_cache_hit_miss),
    .umasks = amd64_fam1ah_zen5_op_cache_hit_miss,
  },
  { .name   = "OPS_QUEUE_EMPTY",
    .desc   = "Number of cycles where the op queue is empty",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xa9,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "OPS_SOURCE_DISPATCHED_FROM_DECODER",
    .desc   = "Number of ops dispatched from the decoder classified by op source",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xaa,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_ops_source_dispatched_from_decoder),
    .umasks = amd64_fam1ah_zen5_ops_source_dispatched_from_decoder,
  },
  { .name   = "OPS_TYPE_DISPATCHED_FROM_DECODER",
    .desc   = "Number of ops dispatched from the decoder classified by op type",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xab,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_ops_type_dispatched_from_decoder),
    .umasks = amd64_fam1ah_zen5_ops_type_dispatched_from_decoder,
  },
  { .name   = "DISPATCH_RESOURCE_STALL_CYCLES_1",
    .desc   = "Number of cycles where a dispatch group is valid but does not get dispatched due to a Token Stall",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xae,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_dispatch_resource_stall_cycles_1),
    .umasks = amd64_fam1ah_zen5_dispatch_resource_stall_cycles_1,
  },
  { .name   = "DISPATCH_RESOURCE_STALL_CYCLES_2",
    .desc   = "Number of cycles where a dispatch group is valid but does not get dispatched due to a Token Stall",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xaf,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_dispatch_resource_stall_cycles_2),
    .umasks = amd64_fam1ah_zen5_dispatch_resource_stall_cycles_2,
  },
  { .name   = "DISPATCH_STALLS_1",
    .desc   = "For each cycle, counts the number of dispatch slots that remained unused for a given reason",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x1a0,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_dispatch_stalls_1),
    .umasks = amd64_fam1ah_zen5_dispatch_stalls_1,
  },
  { .name   = "DISPATCH_STALLS_2",
    .desc   = "For each cycle, counts the number of dispatch slots that remained unused for a given reason",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x1a2,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_dispatch_stalls_2),
    .umasks = amd64_fam1ah_zen5_dispatch_stalls_2,
  },
  { .name   = "RETIRED_INSTRUCTIONS",
    .desc   = "Number of instructions retired",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xc0,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "RETIRED_OPS",
    .desc   = "Number of macro-ops retired",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xc1,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "RETIRED_BRANCH_INSTRUCTIONS",
    .desc   = "Number of branch instructions retired. This includes all types of architectural control flow changes, including exceptions and interrupts",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xc2,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "RETIRED_BRANCH_INSTRUCTIONS_MISPREDICTED",
    .desc   = "Number of retired branch instructions, that were mispredicted",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xc3,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "RETIRED_TAKEN_BRANCH_INSTRUCTIONS",
    .desc   = "Number of taken branches that were retired. This includes all types of architectural control flow changes, including exceptions and interrupts",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xc4,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "RETIRED_TAKEN_BRANCH_INSTRUCTIONS_MISPREDICTED",
    .desc   = "Number of retired taken branch instructions that were mispredicted",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xc5,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "RETIRED_FAR_CONTROL_TRANSFERS",
    .desc   = "Number of far control transfers retired including far call/jump/return, IRET, SYSCALL and SYSRET, plus exceptions and interrupts. Far control transfers are not subject to branch prediction",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xc6,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "RETIRED_NEAR_RETURNS",
    .desc   = "Number of near return instructions (RET or RET Iw) retired",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xc8,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "RETIRED_NEAR_RETURNS_MISPREDICTED",
    .desc   = "Number of near returns retired that were not correctly predicted by the return address predictor. Each such mispredict incurs the same penalty as a mispredicted conditional branch instruction. Only EX mispredicts are counted",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xc9,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "RETIRED_INDIRECT_BRANCH_INSTRUCTIONS_MISPREDICTED",
    .desc   = "Number of indirect branches retired there were not correctly predicted. Each such mispredict incurs the same penalty as a mispredicted condition branch instruction. Only EX mispredicts are counted",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xca,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "RETIRED_MMX_FP_INSTRUCTIONS",
    .desc   = "Number of MMX, SSE or x87 instructions retired. The UnitMask allows the selection of the individual classes of instructions as given in the table. Each increment represents one complete instruction. Since this event includes non-numeric instructions, it is not suitable for measuring MFLOPS",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xcb,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_retired_mmx_fp_instructions),
    .umasks = amd64_fam1ah_zen5_retired_mmx_fp_instructions,
  },
  { .name   = "RETIRED_INDIRECT_BRANCH_INSTRUCTIONS",
    .desc   = "Number of indirect branches retired",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xcc,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "RETIRED_CONDITIONAL_BRANCH_INSTRUCTIONS",
    .desc   = "Number of retired conditional branch instructions",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xd1,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "DIV_CYCLES_BUSY_COUNT",
    .desc   = "Number of cycles when the divider is busy",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xd3,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "DIV_OP_COUNT",
    .desc   = "Number of divide ops",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xd4,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "CYCLES_NO_RETIRE",
    .desc   = "Counts cycles when the hardware does not retire any ops for a given reason. Event can only track one reason at a time. If multiple reasons apply for a given cycle, the lowest numbered reason is counted",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0xd6,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_cycles_no_retire),
    .umasks = amd64_fam1ah_zen5_cycles_no_retire,
  },
  { .name   = "RETIRED_UCODE_INSTRUCTIONS",
    .desc   = "Number of microcode instructions retired",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x1c1,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "RETIRED_UCODE_OPS",
    .desc   = "Number of microcode ops retired",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x1c2,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "RETIRED_BRANCH_MISPREDICTED_DIRECTION_MISMATCH",
    .desc   = "Number of retired conditional branch instructions that were not correctly predicted because of branch direction mismatch",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x1c7,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "RETIRED_UNCONDITIONAL_INDIRECT_BRANCH_INSTRUCTIONS_MISPREDICTED",
    .desc   = "Number of retired unconditional indirect branch instructions that were mispredicted",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x1c8,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "RETIRED_UNCONDITIONAL_BRANCH_INSTRUCTIONS",
    .desc   = "Number of retired unconditional branch instructions",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x1c9,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "TAGGED_IBS_OPS",
    .desc   = "Counts Op IBS related events",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x1cf,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_tagged_ibs_ops),
    .umasks = amd64_fam1ah_zen5_tagged_ibs_ops,
  },
  { .name   = "RETIRED_FUSED_INSTRUCTIONS",
    .desc   = "Counts retired fused instructions",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x1d0,
    .flags   = 0,
    .ngrp    = 0,
  },
  { .name   = "REQUESTS_TO_L2_GROUP1",
    .desc   = "All L2 cache requests",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x60,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_requests_to_l2_group1),
    .umasks = amd64_fam1ah_zen5_requests_to_l2_group1,
  },
  { .name   = "REQUESTS_TO_L2_GROUP2",
    .desc   = "All L2 cache requests (rare cases)",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x61,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_requests_to_l2_group2),
    .umasks = amd64_fam1ah_zen5_requests_to_l2_group2,
  },
  { .name   = "LS_TO_L2_WBC_REQUESTS",
    .desc   = "Number of write combining buffer (WCB) operations",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x63,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_ls_to_l2_wbc_requests),
    .umasks = amd64_fam1ah_zen5_ls_to_l2_wbc_requests,
  },
  { .name    = "CORE_TO_L2_CACHEABLE_REQUEST_ACCESS_STATUS",
    .desc    = "L2 cache request outcomes. This event does not count accesses to the L2 cache by the L2 prefetcher",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x64,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_core_to_l2_cacheable_request_access_status),
    .umasks = amd64_fam1ah_zen5_core_to_l2_cacheable_request_access_status,
  },
  { .name    = "L2_PREFETCH_HIT_L2",
    .desc    = "Number of L2 prefetches that hit in the L2",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x70,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_l2_prefetch_hit_l2),
    .umasks  = amd64_fam1ah_zen5_l2_prefetch_hit_l2,  /* shared */
  },
  { .name    = "L2_PREFETCH_HIT_L3",
    .desc    = "Number of L2 prefetches accepted by the L2 pipeline which miss the L2 cache and hit the L3",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x71,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_l2_prefetch_hit_l2),
    .umasks  = amd64_fam1ah_zen5_l2_prefetch_hit_l2,  /* shared */
  },
  { .name    = "L2_FILL_RESPONSE_SRC",
    .desc    = "Number of L2 prefetches accepted by the L2 pipeline which miss the L2 and the L3 caches",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x165,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_l2_fill_resp_src),
    .umasks = amd64_fam1ah_zen5_l2_fill_resp_src,
  },
  { .name    = "L2_PREFETCH_MISS_L3",
    .desc    = "Number of L2 prefetches accepted by the L2 pipeline which miss the L2 and the L3 caches",
    .modmsk  = AMD64_FAM1AH_ATTRS,
    .code    = 0x72,
    .flags   = 0,
    .ngrp    = 1,
    .numasks = LIBPFM_ARRAY_SIZE(amd64_fam1ah_zen5_l2_prefetch_hit_l2),
    .umasks = amd64_fam1ah_zen5_l2_prefetch_hit_l2,  /* shared */
  },
};
