% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fisher_test.R
\name{fisher_test}
\alias{fisher_test}
\alias{pairwise_fisher_test}
\alias{row_wise_fisher_test}
\title{Fisher's Exact Test for Count Data}
\usage{
fisher_test(
  xtab,
  workspace = 2e+05,
  alternative = "two.sided",
  conf.int = TRUE,
  conf.level = 0.95,
  simulate.p.value = FALSE,
  B = 2000,
  detailed = FALSE,
  ...
)

pairwise_fisher_test(xtab, p.adjust.method = "holm", detailed = FALSE, ...)

row_wise_fisher_test(xtab, p.adjust.method = "holm", detailed = FALSE, ...)
}
\arguments{
\item{xtab}{a contingency table in a matrix form.}

\item{workspace}{an integer specifying the size of the workspace
    used in the network algorithm.  In units of 4 bytes.  Only used for
    non-simulated p-values larger than \eqn{2 \times 2}{2 by 2} tables.
    Since \R version 3.5.0, this also increases the internal stack size
    which allows larger problems to be solved, however sometimes needing
    hours.  In such cases, \code{simulate.p.values=TRUE} may be more
    reasonable.}

\item{alternative}{indicates the alternative hypothesis and must be
    one of \code{"two.sided"}, \code{"greater"} or \code{"less"}.
    You can specify just the initial letter.  Only used in the
    \eqn{2 \times 2}{2 by 2} case.}

\item{conf.int}{logical indicating if a confidence interval for the
    odds ratio in a \eqn{2 \times 2}{2 by 2} table should be
    computed (and returned).}

\item{conf.level}{confidence level for the returned confidence
    interval.  Only used in the \eqn{2 \times 2}{2 by 2} case and if
    \code{conf.int = TRUE}.}

\item{simulate.p.value}{a logical indicating whether to compute
    p-values by Monte Carlo simulation, in larger than \eqn{2 \times
      2}{2 by 2} tables.}

\item{B}{an integer specifying the number of replicates used in the
    Monte Carlo test.}

\item{detailed}{logical value. Default is FALSE. If TRUE, a detailed result is
shown.}

\item{...}{Other arguments passed to the function \code{fisher_test()}.}

\item{p.adjust.method}{method to adjust p values for multiple comparisons.
Used when pairwise comparisons are performed. Allowed values include "holm",
"hochberg", "hommel", "bonferroni", "BH", "BY", "fdr", "none". If you don't
want to adjust the p value (not recommended), use p.adjust.method = "none".}
}
\value{
return a data frame with some the following columns: \itemize{ \item
 \code{group}: the categories in the row-wise proportion tests. \item
 \code{p}: p-value. \item \code{p.adj}: the adjusted p-value. \item
 \code{method}: the used statistical test. \item \code{p.signif,
 p.adj.signif}: the significance level of p-values and adjusted p-values,
 respectively. \item \code{estimate}: an estimate of the odds ratio. Only
 present in the 2 by 2 case. \item \code{alternative}: a character string
 describing the alternative hypothesis. \item \code{conf.low,conf.high}: a
 confidence interval for the odds ratio. Only present in the 2 by 2 case and
 if argument conf.int = TRUE.}

 The \strong{returned object has an attribute called args}, which is a list
 holding the test arguments.
}
\description{
Performs Fisher's exact test for testing the null of independence
 of rows and columns in a contingency table.

 Wrappers around the R base function \code{\link[stats]{fisher.test}()} but
 have the advantage of performing pairwise and row-wise fisher tests, the
 post-hoc tests following a significant chi-square test of homogeneity for 2xc
 and rx2 contingency tables.
}
\section{Functions}{
\itemize{
\item \code{fisher_test()}: performs Fisher's exact test for testing the null of
independence of rows and columns in a contingency table with fixed
marginals. Wrapper around the function \code{\link[stats]{fisher.test}()}.

\item \code{pairwise_fisher_test()}: pairwise comparisons between proportions, a post-hoc
tests following a significant Fisher's exact test of homogeneity for 2xc
design.

\item \code{row_wise_fisher_test()}: performs row-wise Fisher's exact test of count data, a post-hoc tests following a significant chi-square test
of homogeneity for rx2 contingency table. The test is conducted for each category (row).

}}
\examples{

# Comparing two proportions
#\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%
# Data: frequencies of smokers between two groups
xtab <- as.table(rbind(c(490, 10), c(400, 100)))
dimnames(xtab) <- list(
  group = c("grp1", "grp2"),
  smoker = c("yes", "no")
)
xtab
# compare the proportion of smokers
fisher_test(xtab, detailed = TRUE)

# Homogeneity of proportions between groups
#\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%
# H0: the proportion of smokers is similar in the four groups
# Ha:  this proportion is different in at least one of the populations.
#
# Data preparation
grp.size <- c( 106, 113, 156, 102 )
smokers  <- c( 50, 100, 139, 80 )
no.smokers <- grp.size - smokers
xtab <- as.table(rbind(
  smokers,
  no.smokers
))
dimnames(xtab) <- list(
  Smokers = c("Yes", "No"),
  Groups = c("grp1", "grp2", "grp3", "grp4")
)
xtab

# Compare the proportions of smokers between groups
fisher_test(xtab, detailed = TRUE)

# Pairwise comparison between groups
pairwise_fisher_test(xtab)


# Pairwise proportion tests
#\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%
# Data: Titanic
xtab <- as.table(rbind(
  c(122, 167, 528, 673),
  c(203, 118, 178, 212)
))
dimnames(xtab) <- list(
  Survived = c("No", "Yes"),
  Class = c("1st", "2nd", "3rd", "Crew")
)
xtab
# Compare the proportion of survived between groups
pairwise_fisher_test(xtab)

# Row-wise proportion tests
#\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%
# Data: Titanic
xtab <- as.table(rbind(
  c(180, 145), c(179, 106),
  c(510, 196), c(862, 23)
))
dimnames(xtab) <- list(
  Class = c("1st", "2nd", "3rd", "Crew"),
  Gender = c("Male", "Female")
)
xtab
# Compare the proportion of males and females in each category
row_wise_fisher_test(xtab)

# A r x c table  Agresti (2002, p. 57) Job Satisfaction
Job <- matrix(c(1,2,1,0, 3,3,6,1, 10,10,14,9, 6,7,12,11), 4, 4,
              dimnames = list(income = c("< 15k", "15-25k", "25-40k", "> 40k"),
                             satisfaction = c("VeryD", "LittleD", "ModerateS", "VeryS")))
fisher_test(Job)
fisher_test(Job, simulate.p.value = TRUE, B = 1e5)
}
