#!/usr/bin/python

# ----------------------------------------------------------------------------
#
#  Copyright (C) 2018-2022 Fons Adriaensen <fons@linuxaudio.org>
#    
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http:#www.gnu.org/licenses/>.
#
# ----------------------------------------------------------------------------


import numpy as np
import matplotlib.pyplot as plt
from zita_jacktools.jacklr4filt import JackLR4filt
from zita_jacktools.jacksignal import JackSignal


# Measure frequency response of Linkwitz-Riley filters
#

# Create and connect objects.
#
J = JackSignal("JackSignal")
if J.get_state() < 0:
    print ("Failed to create JackSignal -- is Jack running ?")
    exit(1)
J.create_output (0, "out")
J.create_input (0, "in-0")
J.create_input (1, "in-1")
J.create_input (2, "in-2")
J.silence()
F = JackLR4filt (1, 3, 0.1, "JackLR4filt")

J.connect_output (0, "JackLR4filt:in_0")
J.connect_input (0, "JackLR4filt:out_0")
J.connect_input (1, "JackLR4filt:out_1")
J.connect_input (2, "JackLR4filt:out_2")


# Get Jack info.
#
name, fsamp, period = J.get_jack_info ()

# Parameters.
#
impval = 1.0
fftlen = 16 * 1024

# Generate data.
#
Aout = np.zeros ([100,], dtype = np.float32)
Aout [0] = impval
Ahp = np.empty ([fftlen,], dtype = np.float32)
Alp = np.empty ([fftlen,], dtype = np.float32)
Abp = np.empty ([fftlen,], dtype = np.float32)
Fax = np.linspace (0, fsamp / 2, num = fftlen // 2 + 1)

J.set_output_data (0, Aout)
J.set_input_data (0, Ahp, nskip = period)
J.set_input_data (1, Alp, nskip = period)
J.set_input_data (2, Abp, nskip = period)

F.set_filter (0, 0, 500, -6.0, 0, 0)   # HP, 500 Hz 
F.set_filter (0, 1, 0, 0, 500, -6.0)   # LP, 500 Hz
F.set_filter (0, 2, 50, -3, 200, -3.0) # BP, 50...200 Hz

# Some experiments.
#F.set_gain (0, 0.316) # -10 dB
#F.set_delay (0, 0.01) # 10 ms

# Measure.
#
J.process()
J.wait()
del J, F

Fhp = np.fft.rfft (Ahp)
Flp = np.fft.rfft (Alp)
Fbp = np.fft.rfft (Abp)

# Plot result.
#
fig = plt.figure (figsize=(8,5), facecolor='white')
fig.canvas.manager.set_window_title ('oct1band')
ax1 = fig.add_axes ([0.06, 0.06, 0.90, 0.90])
ax1.set_xlim (1e1, 24e3)
ax1.set_ylim (-50, 5)
ax1.set_xscale ('log')
ax1.grid ()
ax1.plot (Fax, 20 * np.log10 (np.abs (Fhp) + 1e-10), color='b', lw=1)
ax1.plot (Fax, 20 * np.log10 (np.abs (Flp) + 1e-10), color='g', lw=1)
ax1.plot (Fax, 20 * np.log10 (np.abs (Fhp + Flp) + 1e-10), color='r', lw=1)
ax1.plot (Fax, 20 * np.log10 (np.abs (Fbp) + 1e-10), color='k', lw=1)
plt.show()

