/*  This file is part of "choosewm"
 *  Copyright (C) 2005,2009 Bernhard R. Link
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include <config.h>

#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#define GTK_DISABLE_DEPRECATED 1
#include <gtk/gtk.h>
#include <stdbool.h>

#include "global.h"

/*#include <locale.h>
#include <libintl.h>*/

static GtkWidget *window;
static GtkWidget *checker;
static GtkTreeSelection *wmselection;
static gboolean textmode = false;
static gboolean configmode = false;
static gboolean printmode = false;
static gboolean askagain = true;
static gint wmnumber = -1;

#define STRINGS S(CONFIGWELCOME), S(WELCOME), S(CONFIGHEADER), S(HEADER), S(CONFIGASK), S(ASK), S(CONFIGSELECT), S(SELECT), S(CONFIGCANCEL), S(CANCEL)
const char *stringkeys[] = {
#define S(s)  #s
	STRINGS
#undef S
	,NULL
};
enum stringoffsets {
#define S(s) STRING_ ## s
	STRINGS
#undef S
	,STRINGS_count
};
char *stringvalues[STRINGS_count];

#define _(a) a

static void inittrings(void) {
	stringvalues[STRING_CONFIGHEADER] = xstrdup(_("Choose default Window Manager"));
	stringvalues[STRING_HEADER] = xstrdup(_("Choose Window Manager"));
	stringvalues[STRING_CONFIGWELCOME] = xstrdup(_("Please choose the default window manager and whether to reask at start time."));
	stringvalues[STRING_WELCOME] = xstrdup(_("Welcome, please select a Window manager to use."));
	stringvalues[STRING_CONFIGASK] = xstrdup(_("_Ask when starting"));
	stringvalues[STRING_ASK] = xstrdup(_("_Ask again at every login"));
	stringvalues[STRING_CONFIGSELECT] = xstrdup(_("_Select"));
	stringvalues[STRING_SELECT] = xstrdup(_("_Start"));
	stringvalues[STRING_CONFIGCANCEL] = xstrdup(_("_Cancel"));
	stringvalues[STRING_CANCEL] = xstrdup(_("_Log out"));
}

enum { col_name=0, col_number, cols_count};

static void logout_clicked(GtkWidget *widget, gpointer data) {
	gtk_main_quit();
}
static void start_clicked(GtkWidget *widget, gpointer data) {
	GtkTreeModel     *model;
	GtkTreeIter       iter;

	askagain = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checker));

	if(gtk_tree_selection_get_selected(wmselection, &model, &iter)) {

		gtk_tree_model_get(model, &iter, col_number, &wmnumber, -1);
	} else {
		wmnumber = -1;
	}
	gtk_main_quit();
}

static void delete_event(GtkWidget *widget, gpointer data) {
	gtk_main_quit();
}

static void row_activated(GtkWidget *widget, gpointer data) {
	gtk_window_activate_default(GTK_WINDOW(window));
}

gboolean defaultavailable = FALSE;
GtkTreeIter defaultiter;

static void makeWMItems(GtkListStore *store) {
	struct windowmanager *wm;
	gint number = 0;

	for( wm = windowmanagers; wm != NULL ; wm = wm->next,number++ ) {
		GtkTreeIter iter;

		gtk_list_store_append(store,&iter);
		gtk_list_store_set(store,&iter,
			col_name,wm->name,
			col_number,number,
			-1);
		if( wm->isdefault ) {
			defaultiter = iter;
			defaultavailable = TRUE;
		}
	}
}

static gint wmsortfunc(GtkTreeModel *model, GtkTreeIter  *item1, GtkTreeIter  *item2, gpointer      user_data) {
	gchar *name1,*name2;
	gint ret;

	gtk_tree_model_get(model,item1,col_name,&name1,-1);
	gtk_tree_model_get(model,item2,col_name,&name2,-1);
	if( name1 == NULL )
		if( name2 == NULL )
			ret = 0;
		else
			ret =  -1;
	else
		if( name1 == NULL )
			ret = 1;
		else
			ret = g_utf8_collate(name1,name2);
	g_free(name1);
	g_free(name2);
	return ret;
}

static void makeWMList(GtkBox *outerbox) {
	GtkWidget *frame,*list;
	GtkListStore *store = gtk_list_store_new(cols_count,G_TYPE_STRING,G_TYPE_INT);

	makeWMItems(store);

	gtk_tree_sortable_set_default_sort_func(GTK_TREE_SORTABLE(store),
			&wmsortfunc,NULL,NULL);
	gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(store),
			GTK_TREE_SORTABLE_DEFAULT_SORT_COLUMN_ID,
			GTK_SORT_ASCENDING);

	list = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));
	gtk_tree_view_append_column(GTK_TREE_VIEW(list),
		gtk_tree_view_column_new_with_attributes(
			stringvalues[configmode?STRING_CONFIGHEADER:STRING_HEADER],
			gtk_cell_renderer_text_new(),
			"text",col_name,NULL));
	wmselection = gtk_tree_view_get_selection(GTK_TREE_VIEW(list));
	gtk_tree_selection_set_mode(wmselection,GTK_SELECTION_BROWSE);

	if( defaultavailable ) {
		gtk_tree_selection_select_iter(wmselection,&defaultiter);

	}

	/* enter presses login-button */
	g_signal_connect(G_OBJECT(list),
			"row-activated",
			G_CALLBACK(row_activated),NULL);


	frame = gtk_scrolled_window_new(NULL,NULL);

	gtk_container_add(GTK_CONTAINER(frame),list);
	gtk_widget_show(list);
	gtk_box_pack_start(outerbox,frame,TRUE,TRUE,0);
	gtk_widget_show(frame);
}

static void makeGreeter(GtkBox *outerbox) {
	GtkWidget *greeter;

	greeter = gtk_label_new(
		stringvalues[configmode?STRING_CONFIGWELCOME:STRING_WELCOME]);
	gtk_box_pack_start(outerbox,greeter,FALSE,TRUE,3);
	gtk_widget_show(greeter);

}

static void makeChecker(GtkBox *outerbox) {
	checker = gtk_check_button_new_with_mnemonic(
		stringvalues[configmode?STRING_CONFIGASK:STRING_ASK]);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checker),askagain);

	gtk_box_pack_start(outerbox,checker,FALSE,TRUE,3);
	gtk_widget_show(checker);

}
GtkWidget *startbutton;

static void makeControls(GtkBox *outerbox) {
	GtkWidget *box;
	GtkWidget *logoutbutton;

	box = gtk_hbox_new(TRUE, 10);

	startbutton = gtk_button_new_with_mnemonic(
		stringvalues[configmode?STRING_CONFIGSELECT:STRING_SELECT]);
	g_signal_connect(G_OBJECT(startbutton),
			"clicked",
			G_CALLBACK(start_clicked),NULL);
	logoutbutton = gtk_button_new_with_mnemonic(
		stringvalues[configmode?STRING_CONFIGCANCEL:STRING_CANCEL]);
	g_signal_connect(G_OBJECT(logoutbutton),
			"clicked",
			G_CALLBACK(logout_clicked),NULL);
	gtk_box_pack_start(GTK_BOX(box),startbutton,TRUE,TRUE,10);
	gtk_widget_show(startbutton);
	gtk_box_pack_end(GTK_BOX(box),logoutbutton,TRUE,TRUE,10);
	gtk_widget_show(logoutbutton);

	gtk_box_pack_end(outerbox,box,FALSE,TRUE,3);
	gtk_widget_show(box);
}

static void makeWindowContent(GtkContainer *container) {
	GtkWidget *vbox;

	vbox = gtk_vbox_new(FALSE, 3);

	makeGreeter(GTK_BOX(vbox));
	makeWMList(GTK_BOX(vbox));
	makeChecker(GTK_BOX(vbox));
	makeControls(GTK_BOX(vbox));

	gtk_container_add(container,vbox);
	gtk_widget_show(vbox);
}

int main(int argc, char *argv[]){
	const struct option longopts[] = {
		{"text",no_argument,NULL,'t'},
		{"print",no_argument,NULL,'p'},
		{"debug",no_argument,NULL,'d'},
		{"force",no_argument,NULL,'f'},
		{"config",no_argument,NULL,'c'},
		{"help",no_argument,NULL,'h'},
		{NULL,no_argument,NULL,0}
	};
	int opt;
	gboolean forced = FALSE;
	GdkScreen *screen;

/*	setlocale(LC_MESSAGES,"");
	if( textdomain("choosewm") == NULL) {
		g_print("Error loading locale!\n");
		return EXIT_FAILURE;
	}*/
	inittrings();
	while( (opt = getopt_long(argc,argv,"dcfpht",longopts,NULL)) != -1 ) {
		switch( opt ) {
			case 'h':
				puts(
"choosewm - select a window manager to be used\n"
"Syntax: choosewm [--force] [--print|--config]\n"
" --text:	Ask in terminal\n"
" --config:	Do not start but only change setting\n"
" --print:	Do not start but print\n"
" --force:	Always ask\n");
				return EXIT_SUCCESS;
			case 't':
				textmode = TRUE;
				break;
			case 'd':
				break;
			case 'c':
				configmode = TRUE;
				break;
			case 'p':
				printmode = TRUE;
				break;
			case 'f':
				forced = TRUE;
				break;
			case '?':
				exit(EXIT_FAILURE);
			default:
				fprintf(stderr,"Unknown option %c\n",(char)opt);
				exit(EXIT_FAILURE);
		}
	}
	if( textmode && !isatty(0) ) {
		fprintf(stderr, "%s: ERROR: No terminal for --textmode!\n",
				argv[0]);
		return EXIT_FAILURE;
	}
	if( configmode && getenv("DISPLAY") == NULL && !isatty(0) ) {
		fprintf(stderr, "%s: ERROR: No terminal and no DISPLAY set!\n",
				argv[0]);
		return EXIT_FAILURE;
	}
	readconfigs(stringkeys,stringvalues);
	askagain = !dontask;
	if( haderrors )
		forced = TRUE;

	if( dontask && !forced && !configmode ) {
		struct windowmanager *wm;

		for( wm = windowmanagers; wm != NULL ; wm = wm->next ) {
			if( wm->isdefault ) {
				if( printmode )
					printwm(wm,dontask);
				else
					startwm(wm,dontask);
			}
		}
	}

	if( textmode || (getenv("DISPLAY") == NULL && configmode) ) {
		struct windowmanager *wm;
		char buffer[100], *s, *e;
		int i = 0, j;

		puts("Please select your default WindowManager:");
		for( wm = windowmanagers; wm != NULL ; wm = wm->next ) {
			printf("%c%2d: %s\n", wm->isdefault?'*':' ',
					i++, wm->name);
		}
		fputs("Your choice? ", stdout);
		if( fgets(buffer, 99, stdin) == NULL )
			return EXIT_FAILURE;
		s = buffer + strlen(buffer);
		while( s > buffer && (*(s-1) == '\n' || *(s-1) == '\r'
					|| *(s-1) == ' ' || *(s-1) == '\t') )
			*(--s) = '\0';
		s = buffer;
		while( *s == ' ' )
			s++;
		j = strtol(s, &e, 10);
		if( *e != '\0' && *e != '*' ) {
			fprintf(stderr, "%s: Error parsing '%s'\n", argv[0], s);
			return EXIT_FAILURE;
		}
		if( s == e || *e == '*' ) {
			wm = windowmanagers;
			wmnumber = 0;
		        while( wm != NULL && !wm->isdefault ) {
				wm = wm->next;
				wmnumber++;
			}
			if( wm == NULL ) {
				fputs("ERROR: no default window manager\n", stderr);
				return EXIT_FAILURE;
			}
		} else if( j < 0 || j >= i ) {
			fprintf(stderr, "ERROR: Not a valid choice: %d\n", j);
			return EXIT_FAILURE;
		} else
			wmnumber = j;
	} else {
		gtk_init(&argc, &argv);

		window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		g_signal_connect(G_OBJECT(window),
				"delete_event",
				G_CALLBACK(delete_event), NULL);
		gtk_container_set_border_width(GTK_CONTAINER(window), 10);
		makeWindowContent(GTK_CONTAINER(window));

		GTK_WIDGET_SET_FLAGS(startbutton, GTK_CAN_DEFAULT);
		GTK_WIDGET_SET_FLAGS(startbutton, GTK_RECEIVES_DEFAULT);
		gtk_widget_grab_default(startbutton);

		screen = gtk_window_get_screen(GTK_WINDOW(window));

		if( configmode ) {
			gtk_window_maximize(GTK_WINDOW(window));
		} else {
			/* gtk_window_fullscreen does not work when there
			 * is not yet a window manager, so make the window
			 * as large as possible manually. (This both makes
			 * sure the list is properly navigateable and the
			 * keys work without any mouse-moving or key-grabbing */
			gtk_window_set_gravity(GTK_WINDOW(window),GDK_GRAVITY_STATIC);
			gtk_window_move(GTK_WINDOW(window),0,0);
			gtk_window_resize(GTK_WINDOW(window),gdk_screen_get_width(screen),gdk_screen_get_height(screen));
		}
		gtk_widget_show(window);

		gtk_main();
	}
	if( wmnumber >= 0) {
		struct windowmanager *wm = windowmanagers;
		while( wmnumber > 0 ) {
			if( wm == NULL ) {
				fputs("Internal error!\n",stderr);
				exit(EXIT_FAILURE);
			}
			wm = wm->next;
			wmnumber--;
		}
		if( configmode )
			savewm(wm,!askagain);
		else if( printmode )
			printwm(wm,!askagain);
		else
			startwm(wm,!askagain);
	} else {
		exit(EXIT_FAILURE);
	}
}

