/*
  DB Mixer
  ========
  Description: 
  Interface for handling sample editing within dbmixer. 

  Copyright (c) 1999, 2000 Robert Michael S Dean

  Author: Robert Michael S Dean
  Version: 1.0


  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public Licensse as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

*/


#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <math.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <limits.h>
#include <fcntl.h>
#include <signal.h>
#include <gtk/gtk.h>
#include <sys/shm.h>
#include <xmms/plugin.h>
#include <dbchannel.h>
#include <dbaudiolib.h>
#include <dbdebug.h>
#include <dbsoundcard.h>

#include "dbmixer.h"

extern int * channel_indexes;
extern local_channel * local_channels;
extern dbfsd_data * sysdata;
extern channel_widgets * widgets;

static GtkToggleButton * record_button;
static int sample_channel, copy_channel;
static GtkWidget * start_scale, * end_scale;
static float start_value, end_value;
static float buftime;
static GtkWidget * sampler_option_menu;
static GtkWidget * copy_option_menu;
static GtkAdjustment * start_adj, * end_adj;
static GtkButton * copy_button, * load_button, * save_button;
GtkWidget *file_selector;
static int copy_count;

#define SAMPLES_PER_SECOND (DB_SAMPLE_RATE * DB_SAMPLE_SIZE * DB_CHANNEL_NUM_CHANNELS)

GtkWidget* make_sampler_channel_menu();


int update_sample_sliders(gpointer data)
{
	int   modvalue;
	volatile char write_flag;
	local_channel * ch;
	int   startoffset, endoffset;

	startoffset = 0;
	endoffset = 0;
	modvalue = 0;

	ch = &(local_channels[channel_indexes[sample_channel]]);

	/* pause output */
/* 	send_msg(ch,DBMSG_PAUSE,0); */
   
	/* if we are playing a sample right now, reset the sample to the begining */
	if ((ch->sampler_state == SAMPLER_PLAY_LOOP) 
		|| (ch->sampler_state == SAMPLER_PLAY_SINGLE))
	{
		ch->pause = 1;
		write_flag = ch->writing;
		
		while (write_flag)
		{
			write_flag = ch->writing;
		}
	}
	
	startoffset = (int)((float)start_value * (float)SAMPLES_PER_SECOND);

	endoffset = (int)((float)end_value * (float)SAMPLES_PER_SECOND);

	/* make sure offsets sit on boundry of a sample */
	/* if we have a portion of a sample, include that sample */
	/* get offset into buffer: time * samples/second */
	modvalue = startoffset % (DB_SAMPLE_SIZE * DB_CHANNEL_NUM_CHANNELS);

	/* adjust startoffset to sit on a sample boundry */
	if (modvalue)
	{
		startoffset += ((DB_SAMPLE_SIZE * DB_CHANNEL_NUM_CHANNELS) - modvalue);
	}

	modvalue = endoffset % (DB_SAMPLE_SIZE * DB_CHANNEL_NUM_CHANNELS);

	if (modvalue)
	{
		endoffset += ((DB_SAMPLE_SIZE * DB_CHANNEL_NUM_CHANNELS) - modvalue);
	}

	ch->sampler_startoffset = startoffset;
	ch->sampler_endoffset = endoffset;

	/* if we are playing a sample right now, reset the sample to the begining */
	if ((ch->sampler_state == SAMPLER_PLAY_LOOP) 
		|| (ch->sampler_state == SAMPLER_PLAY_SINGLE))
	{
		ch->sampler_readoffset = startoffset;
		ch->pause = 0;
	}

	/* unpause output */
/* 	send_msg(ch,DBMSG_UNPAUSE,0); */

	return SUCCESS;
}


void sampler_channel_menu_select(GtkWidget *w, gpointer * data)
{
	sample_channel = (int)data;

	update_sampler(NULL);
}


void copy_channel_menu_select(GtkWidget *w, gpointer * data)
{
	copy_channel = (int)data;
}

int update_sampler(gpointer data)
{	
	GtkWidget * tempwidget;
	local_channel * ch;
	ch = &(local_channels[channel_indexes[sample_channel]]);

	switch (ch->sampler_state)
	{
		case SAMPLER_OFF:
			/* disable playback buttons */
			gtk_widget_set_sensitive(GTK_WIDGET(widgets[sample_channel].playloop_button), FALSE);
			gtk_widget_set_sensitive(GTK_WIDGET(widgets[sample_channel].playsingle_button), FALSE);
			gtk_toggle_button_set_active(record_button,0);			

			/* disable sliders */
			gtk_widget_set_sensitive(GTK_WIDGET(start_scale), FALSE);
			gtk_widget_set_sensitive(GTK_WIDGET(end_scale), FALSE);
			break;

		case SAMPLER_READY:
			/* calculate time of sample, and set slider values */
			buftime = (float)ch->sampler_size / (float)SAMPLES_PER_SECOND;
			ch->sampler_time = buftime;
			
			gtk_adjustment_set_value(start_adj, (float)ch->sampler_startoffset / (float)SAMPLES_PER_SECOND);
			gtk_adjustment_set_value(end_adj, (float)ch->sampler_endoffset / (float)SAMPLES_PER_SECOND);
			
			/* enable plaback buttons */
			gtk_widget_set_sensitive(GTK_WIDGET(widgets[sample_channel].playloop_button),TRUE);
			gtk_widget_set_sensitive(GTK_WIDGET(widgets[sample_channel].playsingle_button),TRUE);
			
			gtk_toggle_button_set_active(record_button,0);
			gtk_toggle_button_set_active(widgets[sample_channel].playloop_button,0);

			/* turn on loop control sliders */
			gtk_widget_set_sensitive(GTK_WIDGET(start_scale), TRUE);
			gtk_widget_set_sensitive(GTK_WIDGET(end_scale), TRUE);
			break;

		case SAMPLER_RECORD:
		case SAMPLER_PLAY_LOOP:
		case SAMPLER_PLAY_SINGLE:
			break;
		default:
			
			break;
	}

	if (!GTK_WIDGET_VISIBLE(GTK_WIDGET(gtk_option_menu_get_menu
									   (GTK_OPTION_MENU(sampler_option_menu)))))
	{
		tempwidget = gtk_option_menu_get_menu(GTK_OPTION_MENU(sampler_option_menu));
		gtk_option_menu_remove_menu(GTK_OPTION_MENU(sampler_option_menu));
		/* 		gtk_widget_destroy(tempwidget); */
		tempwidget = make_sampler_channel_menu(sampler_channel_menu_select);
		gtk_menu_set_active(GTK_MENU(tempwidget),sample_channel);
		gtk_option_menu_set_menu(GTK_OPTION_MENU(sampler_option_menu),tempwidget);	
	}


	if (!GTK_WIDGET_VISIBLE(GTK_WIDGET(gtk_option_menu_get_menu
									   (GTK_OPTION_MENU(copy_option_menu)))))
	{
		tempwidget = gtk_option_menu_get_menu(GTK_OPTION_MENU(copy_option_menu));
		gtk_option_menu_remove_menu(GTK_OPTION_MENU(copy_option_menu));
		/* 		gtk_widget_destroy(tempwidget); */
		tempwidget = make_sampler_channel_menu(copy_channel_menu_select);
		gtk_menu_set_active(GTK_MENU(tempwidget),copy_channel);
		gtk_option_menu_set_menu(GTK_OPTION_MENU(copy_option_menu),tempwidget);	
	}
	
	return TRUE;
}


void start_changed(GtkAdjustment * adj)
{
	start_value = adj->value;

	/* make sure change is within range */
	if (start_value >= end_value)
	{
		start_value = end_value - (MIN_LOOP_SECONDS);

		if (start_value < 0.0)
		{
			start_value = 0.0;
		}

		gtk_adjustment_set_value(end_adj,end_value);
	}

	update_sample_sliders(NULL);
}


void end_changed(GtkAdjustment * adj)
{
	end_value = adj->value;

	if (end_value > buftime)
	{
		end_value = buftime;
		gtk_adjustment_set_value(end_adj,end_value);
	}

	if (end_value <= start_value)
	{
		end_value = start_value + (MIN_LOOP_SECONDS);

		gtk_adjustment_set_value(end_adj,end_value);
	}

	update_sample_sliders(NULL);
}


GtkWidget* make_sampler_channel_menu(GtkSignalFunc sigfxn)
{
	GtkWidget * menu;
	GtkWidget * item;
	gint i;
	gint channeldata;

	menu = gtk_menu_new();
		
	/* add options to channel selector */
	for (i = 0; i < sysdata->num_channels;i++)
	{
		channeldata = i;
		
		item = make_menu_item (local_channels[i].channel_name,
							   GTK_SIGNAL_FUNC(sigfxn),
							   (gpointer)channeldata);
		gtk_menu_append (GTK_MENU (menu), item);
	}
	
	gtk_menu_set_active(GTK_MENU(menu),(guint) sample_channel);

	return menu;
}


void sampler_record_button_clicked(GtkWidget *w, gint * data)
{
	local_channel * ch;
	gint index = sample_channel;

	ch = &(local_channels[channel_indexes[index]]);

	if (ch->free_channel)
	{
		db_message_box(NO_CLIENT_STR);
		return;
	}

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
	{
		/* if currently playing, stop playback */
		if ((ch->sampler_state == SAMPLER_PLAY_LOOP)
			|| (ch->sampler_state == SAMPLER_PLAY_SINGLE))
		{
			/* disable playback */
			ch->sampler_state = SAMPLER_READY;
			
			/* give time to stop */
			usleep(100000);
		}

		ch->sampler_size = 0;
		ch->sampler_readoffset = 0;
		ch->sampler_startoffset = 0;

		/* turn off play buttons while recording */
		gtk_widget_set_sensitive(GTK_WIDGET(widgets[index].playsingle_button), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(widgets[index].playloop_button), FALSE);

		/* turn off loop control sliders while recording */
		gtk_widget_set_sensitive(GTK_WIDGET(start_scale), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(end_scale), FALSE);

		/* set record */
		ch->sampler_state = SAMPLER_RECORD;
	}
	else
	{
		/* stop record */
		if (ch->sampler_buf != NULL)
		{
			ch->sampler_state = SAMPLER_READY;
		}
	}
}


void sampler_copy_button_clicked(GtkWidget *w, gint * data)
{
	volatile int local_flag;

	if (local_channels[sample_channel].free_channel)
	{
		db_message_box(NO_CLIENT_STR);
		return;
	}

	if (local_channels[copy_channel].free_channel)
	{
		db_message_box(NO_CLIENT_STR);
		return;
	}

	if (copy_count > 20000) copy_count = 1;

	sprintf(sysdata->filename,"/tmp/dbmixtempfile%d",copy_count);
	copy_count++;

	sysdata->sampler_op_flag = 1;

	send_msg(&(local_channels[sample_channel]),DBMSG_SAMPLERSAVE,0);

	local_flag = sysdata->sampler_op_flag;

	while (local_flag)
	{
		local_flag = sysdata->sampler_op_flag;
		usleep(50);
	}
		
	sysdata->sampler_op_flag = 1;

	send_msg(&(local_channels[copy_channel]),DBMSG_SAMPLERLOAD,0);

	local_flag = sysdata->sampler_op_flag;

	while (local_flag)
	{
		local_flag = sysdata->sampler_op_flag;
		usleep(USLEEP_TIME);
	}

	unlink(sysdata->filename);
	
	sysdata->filename[0] = '\0';
}


void sampler_load(GtkWidget *w, gint * data)
{
	volatile int local_flag;
	gchar * filename;

	/* get filename */
	filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION(file_selector));

	/* copy filename to sysdata buffer */
	strcpy(sysdata->filename,filename);
	
	sysdata->sampler_op_flag = 1;

	send_msg(&(local_channels[sample_channel]),DBMSG_SAMPLERLOAD,0);

	local_flag = sysdata->sampler_op_flag;

	while (local_flag)
	{
		local_flag = sysdata->sampler_op_flag;
		usleep(USLEEP_TIME);
	}

	update_sample_sliders(NULL);
}


void sampler_load_button_clicked(GtkWidget *w, gint * data)
{
	if (local_channels[sample_channel].free_channel)
	{
		db_message_box(NO_CLIENT_STR);
		return;
	}

	/* Create the selector */   
	file_selector = gtk_file_selection_new(SAMPLE_FILE_STR);
	
	gtk_signal_connect(GTK_OBJECT (GTK_FILE_SELECTION(file_selector)->ok_button),
						"clicked", GTK_SIGNAL_FUNC (sampler_load), NULL);
	
	/* Ensure that the dialog box is destroyed when the user clicks a button. */
	
	gtk_signal_connect_object(GTK_OBJECT (GTK_FILE_SELECTION(file_selector)->ok_button),
							   "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
							   (gpointer) file_selector);
	
	gtk_signal_connect_object(GTK_OBJECT (GTK_FILE_SELECTION(file_selector)->cancel_button),
							   "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
							   (gpointer) file_selector);
	
	/* Display that dialog */
	gtk_widget_show (file_selector);
}


void sampler_save(GtkFileSelection *selector, gpointer user_data)
{
	volatile int local_flag;
	gchar * filename;

	/* get filename */
	filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION(file_selector));

	/* copy filename to sysdata buffer */
	strcpy(sysdata->filename,filename);

	/* send message to channel */
	sysdata->sampler_op_flag = 1;

	send_msg(&(local_channels[sample_channel]),DBMSG_SAMPLERSAVE,0);

	local_flag = sysdata->sampler_op_flag;

	/* wait for save to complete */
	while (local_flag)
	{
		local_flag = sysdata->sampler_op_flag;
		usleep(USLEEP_TIME);
	}
}


void sampler_save_button_clicked(GtkWidget *w, gint * data)
{
	if (local_channels[sample_channel].free_channel)
	{
		db_message_box(NO_CLIENT_STR);
		return;
	}

	/* Create the selector */   
	file_selector = gtk_file_selection_new(SAMPLE_FILE_STR);
	
	gtk_signal_connect(GTK_OBJECT (GTK_FILE_SELECTION(file_selector)->ok_button),
						"clicked", GTK_SIGNAL_FUNC (sampler_save), NULL);
	
	/* Ensure that the dialog box is destroyed when the user clicks a button. */
	
	gtk_signal_connect_object(GTK_OBJECT (GTK_FILE_SELECTION(file_selector)->ok_button),
							   "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
							   (gpointer) file_selector);
	
	gtk_signal_connect_object(GTK_OBJECT (GTK_FILE_SELECTION(file_selector)->cancel_button),
							   "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
							   (gpointer) file_selector);
	
	/* Display that dialog */
	gtk_widget_show (file_selector);
}

/************************************************************************/

GtkWidget * create_sample_editor()
{
	GtkWidget * vbox;       /* main box for the window */
	GtkWidget * hbox;
	GtkWidget * vbox2;
	GtkWidget * hbox2;
	GtkWidget * label;
	GtkWidget * menu;  
	
	copy_count = 1;
	sample_channel = 0;
	start_value = 0.0;
	end_value = DB_SAMPLER_DEFAULT_TIME;

	/* create main box for the window */
	vbox = (GtkWidget*)gtk_vbox_new(FALSE,0);
	gtk_widget_show(vbox);

	/* make sampler editor title */
	label = gtk_label_new(SAMPLER_STR);
	gtk_box_pack_start(GTK_BOX(vbox),label,FALSE,FALSE,10);
/* 	gtk_widget_show(label);		 */
  
	/* make label for channel selection */
	sampler_option_menu = gtk_option_menu_new();
	menu = make_sampler_channel_menu(sampler_channel_menu_select);

	/* add option list to channel selector */
	gtk_option_menu_set_menu (GTK_OPTION_MENU (sampler_option_menu), menu);
	gtk_box_pack_start (GTK_BOX (vbox), sampler_option_menu, FALSE, FALSE, 0);
	gtk_widget_show (sampler_option_menu);

	/* load/save button box */
	{
		hbox2 = (GtkWidget*)gtk_hbox_new(TRUE,0);
		gtk_widget_show(hbox2);

		load_button = (GtkButton *)gtk_button_new();
		label = gtk_label_new(LOAD_SAMPLE_STR);
		gtk_container_add(GTK_CONTAINER(load_button),label);
		gtk_box_pack_start(GTK_BOX(hbox2),GTK_WIDGET(load_button),
						   TRUE,FALSE,0);
		gtk_signal_connect (GTK_OBJECT(load_button), "clicked", 
							GTK_SIGNAL_FUNC(sampler_load_button_clicked),NULL);
		gtk_widget_show(label);
		gtk_widget_show(GTK_WIDGET(load_button));

		save_button = (GtkButton *)gtk_button_new();
		label = gtk_label_new(SAVE_SAMPLE_STR);
		gtk_container_add(GTK_CONTAINER(save_button),label);
		gtk_box_pack_start(GTK_BOX(hbox2),GTK_WIDGET(save_button),
						   TRUE,FALSE,0);
		gtk_signal_connect (GTK_OBJECT(save_button), "clicked", 
							GTK_SIGNAL_FUNC(sampler_save_button_clicked),NULL);
		gtk_widget_show(label);
		gtk_widget_show(GTK_WIDGET(save_button));

		gtk_box_pack_start(GTK_BOX(vbox),hbox2,FALSE,FALSE,5);
	}
	
	/* record button */
	{
		record_button = (GtkToggleButton *)gtk_toggle_button_new();
		label = gtk_label_new(RECORD_STR);
		gtk_container_add(GTK_CONTAINER(record_button),label);
		gtk_box_pack_start(GTK_BOX(vbox),GTK_WIDGET(record_button),
						   FALSE,FALSE,5);
		gtk_signal_connect (GTK_OBJECT(record_button), "clicked", 
							GTK_SIGNAL_FUNC(sampler_record_button_clicked),NULL);
		gtk_widget_show(label);
		gtk_widget_show(GTK_WIDGET(record_button));
	}
	
	/* sample start offset */
	{
		hbox = (GtkWidget*)gtk_hbox_new(TRUE,0);
		gtk_widget_show(hbox);

		vbox2 = (GtkWidget*)gtk_vbox_new(FALSE,0);
		gtk_widget_show(vbox2);

		start_adj = (GtkAdjustment*) gtk_adjustment_new(0.0,0.0,DB_SAMPLER_DEFAULT_TIME,0.01,0.1,0.0);
	
		label = gtk_label_new("Start:");
		gtk_box_pack_start(GTK_BOX(vbox2),label,FALSE,FALSE,5);
		gtk_widget_show(label);
	
		start_scale = (GtkWidget*) gtk_vscale_new(start_adj);
		gtk_scale_set_digits((GtkScale*)start_scale,2);

		gtk_box_pack_start(GTK_BOX(vbox2),(GtkWidget*)start_scale,FALSE,FALSE,0);		
	
		gtk_signal_connect (GTK_OBJECT(start_adj), "value_changed", 
							GTK_SIGNAL_FUNC(start_changed),NULL);

		gtk_widget_show(start_scale);

		gtk_box_pack_start(GTK_BOX(hbox),GTK_WIDGET(vbox2),FALSE,FALSE,0);

		gtk_widget_set_usize(start_scale,30,100);
	}	

	/* sample end offset */
	{
		vbox2 = (GtkWidget*)gtk_vbox_new(FALSE,0);
		gtk_widget_show(vbox2);
		
		end_adj = (GtkAdjustment*) gtk_adjustment_new(DB_SAMPLER_DEFAULT_TIME,0.0,DB_SAMPLER_DEFAULT_TIME,0.01,0.1,0.0);
		
		label = gtk_label_new("End:");
		gtk_box_pack_start(GTK_BOX(vbox2),label,FALSE,FALSE,5);
		gtk_widget_show(label);
		
		end_scale = (GtkWidget*) gtk_vscale_new(end_adj);
		gtk_scale_set_digits((GtkScale*)end_scale,2);

		gtk_box_pack_start(GTK_BOX(vbox2),(GtkWidget*)end_scale,FALSE,FALSE,0);		
	
		gtk_signal_connect (GTK_OBJECT(end_adj), "value_changed", 
							GTK_SIGNAL_FUNC(end_changed),NULL);

		gtk_widget_show(end_scale);

		gtk_box_pack_start(GTK_BOX(hbox),GTK_WIDGET(vbox2),FALSE,FALSE,0);
				
		gtk_widget_show(hbox);

		gtk_box_pack_start(GTK_BOX(vbox),GTK_WIDGET(hbox),FALSE,FALSE,0);

		gtk_widget_set_usize(end_scale,30,100);
	}	

	/* add channel selector for copy functionality */
	{
		/* copy button */
		{
			copy_button = (GtkButton *)gtk_button_new();
			label = gtk_label_new(COPY_SAMPLE_STR);
			gtk_container_add(GTK_CONTAINER(copy_button),label);
			gtk_box_pack_start(GTK_BOX(vbox),GTK_WIDGET(copy_button),
							   FALSE,FALSE,5);
			gtk_signal_connect (GTK_OBJECT(copy_button), "clicked", 
								GTK_SIGNAL_FUNC(sampler_copy_button_clicked),NULL);
			gtk_widget_show(label);
			gtk_widget_show(GTK_WIDGET(copy_button));
		}
		
		/* add channel selector */
		{
			/* make label for channel selection */
			copy_option_menu = gtk_option_menu_new();
			menu = make_sampler_channel_menu(copy_channel_menu_select);
			
			/* add option list to channel selector */
			gtk_option_menu_set_menu (GTK_OPTION_MENU (copy_option_menu), menu);
			gtk_box_pack_start (GTK_BOX (vbox), copy_option_menu, FALSE, FALSE, 0);
			gtk_widget_show (copy_option_menu);
		}
	}

	update_sampler(NULL);
	update_sample_sliders(NULL);

	return vbox;
}
