#  Licensed to Elasticsearch B.V. under one or more contributor
#  license agreements. See the NOTICE file distributed with
#  this work for additional information regarding copyright
#  ownership. Elasticsearch B.V. licenses this file to you under
#  the Apache License, Version 2.0 (the "License"); you may
#  not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
# 	http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing,
#  software distributed under the License is distributed on an
#  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
#  KIND, either express or implied.  See the License for the
#  specific language governing permissions and limitations
#  under the License.

import typing as t

from elastic_transport import ObjectApiResponse

from ._base import NamespacedClient
from .utils import (
    Stability,
    _availability_warning,
    _rewrite_parameters,
)


class ProjectClient(NamespacedClient):

    @_rewrite_parameters()
    @_availability_warning(Stability.EXPERIMENTAL)
    def tags(
        self,
        *,
        error_trace: t.Optional[bool] = None,
        filter_path: t.Optional[t.Union[str, t.Sequence[str]]] = None,
        human: t.Optional[bool] = None,
        pretty: t.Optional[bool] = None,
        project_routing: t.Optional[str] = None,
    ) -> ObjectApiResponse[t.Any]:
        """
        .. raw:: html

          <p>Get tags.</p>
          <p>Get the tags that are defined for the project.</p>


        `<https://www.elastic.co/docs/api/doc/elasticsearch-serverless/operation/operation-project-tags>`_

        :param project_routing: A Lucene query using project metadata tags used to filter
            which projects are returned in the response, such as _alias:_origin or _alias:*pr*.
        """
        __path_parts: t.Dict[str, str] = {}
        __path = "/_project/tags"
        __query: t.Dict[str, t.Any] = {}
        if error_trace is not None:
            __query["error_trace"] = error_trace
        if filter_path is not None:
            __query["filter_path"] = filter_path
        if human is not None:
            __query["human"] = human
        if pretty is not None:
            __query["pretty"] = pretty
        if project_routing is not None:
            __query["project_routing"] = project_routing
        __headers = {"accept": "application/json"}
        return self.perform_request(  # type: ignore[return-value]
            "POST",
            __path,
            params=__query,
            headers=__headers,
            endpoint_id="project.tags",
            path_parts=__path_parts,
        )
