﻿' Licensed to the .NET Foundation under one or more agreements.
' The .NET Foundation licenses this file to you under the MIT license.
' See the LICENSE file in the project root for more information.

Imports Microsoft.CodeAnalysis.Test.Utilities
Imports Roslyn.Test.Utilities

Namespace Microsoft.CodeAnalysis.VisualBasic.UnitTests

    Public Class CodeGenCallTests
        Inherits BasicTestBase

        <Theory>
        <CombinatorialData>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Call_Class(asRValue As Boolean)

            Dim leftParen As String = ""
            Dim rightParen As String = ""

            If asRValue Then
                leftParen = "("
                rightParen = ")"
            End If

            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Sub GetName(x As Integer)
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Sub GetName(x As Integer) Implements IMoveable.GetName
        Console.WriteLine("Position GetName for item '{0}'", Me.Name)
    End Sub
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)

        Dim item3 = New Item With {.Name = "3"}
        Call3(item3)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(item As T)
        call <%= leftParen %>item<%= rightParen %>.GetName(GetOffset(item))
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        call <%= leftParen %>item<%= rightParen %>.GetName(GetOffset(item))
    End Sub

    Private Shared Sub Call3(item As Item)
        call <%= leftParen %>item<%= rightParen %>.GetName(GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position GetName for item '1'
Position GetName for item '2'
Position GetName for item '3'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       19 (0x13)
  .maxstack  2
  IL_0000:  ldarg.0
  IL_0001:  box        "T"
  IL_0006:  ldarga.s   V_0
  IL_0008:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_000d:  callvirt   "Sub IMoveable.GetName(Integer)"
  IL_0012:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
  // Code size       45 (0x2d)
  .maxstack  2
  .locals init (T V_0)
  IL_0000:  ldarga.s   V_0
  IL_0002:  ldloca.s   V_0
  IL_0004:  initobj    "T"
  IL_000a:  ldloc.0
  IL_000b:  box        "T"
  IL_0010:  brtrue.s   IL_001a
  IL_0012:  ldobj      "T"
  IL_0017:  stloc.0
  IL_0018:  ldloca.s   V_0
  IL_001a:  ldarga.s   V_0
  IL_001c:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0021:  constrained. "T"
  IL_0027:  callvirt   "Sub IMoveable.GetName(Integer)"
  IL_002c:  ret
}
]]>)

            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Theory>
        <CombinatorialData>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Call_Class_InWith(asRValue As Boolean)

            Dim leftParen As String = ""
            Dim rightParen As String = ""

            If asRValue Then
                leftParen = "("
                rightParen = ")"
            End If

            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Sub GetName(x As Integer)
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Sub GetName(x As Integer) Implements IMoveable.GetName
        Console.WriteLine("Position GetName for item '{0}'", Me.Name)
    End Sub
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)

        Dim item3 = New Item With {.Name = "3"}
        Call3(item3)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(item As T)
        With <%= leftParen %>item<%= rightParen %>
            call .GetName(GetOffset(item))
        End With
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        With <%= leftParen %>item<%= rightParen %>
            call .GetName(GetOffset(item))
        End With
    End Sub

    Private Shared Sub Call3(item As Item)
        With <%= leftParen %>item<%= rightParen %>
            call .GetName(GetOffset(item))
        End With
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position GetName for item '1'
Position GetName for item '2'
Position GetName for item '3'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       19 (0x13)
  .maxstack  2
  IL_0000:  ldarg.0
  IL_0001:  box        "T"
  IL_0006:  ldarga.s   V_0
  IL_0008:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_000d:  callvirt   "Sub IMoveable.GetName(Integer)"
  IL_0012:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
                              If(asRValue,
            <![CDATA[
{
      // Code size       55 (0x37)
  .maxstack  2
  .locals init (T V_0, //$W0
                T V_1)
  IL_0000:  ldarg.0
  IL_0001:  stloc.0
  IL_0002:  ldloca.s   V_0
  IL_0004:  ldloca.s   V_1
  IL_0006:  initobj    "T"
  IL_000c:  ldloc.1
  IL_000d:  box        "T"
  IL_0012:  brtrue.s   IL_001c
  IL_0014:  ldobj      "T"
  IL_0019:  stloc.1
  IL_001a:  ldloca.s   V_1
  IL_001c:  ldarga.s   V_0
  IL_001e:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0023:  constrained. "T"
  IL_0029:  callvirt   "Sub IMoveable.GetName(Integer)"
  IL_002e:  ldloca.s   V_0
  IL_0030:  initobj    "T"
  IL_0036:  ret
}
]]>,
            <![CDATA[
{
      // Code size       45 (0x2d)
  .maxstack  2
  .locals init (T V_0)
  IL_0000:  ldarga.s   V_0
  IL_0002:  ldloca.s   V_0
  IL_0004:  initobj    "T"
  IL_000a:  ldloc.0
  IL_000b:  box        "T"
  IL_0010:  brtrue.s   IL_001a
  IL_0012:  ldobj      "T"
  IL_0017:  stloc.0
  IL_0018:  ldloca.s   V_0
  IL_001a:  ldarga.s   V_0
  IL_001c:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0021:  constrained. "T"
  IL_0027:  callvirt   "Sub IMoveable.GetName(Integer)"
  IL_002c:  ret
}
]]>))
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Theory>
        <CombinatorialData>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Call_Struct(asRValue As Boolean)

            Dim leftParen As String = ""
            Dim rightParen As String = ""

            If asRValue Then
                leftParen = "("
                rightParen = ")"
            End If

            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Sub GetName(x As Integer)
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Sub GetName(x As Integer) Implements IMoveable.GetName
        Console.WriteLine("Position GetName for item '{0}'", Me.Name)
    End Sub
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)

        Dim item3 = New Item With {.Name = "3"}
        Call3(item3)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(item As T)
        call <%= leftParen %>item<%= rightParen %>.GetName(GetOffset(item))
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        call <%= leftParen %>item<%= rightParen %>.GetName(GetOffset(item))
    End Sub

    Private Shared Sub Call3(item As Item)
        call <%= leftParen %>item<%= rightParen %>.GetName(GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position GetName for item '-1'
Position GetName for item '-2'
Position GetName for item '-3'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       21 (0x15)
  .maxstack  2
  IL_0000:  ldarga.s   V_0
  IL_0002:  ldarga.s   V_0
  IL_0004:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0009:  constrained. "T"
  IL_000f:  callvirt   "Sub IMoveable.GetName(Integer)"
  IL_0014:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Theory>
        <CombinatorialData>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Call_Class_Ref(asRValue As Boolean)

            Dim leftParen As String = ""
            Dim rightParen As String = ""

            If asRValue Then
                leftParen = "("
                rightParen = ")"
            End If

            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Sub GetName(x As Integer)
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Sub GetName(x As Integer) Implements IMoveable.GetName
        Console.WriteLine("Position GetName for item '{0}'", Me.Name)
    End Sub
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)

        Dim item3 = New Item With {.Name = "3"}
        Call3(item3)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(ByRef item As T)
        call <%= leftParen %>item<%= rightParen %>.GetName(GetOffset(item))
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        call <%= leftParen %>item<%= rightParen %>.GetName(GetOffset(item))
    End Sub

    Private Shared Sub Call3(ByRef item As Item)
        call <%= leftParen %>item<%= rightParen %>.GetName(GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position GetName for item '1'
Position GetName for item '2'
Position GetName for item '3'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       23 (0x17)
  .maxstack  2
  IL_0000:  ldarg.0
  IL_0001:  ldobj      "T"
  IL_0006:  box        "T"
  IL_000b:  ldarg.0
  IL_000c:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0011:  callvirt   "Sub IMoveable.GetName(Integer)"
  IL_0016:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
  // Code size       43 (0x2b)
  .maxstack  2
  .locals init (T V_0)
  IL_0000:  ldarg.0
  IL_0001:  ldloca.s   V_0
  IL_0003:  initobj    "T"
  IL_0009:  ldloc.0
  IL_000a:  box        "T"
  IL_000f:  brtrue.s   IL_0019
  IL_0011:  ldobj      "T"
  IL_0016:  stloc.0
  IL_0017:  ldloca.s   V_0
  IL_0019:  ldarg.0
  IL_001a:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_001f:  constrained. "T"
  IL_0025:  callvirt   "Sub IMoveable.GetName(Integer)"
  IL_002a:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Theory>
        <CombinatorialData>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Call_Struct_Ref(asRValue As Boolean)

            Dim leftParen As String = ""
            Dim rightParen As String = ""

            If asRValue Then
                leftParen = "("
                rightParen = ")"
            End If

            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Sub GetName(x As Integer)
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Sub GetName(x As Integer) Implements IMoveable.GetName
        Console.WriteLine("Position GetName for item '{0}'", Me.Name)
    End Sub
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)

        Dim item3 = New Item With {.Name = "3"}
        Call3(item3)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(ByRef item As T)
        call <%= leftParen %>item<%= rightParen %>.GetName(GetOffset(item))
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        call <%= leftParen %>item<%= rightParen %>.GetName(GetOffset(item))
    End Sub

    Private Shared Sub Call3(ByRef item As Item)
        call <%= leftParen %>item<%= rightParen %>.GetName(GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position GetName for item '-1'
Position GetName for item '-2'
Position GetName for item '-3'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       19 (0x13)
  .maxstack  2
  IL_0000:  ldarg.0
  IL_0001:  ldarg.0
  IL_0002:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0007:  constrained. "T"
  IL_000d:  callvirt   "Sub IMoveable.GetName(Integer)"
  IL_0012:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Theory>
        <CombinatorialData>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Call_Class_Async_01(asRValue As Boolean)

            Dim leftParen As String = ""
            Dim rightParen As String = ""

            If asRValue Then
                leftParen = "("
                rightParen = ")"
            End If

            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Sub GetName(x As Integer)
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Sub GetName(x As Integer) Implements IMoveable.GetName
        Console.WriteLine("Position GetName for item '{0}'", Me.Name)
    End Sub
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()

        Dim item3 = New Item With {.Name = "3"}
        Call3(item3).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        call <%= leftParen %>item<%= rightParen %>.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        call <%= leftParen %>item<%= rightParen %>.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call3(item As Item) As Task
        call <%= leftParen %>item<%= rightParen %>.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position GetName for item '1'
Position GetName for item '2'
Position GetName for item '3'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      217 (0xd9)
  .maxstack  3
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                System.Exception V_2)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0058
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0011:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0016:  ldarg.0
    IL_0017:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_001c:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0021:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0026:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_002b:  stloc.1
    IL_002c:  ldloca.s   V_1
    IL_002e:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0033:  brtrue.s   IL_0074
    IL_0035:  ldarg.0
    IL_0036:  ldc.i4.0
    IL_0037:  dup
    IL_0038:  stloc.0
    IL_0039:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_003e:  ldarg.0
    IL_003f:  ldloc.1
    IL_0040:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0045:  ldarg.0
    IL_0046:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_004b:  ldloca.s   V_1
    IL_004d:  ldarg.0
    IL_004e:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0053:  leave      IL_00d8
    IL_0058:  ldarg.0
    IL_0059:  ldc.i4.m1
    IL_005a:  dup
    IL_005b:  stloc.0
    IL_005c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0061:  ldarg.0
    IL_0062:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0067:  stloc.1
    IL_0068:  ldarg.0
    IL_0069:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_006e:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0074:  ldarg.0
    IL_0075:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_007a:  box        "SM$T"
    IL_007f:  ldloca.s   V_1
    IL_0081:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0086:  ldloca.s   V_1
    IL_0088:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008e:  callvirt   "Sub IMoveable.GetName(Integer)"
    IL_0093:  ldarg.0
    IL_0094:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0099:  initobj    "SM$T"
    IL_009f:  leave.s    IL_00c3
  }
  catch System.Exception
  {
    IL_00a1:  dup
    IL_00a2:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00a7:  stloc.2
    IL_00a8:  ldarg.0
    IL_00a9:  ldc.i4.s   -2
    IL_00ab:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00b0:  ldarg.0
    IL_00b1:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00b6:  ldloc.2
    IL_00b7:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00bc:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00c1:  leave.s    IL_00d8
  }
  IL_00c3:  ldarg.0
  IL_00c4:  ldc.i4.s   -2
  IL_00c6:  dup
  IL_00c7:  stloc.0
  IL_00c8:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00cd:  ldarg.0
  IL_00ce:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00d3:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_00d8:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      258 (0x102)
  .maxstack  3
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                SM$T V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0068
    IL_000a:  ldloca.s   V_2
    IL_000c:  initobj    "SM$T"
    IL_0012:  ldloc.2
    IL_0013:  box        "SM$T"
    IL_0018:  brtrue.s   IL_0026
    IL_001a:  ldarg.0
    IL_001b:  ldarg.0
    IL_001c:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0021:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T"
    IL_0026:  ldarg.0
    IL_0027:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_002c:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0031:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0036:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_003b:  stloc.1
    IL_003c:  ldloca.s   V_1
    IL_003e:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0043:  brtrue.s   IL_0084
    IL_0045:  ldarg.0
    IL_0046:  ldc.i4.0
    IL_0047:  dup
    IL_0048:  stloc.0
    IL_0049:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_004e:  ldarg.0
    IL_004f:  ldloc.1
    IL_0050:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0055:  ldarg.0
    IL_0056:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_005b:  ldloca.s   V_1
    IL_005d:  ldarg.0
    IL_005e:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_0063:  leave      IL_0101
    IL_0068:  ldarg.0
    IL_0069:  ldc.i4.m1
    IL_006a:  dup
    IL_006b:  stloc.0
    IL_006c:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0071:  ldarg.0
    IL_0072:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0077:  stloc.1
    IL_0078:  ldarg.0
    IL_0079:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007e:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0084:  ldloca.s   V_2
    IL_0086:  initobj    "SM$T"
    IL_008c:  ldloc.2
    IL_008d:  box        "SM$T"
    IL_0092:  brtrue.s   IL_009c
    IL_0094:  ldarg.0
    IL_0095:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T"
    IL_009a:  br.s       IL_00a2
    IL_009c:  ldarg.0
    IL_009d:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00a2:  ldloca.s   V_1
    IL_00a4:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00a9:  ldloca.s   V_1
    IL_00ab:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b1:  constrained. "SM$T"
    IL_00b7:  callvirt   "Sub IMoveable.GetName(Integer)"
    IL_00bc:  ldarg.0
    IL_00bd:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T"
    IL_00c2:  initobj    "SM$T"
    IL_00c8:  leave.s    IL_00ec
  }
  catch System.Exception
  {
    IL_00ca:  dup
    IL_00cb:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00d0:  stloc.3
    IL_00d1:  ldarg.0
    IL_00d2:  ldc.i4.s   -2
    IL_00d4:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00d9:  ldarg.0
    IL_00da:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00df:  ldloc.3
    IL_00e0:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00e5:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00ea:  leave.s    IL_0101
  }
  IL_00ec:  ldarg.0
  IL_00ed:  ldc.i4.s   -2
  IL_00ef:  dup
  IL_00f0:  stloc.0
  IL_00f1:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_00f6:  ldarg.0
  IL_00f7:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00fc:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0101:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Theory>
        <CombinatorialData>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Call_Struct_Async_01(asRValue As Boolean)

            Dim leftParen As String = ""
            Dim rightParen As String = ""

            If asRValue Then
                leftParen = "("
                rightParen = ")"
            End If

            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Sub GetName(x As Integer)
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Sub GetName(x As Integer) Implements IMoveable.GetName
        Console.WriteLine("Position GetName for item '{0}'", Me.Name)
    End Sub
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()

        Dim item3 = New Item With {.Name = "3"}
        Call3(item3).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        call <%= leftParen %>item<%= rightParen %>.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        call <%= leftParen %>item<%= rightParen %>.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call3(item As Item) As Task
        call <%= leftParen %>item<%= rightParen %>.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position GetName for item '-1'
Position GetName for item '-2'
Position GetName for item '-3'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      191 (0xbf)
  .maxstack  3
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                System.Exception V_2)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0049
    IL_000a:  ldarg.0
    IL_000b:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0010:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0015:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_001a:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_001f:  stloc.1
    IL_0020:  ldloca.s   V_1
    IL_0022:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0027:  brtrue.s   IL_0065
    IL_0029:  ldarg.0
    IL_002a:  ldc.i4.0
    IL_002b:  dup
    IL_002c:  stloc.0
    IL_002d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0032:  ldarg.0
    IL_0033:  ldloc.1
    IL_0034:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0039:  ldarg.0
    IL_003a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003f:  ldloca.s   V_1
    IL_0041:  ldarg.0
    IL_0042:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0047:  leave.s    IL_00be
    IL_0049:  ldarg.0
    IL_004a:  ldc.i4.m1
    IL_004b:  dup
    IL_004c:  stloc.0
    IL_004d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0052:  ldarg.0
    IL_0053:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0058:  stloc.1
    IL_0059:  ldarg.0
    IL_005a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_005f:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0065:  ldarg.0
    IL_0066:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_006b:  ldloca.s   V_1
    IL_006d:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0072:  ldloca.s   V_1
    IL_0074:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007a:  constrained. "SM$T"
    IL_0080:  callvirt   "Sub IMoveable.GetName(Integer)"
    IL_0085:  leave.s    IL_00a9
  }
  catch System.Exception
  {
    IL_0087:  dup
    IL_0088:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_008d:  stloc.2
    IL_008e:  ldarg.0
    IL_008f:  ldc.i4.s   -2
    IL_0091:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0096:  ldarg.0
    IL_0097:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_009c:  ldloc.2
    IL_009d:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00a2:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00a7:  leave.s    IL_00be
  }
  IL_00a9:  ldarg.0
  IL_00aa:  ldc.i4.s   -2
  IL_00ac:  dup
  IL_00ad:  stloc.0
  IL_00ae:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00b3:  ldarg.0
  IL_00b4:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00b9:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_00be:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Call_Class_Async_02()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Sub GetName(x As Integer)
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Sub GetName(x As Integer) Implements IMoveable.GetName
        Console.WriteLine("Position GetName for item '{0}'", Me.Name)
    End Sub
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        await Task.Yield()
        item.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        await Task.Yield()
        item.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position GetName for item '1'
Position GetName for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      327 (0x147)
  .maxstack  3
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter V_1,
                System.Runtime.CompilerServices.YieldAwaitable V_2,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004b
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_00c4
    IL_0011:  call       "Function System.Threading.Tasks.Task.Yield() As System.Runtime.CompilerServices.YieldAwaitable"
    IL_0016:  stloc.2
    IL_0017:  ldloca.s   V_2
    IL_0019:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.GetAwaiter() As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_001e:  stloc.1
    IL_001f:  ldloca.s   V_1
    IL_0021:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.get_IsCompleted() As Boolean"
    IL_0026:  brtrue.s   IL_0067
    IL_0028:  ldarg.0
    IL_0029:  ldc.i4.0
    IL_002a:  dup
    IL_002b:  stloc.0
    IL_002c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0031:  ldarg.0
    IL_0032:  ldloc.1
    IL_0033:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0038:  ldarg.0
    IL_0039:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003e:  ldloca.s   V_1
    IL_0040:  ldarg.0
    IL_0041:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0046:  leave      IL_0146
    IL_004b:  ldarg.0
    IL_004c:  ldc.i4.m1
    IL_004d:  dup
    IL_004e:  stloc.0
    IL_004f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0054:  ldarg.0
    IL_0055:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_005a:  stloc.1
    IL_005b:  ldarg.0
    IL_005c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0061:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0067:  ldloca.s   V_1
    IL_0069:  call       "Sub System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.GetResult()"
    IL_006e:  ldloca.s   V_1
    IL_0070:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0076:  ldarg.0
    IL_0077:  ldarg.0
    IL_0078:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_007d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0082:  ldarg.0
    IL_0083:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0088:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_008d:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0092:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0097:  stloc.3
    IL_0098:  ldloca.s   V_3
    IL_009a:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_009f:  brtrue.s   IL_00e0
    IL_00a1:  ldarg.0
    IL_00a2:  ldc.i4.1
    IL_00a3:  dup
    IL_00a4:  stloc.0
    IL_00a5:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00aa:  ldarg.0
    IL_00ab:  ldloc.3
    IL_00ac:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b1:  ldarg.0
    IL_00b2:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00b7:  ldloca.s   V_3
    IL_00b9:  ldarg.0
    IL_00ba:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_00bf:  leave      IL_0146
    IL_00c4:  ldarg.0
    IL_00c5:  ldc.i4.m1
    IL_00c6:  dup
    IL_00c7:  stloc.0
    IL_00c8:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00cd:  ldarg.0
    IL_00ce:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d3:  stloc.3
    IL_00d4:  ldarg.0
    IL_00d5:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00da:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00e0:  ldarg.0
    IL_00e1:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_00e6:  box        "SM$T"
    IL_00eb:  ldloca.s   V_3
    IL_00ed:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00f2:  ldloca.s   V_3
    IL_00f4:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00fa:  callvirt   "Sub IMoveable.GetName(Integer)"
    IL_00ff:  ldarg.0
    IL_0100:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0105:  initobj    "SM$T"
    IL_010b:  leave.s    IL_0131
  }
  catch System.Exception
  {
    IL_010d:  dup
    IL_010e:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0113:  stloc.s    V_4
    IL_0115:  ldarg.0
    IL_0116:  ldc.i4.s   -2
    IL_0118:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_011d:  ldarg.0
    IL_011e:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0123:  ldloc.s    V_4
    IL_0125:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_012a:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_012f:  leave.s    IL_0146
  }
  IL_0131:  ldarg.0
  IL_0132:  ldc.i4.s   -2
  IL_0134:  dup
  IL_0135:  stloc.0
  IL_0136:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_013b:  ldarg.0
  IL_013c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0141:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0146:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      370 (0x172)
  .maxstack  3
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter V_1,
                System.Runtime.CompilerServices.YieldAwaitable V_2,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_3,
                SM$T V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004b
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_00d5
    IL_0011:  call       "Function System.Threading.Tasks.Task.Yield() As System.Runtime.CompilerServices.YieldAwaitable"
    IL_0016:  stloc.2
    IL_0017:  ldloca.s   V_2
    IL_0019:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.GetAwaiter() As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_001e:  stloc.1
    IL_001f:  ldloca.s   V_1
    IL_0021:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.get_IsCompleted() As Boolean"
    IL_0026:  brtrue.s   IL_0067
    IL_0028:  ldarg.0
    IL_0029:  ldc.i4.0
    IL_002a:  dup
    IL_002b:  stloc.0
    IL_002c:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0031:  ldarg.0
    IL_0032:  ldloc.1
    IL_0033:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0038:  ldarg.0
    IL_0039:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003e:  ldloca.s   V_1
    IL_0040:  ldarg.0
    IL_0041:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_0046:  leave      IL_0171
    IL_004b:  ldarg.0
    IL_004c:  ldc.i4.m1
    IL_004d:  dup
    IL_004e:  stloc.0
    IL_004f:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0054:  ldarg.0
    IL_0055:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_005a:  stloc.1
    IL_005b:  ldarg.0
    IL_005c:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0061:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0067:  ldloca.s   V_1
    IL_0069:  call       "Sub System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.GetResult()"
    IL_006e:  ldloca.s   V_1
    IL_0070:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0076:  ldloca.s   V_4
    IL_0078:  initobj    "SM$T"
    IL_007e:  ldloc.s    V_4
    IL_0080:  box        "SM$T"
    IL_0085:  brtrue.s   IL_0093
    IL_0087:  ldarg.0
    IL_0088:  ldarg.0
    IL_0089:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_008e:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T"
    IL_0093:  ldarg.0
    IL_0094:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0099:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_009e:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_00a3:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a8:  stloc.3
    IL_00a9:  ldloca.s   V_3
    IL_00ab:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_00b0:  brtrue.s   IL_00f1
    IL_00b2:  ldarg.0
    IL_00b3:  ldc.i4.1
    IL_00b4:  dup
    IL_00b5:  stloc.0
    IL_00b6:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00bb:  ldarg.0
    IL_00bc:  ldloc.3
    IL_00bd:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00c2:  ldarg.0
    IL_00c3:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00c8:  ldloca.s   V_3
    IL_00ca:  ldarg.0
    IL_00cb:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_00d0:  leave      IL_0171
    IL_00d5:  ldarg.0
    IL_00d6:  ldc.i4.m1
    IL_00d7:  dup
    IL_00d8:  stloc.0
    IL_00d9:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00de:  ldarg.0
    IL_00df:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00e4:  stloc.3
    IL_00e5:  ldarg.0
    IL_00e6:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00eb:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00f1:  ldloca.s   V_4
    IL_00f3:  initobj    "SM$T"
    IL_00f9:  ldloc.s    V_4
    IL_00fb:  box        "SM$T"
    IL_0100:  brtrue.s   IL_010a
    IL_0102:  ldarg.0
    IL_0103:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T"
    IL_0108:  br.s       IL_0110
    IL_010a:  ldarg.0
    IL_010b:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0110:  ldloca.s   V_3
    IL_0112:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0117:  ldloca.s   V_3
    IL_0119:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_011f:  constrained. "SM$T"
    IL_0125:  callvirt   "Sub IMoveable.GetName(Integer)"
    IL_012a:  ldarg.0
    IL_012b:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T"
    IL_0130:  initobj    "SM$T"
    IL_0136:  leave.s    IL_015c
  }
  catch System.Exception
  {
    IL_0138:  dup
    IL_0139:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_013e:  stloc.s    V_5
    IL_0140:  ldarg.0
    IL_0141:  ldc.i4.s   -2
    IL_0143:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0148:  ldarg.0
    IL_0149:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_014e:  ldloc.s    V_5
    IL_0150:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0155:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_015a:  leave.s    IL_0171
  }
  IL_015c:  ldarg.0
  IL_015d:  ldc.i4.s   -2
  IL_015f:  dup
  IL_0160:  stloc.0
  IL_0161:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0166:  ldarg.0
  IL_0167:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_016c:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0171:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Call_Struct_Async_02()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Sub GetName(x As Integer)
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Sub GetName(x As Integer) Implements IMoveable.GetName
        Console.WriteLine("Position GetName for item '{0}'", Me.Name)
    End Sub
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        await Task.Yield()
        item.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        await Task.Yield()
        item.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position GetName for item '-1'
Position GetName for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      301 (0x12d)
  .maxstack  3
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter V_1,
                System.Runtime.CompilerServices.YieldAwaitable V_2,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004b
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_00b5
    IL_0011:  call       "Function System.Threading.Tasks.Task.Yield() As System.Runtime.CompilerServices.YieldAwaitable"
    IL_0016:  stloc.2
    IL_0017:  ldloca.s   V_2
    IL_0019:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.GetAwaiter() As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_001e:  stloc.1
    IL_001f:  ldloca.s   V_1
    IL_0021:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.get_IsCompleted() As Boolean"
    IL_0026:  brtrue.s   IL_0067
    IL_0028:  ldarg.0
    IL_0029:  ldc.i4.0
    IL_002a:  dup
    IL_002b:  stloc.0
    IL_002c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0031:  ldarg.0
    IL_0032:  ldloc.1
    IL_0033:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0038:  ldarg.0
    IL_0039:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003e:  ldloca.s   V_1
    IL_0040:  ldarg.0
    IL_0041:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0046:  leave      IL_012c
    IL_004b:  ldarg.0
    IL_004c:  ldc.i4.m1
    IL_004d:  dup
    IL_004e:  stloc.0
    IL_004f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0054:  ldarg.0
    IL_0055:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_005a:  stloc.1
    IL_005b:  ldarg.0
    IL_005c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0061:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0067:  ldloca.s   V_1
    IL_0069:  call       "Sub System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.GetResult()"
    IL_006e:  ldloca.s   V_1
    IL_0070:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0076:  ldarg.0
    IL_0077:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_007c:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0081:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0086:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008b:  stloc.3
    IL_008c:  ldloca.s   V_3
    IL_008e:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0093:  brtrue.s   IL_00d1
    IL_0095:  ldarg.0
    IL_0096:  ldc.i4.1
    IL_0097:  dup
    IL_0098:  stloc.0
    IL_0099:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_009e:  ldarg.0
    IL_009f:  ldloc.3
    IL_00a0:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a5:  ldarg.0
    IL_00a6:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00ab:  ldloca.s   V_3
    IL_00ad:  ldarg.0
    IL_00ae:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_00b3:  leave.s    IL_012c
    IL_00b5:  ldarg.0
    IL_00b6:  ldc.i4.m1
    IL_00b7:  dup
    IL_00b8:  stloc.0
    IL_00b9:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00be:  ldarg.0
    IL_00bf:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00c4:  stloc.3
    IL_00c5:  ldarg.0
    IL_00c6:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00cb:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d1:  ldarg.0
    IL_00d2:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_00d7:  ldloca.s   V_3
    IL_00d9:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00de:  ldloca.s   V_3
    IL_00e0:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00e6:  constrained. "SM$T"
    IL_00ec:  callvirt   "Sub IMoveable.GetName(Integer)"
    IL_00f1:  leave.s    IL_0117
  }
  catch System.Exception
  {
    IL_00f3:  dup
    IL_00f4:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00f9:  stloc.s    V_4
    IL_00fb:  ldarg.0
    IL_00fc:  ldc.i4.s   -2
    IL_00fe:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0103:  ldarg.0
    IL_0104:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0109:  ldloc.s    V_4
    IL_010b:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0110:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0115:  leave.s    IL_012c
  }
  IL_0117:  ldarg.0
  IL_0118:  ldc.i4.s   -2
  IL_011a:  dup
  IL_011b:  stloc.0
  IL_011c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0121:  ldarg.0
  IL_0122:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0127:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_012c:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Theory>
        <CombinatorialData>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Call_Class_Async_01_ThroughArray(asRValue As Boolean)

            Dim leftParen As String = ""
            Dim rightParen As String = ""

            If asRValue Then
                leftParen = "("
                rightParen = ")"
            End If

            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Sub GetName(x As Integer)
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Sub GetName(x As Integer) Implements IMoveable.GetName
        Console.WriteLine("Position GetName for item '{0}'", Me.Name)
    End Sub
End Class

Class Item2
    Inherits Item
End Class

Class Program
    Shared Sub Main()
        Dim item1 = {New Item2 With {.Name = "1"}}
        Call1(DirectCast(item1, Item())).Wait()

        Dim item2 = {New Item2 With {.Name = "2"}}
        Call2(DirectCast(item2, Item())).Wait()

        Dim item3 = {New Item2 With {.Name = "3"}}
        Call3(DirectCast(item3, Item())).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T()) As Task
        call <%= leftParen %>item(GetArrayIndex())<%= rightParen %>.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T()) As Task
        call <%= leftParen %>item(GetArrayIndex())<%= rightParen %>.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call3(item As Item()) As Task
        call <%= leftParen %>item(GetArrayIndex())<%= rightParen %>.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T()) As Integer
        value -= 1
        item(0) = DirectCast(DirectCast(New Item2 With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Function GetArrayIndex() As Integer
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position GetName for item '1'
Position GetName for item '2'
Position GetName for item '3'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      227 (0xe3)
  .maxstack  3
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                System.Exception V_2)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0062
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0011:  call       "Function Program.GetArrayIndex() As Integer"
    IL_0016:  ldelem     "SM$T"
    IL_001b:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0020:  ldarg.0
    IL_0021:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0026:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T()) As Integer"
    IL_002b:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0030:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0035:  stloc.1
    IL_0036:  ldloca.s   V_1
    IL_0038:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_003d:  brtrue.s   IL_007e
    IL_003f:  ldarg.0
    IL_0040:  ldc.i4.0
    IL_0041:  dup
    IL_0042:  stloc.0
    IL_0043:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0048:  ldarg.0
    IL_0049:  ldloc.1
    IL_004a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_004f:  ldarg.0
    IL_0050:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0055:  ldloca.s   V_1
    IL_0057:  ldarg.0
    IL_0058:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_005d:  leave      IL_00e2
    IL_0062:  ldarg.0
    IL_0063:  ldc.i4.m1
    IL_0064:  dup
    IL_0065:  stloc.0
    IL_0066:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_006b:  ldarg.0
    IL_006c:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0071:  stloc.1
    IL_0072:  ldarg.0
    IL_0073:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0078:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007e:  ldarg.0
    IL_007f:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0084:  box        "SM$T"
    IL_0089:  ldloca.s   V_1
    IL_008b:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0090:  ldloca.s   V_1
    IL_0092:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0098:  callvirt   "Sub IMoveable.GetName(Integer)"
    IL_009d:  ldarg.0
    IL_009e:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_00a3:  initobj    "SM$T"
    IL_00a9:  leave.s    IL_00cd
  }
  catch System.Exception
  {
    IL_00ab:  dup
    IL_00ac:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00b1:  stloc.2
    IL_00b2:  ldarg.0
    IL_00b3:  ldc.i4.s   -2
    IL_00b5:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00ba:  ldarg.0
    IL_00bb:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00c0:  ldloc.2
    IL_00c1:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00c6:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00cb:  leave.s    IL_00e2
  }
  IL_00cd:  ldarg.0
  IL_00ce:  ldc.i4.s   -2
  IL_00d0:  dup
  IL_00d1:  stloc.0
  IL_00d2:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00d7:  ldarg.0
  IL_00d8:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00dd:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_00e2:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      335 (0x14f)
  .maxstack  3
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                SM$T V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse    IL_00a1
    IL_000d:  ldarg.0
    IL_000e:  ldarg.0
    IL_000f:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T()"
    IL_0014:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T()"
    IL_0019:  ldarg.0
    IL_001a:  call       "Function Program.GetArrayIndex() As Integer"
    IL_001f:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As Integer"
    IL_0024:  ldarg.0
    IL_0025:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T()"
    IL_002a:  ldarg.0
    IL_002b:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As Integer"
    IL_0030:  readonly.
    IL_0032:  ldelema    "SM$T"
    IL_0037:  pop
    IL_0038:  ldloca.s   V_2
    IL_003a:  initobj    "SM$T"
    IL_0040:  ldloc.2
    IL_0041:  box        "SM$T"
    IL_0046:  brtrue.s   IL_005f
    IL_0048:  ldarg.0
    IL_0049:  ldarg.0
    IL_004a:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T()"
    IL_004f:  ldarg.0
    IL_0050:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As Integer"
    IL_0055:  ldelem     "SM$T"
    IL_005a:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U3 As SM$T"
    IL_005f:  ldarg.0
    IL_0060:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T()"
    IL_0065:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T()) As Integer"
    IL_006a:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_006f:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0074:  stloc.1
    IL_0075:  ldloca.s   V_1
    IL_0077:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_007c:  brtrue.s   IL_00bd
    IL_007e:  ldarg.0
    IL_007f:  ldc.i4.0
    IL_0080:  dup
    IL_0081:  stloc.0
    IL_0082:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0087:  ldarg.0
    IL_0088:  ldloc.1
    IL_0089:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008e:  ldarg.0
    IL_008f:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0094:  ldloca.s   V_1
    IL_0096:  ldarg.0
    IL_0097:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_009c:  leave      IL_014e
    IL_00a1:  ldarg.0
    IL_00a2:  ldc.i4.m1
    IL_00a3:  dup
    IL_00a4:  stloc.0
    IL_00a5:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00aa:  ldarg.0
    IL_00ab:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b0:  stloc.1
    IL_00b1:  ldarg.0
    IL_00b2:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b7:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00bd:  ldloca.s   V_2
    IL_00bf:  initobj    "SM$T"
    IL_00c5:  ldloc.2
    IL_00c6:  box        "SM$T"
    IL_00cb:  brtrue.s   IL_00d5
    IL_00cd:  ldarg.0
    IL_00ce:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U3 As SM$T"
    IL_00d3:  br.s       IL_00e8
    IL_00d5:  ldarg.0
    IL_00d6:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T()"
    IL_00db:  ldarg.0
    IL_00dc:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As Integer"
    IL_00e1:  readonly.
    IL_00e3:  ldelema    "SM$T"
    IL_00e8:  ldloca.s   V_1
    IL_00ea:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00ef:  ldloca.s   V_1
    IL_00f1:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00f7:  constrained. "SM$T"
    IL_00fd:  callvirt   "Sub IMoveable.GetName(Integer)"
    IL_0102:  ldarg.0
    IL_0103:  ldnull
    IL_0104:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T()"
    IL_0109:  ldarg.0
    IL_010a:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U3 As SM$T"
    IL_010f:  initobj    "SM$T"
    IL_0115:  leave.s    IL_0139
  }
  catch System.Exception
  {
    IL_0117:  dup
    IL_0118:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_011d:  stloc.3
    IL_011e:  ldarg.0
    IL_011f:  ldc.i4.s   -2
    IL_0121:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0126:  ldarg.0
    IL_0127:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_012c:  ldloc.3
    IL_012d:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0132:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0137:  leave.s    IL_014e
  }
  IL_0139:  ldarg.0
  IL_013a:  ldc.i4.s   -2
  IL_013c:  dup
  IL_013d:  stloc.0
  IL_013e:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0143:  ldarg.0
  IL_0144:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0149:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_014e:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Theory>
        <CombinatorialData>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Call_Struct_Async_01_ThroughArray(asRValue As Boolean)

            Dim leftParen As String = ""
            Dim rightParen As String = ""

            If asRValue Then
                leftParen = "("
                rightParen = ")"
            End If

            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Sub GetName(x As Integer)
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Sub GetName(x As Integer) Implements IMoveable.GetName
        Console.WriteLine("Position GetName for item '{0}'", Me.Name)
    End Sub
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = {New Item With {.Name = "1"}}
        Call1(item1).Wait()

        Dim item2 = {New Item With {.Name = "2"}}
        Call2(item2).Wait()

        Dim item3 = {New Item With {.Name = "3"}}
        Call3(item3).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T()) As Task
        call <%= leftParen %>item(GetArrayIndex())<%= rightParen %>.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T()) As Task
        call <%= leftParen %>item(GetArrayIndex())<%= rightParen %>.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call3(item As Item()) As Task
        call <%= leftParen %>item(GetArrayIndex())<%= rightParen %>.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T()) As Integer
        value -= 1
        item(0) = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Function GetArrayIndex() As Integer
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position GetName for item '-1'
Position GetName for item '-2'
Position GetName for item '-3'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      257 (0x101)
  .maxstack  3
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                System.Exception V_2)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0077
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0011:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T()"
    IL_0016:  ldarg.0
    IL_0017:  call       "Function Program.GetArrayIndex() As Integer"
    IL_001c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As Integer"
    IL_0021:  ldarg.0
    IL_0022:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T()"
    IL_0027:  ldarg.0
    IL_0028:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As Integer"
    IL_002d:  readonly.
    IL_002f:  ldelema    "SM$T"
    IL_0034:  pop
    IL_0035:  ldarg.0
    IL_0036:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_003b:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T()) As Integer"
    IL_0040:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0045:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_004a:  stloc.1
    IL_004b:  ldloca.s   V_1
    IL_004d:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0052:  brtrue.s   IL_0093
    IL_0054:  ldarg.0
    IL_0055:  ldc.i4.0
    IL_0056:  dup
    IL_0057:  stloc.0
    IL_0058:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_005d:  ldarg.0
    IL_005e:  ldloc.1
    IL_005f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0064:  ldarg.0
    IL_0065:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_006a:  ldloca.s   V_1
    IL_006c:  ldarg.0
    IL_006d:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0072:  leave      IL_0100
    IL_0077:  ldarg.0
    IL_0078:  ldc.i4.m1
    IL_0079:  dup
    IL_007a:  stloc.0
    IL_007b:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0080:  ldarg.0
    IL_0081:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0086:  stloc.1
    IL_0087:  ldarg.0
    IL_0088:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008d:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0093:  ldarg.0
    IL_0094:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T()"
    IL_0099:  ldarg.0
    IL_009a:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As Integer"
    IL_009f:  readonly.
    IL_00a1:  ldelema    "SM$T"
    IL_00a6:  ldloca.s   V_1
    IL_00a8:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00ad:  ldloca.s   V_1
    IL_00af:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b5:  constrained. "SM$T"
    IL_00bb:  callvirt   "Sub IMoveable.GetName(Integer)"
    IL_00c0:  ldarg.0
    IL_00c1:  ldnull
    IL_00c2:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T()"
    IL_00c7:  leave.s    IL_00eb
  }
  catch System.Exception
  {
    IL_00c9:  dup
    IL_00ca:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00cf:  stloc.2
    IL_00d0:  ldarg.0
    IL_00d1:  ldc.i4.s   -2
    IL_00d3:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00d8:  ldarg.0
    IL_00d9:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00de:  ldloc.2
    IL_00df:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00e4:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00e9:  leave.s    IL_0100
  }
  IL_00eb:  ldarg.0
  IL_00ec:  ldc.i4.s   -2
  IL_00ee:  dup
  IL_00ef:  stloc.0
  IL_00f0:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00f5:  ldarg.0
  IL_00f6:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00fb:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0100:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Theory>
        <CombinatorialData>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Call_Class_Async_01_ThroughArray_InWith(asRValue As Boolean)

            Dim leftParen As String = ""
            Dim rightParen As String = ""

            If asRValue Then
                leftParen = "("
                rightParen = ")"
            End If

            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Sub GetName(x As Integer)
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Sub GetName(x As Integer) Implements IMoveable.GetName
        Console.WriteLine("Position GetName for item '{0}'", Me.Name)
    End Sub
End Class

Class Item2
    Inherits Item
End Class

Class Program
    Shared Sub Main()
        Dim item1 = {New Item2 With {.Name = "1"}}
        Call1(DirectCast(item1, Item())).Wait()

        Dim item2 = {New Item2 With {.Name = "2"}}
        Call2(DirectCast(item2, Item())).Wait()

        Dim item3 = {New Item2 With {.Name = "3"}}
        Call3(DirectCast(item3, Item())).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T()) As Task
        With <%= leftParen %>item(GetArrayIndex())<%= rightParen %>
            call .GetName(await GetOffsetAsync(GetOffset(item)))
        End With
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T()) As Task
        With <%= leftParen %>item(GetArrayIndex())<%= rightParen %>
            call .GetName(await GetOffsetAsync(GetOffset(item)))
        End With
    End Function

    Private Shared Async Function Call3(item As Item()) As Task
        With <%= leftParen %>item(GetArrayIndex())<%= rightParen %>
            call .GetName(await GetOffsetAsync(GetOffset(item)))
        End With
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T()) As Integer
        value -= 1
        item(0) = DirectCast(DirectCast(New Item2 With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Function GetArrayIndex() As Integer
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position GetName for item '1'
Position GetName for item '2'
Position GetName for item '3'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
If(asRValue,
            <![CDATA[
{
      // Code size      251 (0xfb)
  .maxstack  3
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                System.Exception V_2)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_006e
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0011:  call       "Function Program.GetArrayIndex() As Integer"
    IL_0016:  ldelem     "SM$T"
    IL_001b:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$W0 As SM$T"
    IL_0020:  ldarg.0
    IL_0021:  ldarg.0
    IL_0022:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$W0 As SM$T"
    IL_0027:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_002c:  ldarg.0
    IL_002d:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0032:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T()) As Integer"
    IL_0037:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_003c:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0041:  stloc.1
    IL_0042:  ldloca.s   V_1
    IL_0044:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0049:  brtrue.s   IL_008a
    IL_004b:  ldarg.0
    IL_004c:  ldc.i4.0
    IL_004d:  dup
    IL_004e:  stloc.0
    IL_004f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0054:  ldarg.0
    IL_0055:  ldloc.1
    IL_0056:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_005b:  ldarg.0
    IL_005c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0061:  ldloca.s   V_1
    IL_0063:  ldarg.0
    IL_0064:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0069:  leave      IL_00fa
    IL_006e:  ldarg.0
    IL_006f:  ldc.i4.m1
    IL_0070:  dup
    IL_0071:  stloc.0
    IL_0072:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0077:  ldarg.0
    IL_0078:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007d:  stloc.1
    IL_007e:  ldarg.0
    IL_007f:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0084:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008a:  ldarg.0
    IL_008b:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0090:  box        "SM$T"
    IL_0095:  ldloca.s   V_1
    IL_0097:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_009c:  ldloca.s   V_1
    IL_009e:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a4:  callvirt   "Sub IMoveable.GetName(Integer)"
    IL_00a9:  ldarg.0
    IL_00aa:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_00af:  initobj    "SM$T"
    IL_00b5:  ldarg.0
    IL_00b6:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$W0 As SM$T"
    IL_00bb:  initobj    "SM$T"
    IL_00c1:  leave.s    IL_00e5
  }
  catch System.Exception
  {
    IL_00c3:  dup
    IL_00c4:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00c9:  stloc.2
    IL_00ca:  ldarg.0
    IL_00cb:  ldc.i4.s   -2
    IL_00cd:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00d2:  ldarg.0
    IL_00d3:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00d8:  ldloc.2
    IL_00d9:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00de:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00e3:  leave.s    IL_00fa
  }
  IL_00e5:  ldarg.0
  IL_00e6:  ldc.i4.s   -2
  IL_00e8:  dup
  IL_00e9:  stloc.0
  IL_00ea:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00ef:  ldarg.0
  IL_00f0:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00f5:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_00fa:  ret
}
]]>,
            <![CDATA[
{
      // Code size      237 (0xed)
  .maxstack  3
  .locals init (Integer V_0,
                SM$T V_1, //$W0
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0064
    IL_000a:  ldarg.0
    IL_000b:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0010:  call       "Function Program.GetArrayIndex() As Integer"
    IL_0015:  ldelem     "SM$T"
    IL_001a:  stloc.1
    IL_001b:  ldarg.0
    IL_001c:  ldloc.1
    IL_001d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0022:  ldarg.0
    IL_0023:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0028:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T()) As Integer"
    IL_002d:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0032:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0037:  stloc.2
    IL_0038:  ldloca.s   V_2
    IL_003a:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_003f:  brtrue.s   IL_0080
    IL_0041:  ldarg.0
    IL_0042:  ldc.i4.0
    IL_0043:  dup
    IL_0044:  stloc.0
    IL_0045:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_004a:  ldarg.0
    IL_004b:  ldloc.2
    IL_004c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0051:  ldarg.0
    IL_0052:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0057:  ldloca.s   V_2
    IL_0059:  ldarg.0
    IL_005a:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_005f:  leave      IL_00ec
    IL_0064:  ldarg.0
    IL_0065:  ldc.i4.m1
    IL_0066:  dup
    IL_0067:  stloc.0
    IL_0068:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_006d:  ldarg.0
    IL_006e:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0073:  stloc.2
    IL_0074:  ldarg.0
    IL_0075:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007a:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0080:  ldarg.0
    IL_0081:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0086:  box        "SM$T"
    IL_008b:  ldloca.s   V_2
    IL_008d:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0092:  ldloca.s   V_2
    IL_0094:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009a:  callvirt   "Sub IMoveable.GetName(Integer)"
    IL_009f:  ldarg.0
    IL_00a0:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_00a5:  initobj    "SM$T"
    IL_00ab:  ldloca.s   V_1
    IL_00ad:  initobj    "SM$T"
    IL_00b3:  leave.s    IL_00d7
  }
  catch System.Exception
  {
    IL_00b5:  dup
    IL_00b6:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00bb:  stloc.3
    IL_00bc:  ldarg.0
    IL_00bd:  ldc.i4.s   -2
    IL_00bf:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00c4:  ldarg.0
    IL_00c5:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00ca:  ldloc.3
    IL_00cb:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00d0:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00d5:  leave.s    IL_00ec
  }
  IL_00d7:  ldarg.0
  IL_00d8:  ldc.i4.s   -2
  IL_00da:  dup
  IL_00db:  stloc.0
  IL_00dc:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00e1:  ldarg.0
  IL_00e2:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00e7:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_00ec:  ret
}
]]>))

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
If(asRValue,
            <![CDATA[
{
      // Code size      292 (0x124)
  .maxstack  3
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                SM$T V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_007e
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T()"
    IL_0011:  call       "Function Program.GetArrayIndex() As Integer"
    IL_0016:  ldelem     "SM$T"
    IL_001b:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$W0 As SM$T"
    IL_0020:  ldloca.s   V_2
    IL_0022:  initobj    "SM$T"
    IL_0028:  ldloc.2
    IL_0029:  box        "SM$T"
    IL_002e:  brtrue.s   IL_003c
    IL_0030:  ldarg.0
    IL_0031:  ldarg.0
    IL_0032:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$W0 As SM$T"
    IL_0037:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T"
    IL_003c:  ldarg.0
    IL_003d:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T()"
    IL_0042:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T()) As Integer"
    IL_0047:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_004c:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0051:  stloc.1
    IL_0052:  ldloca.s   V_1
    IL_0054:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0059:  brtrue.s   IL_009a
    IL_005b:  ldarg.0
    IL_005c:  ldc.i4.0
    IL_005d:  dup
    IL_005e:  stloc.0
    IL_005f:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0064:  ldarg.0
    IL_0065:  ldloc.1
    IL_0066:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_006b:  ldarg.0
    IL_006c:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0071:  ldloca.s   V_1
    IL_0073:  ldarg.0
    IL_0074:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_0079:  leave      IL_0123
    IL_007e:  ldarg.0
    IL_007f:  ldc.i4.m1
    IL_0080:  dup
    IL_0081:  stloc.0
    IL_0082:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0087:  ldarg.0
    IL_0088:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008d:  stloc.1
    IL_008e:  ldarg.0
    IL_008f:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0094:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009a:  ldloca.s   V_2
    IL_009c:  initobj    "SM$T"
    IL_00a2:  ldloc.2
    IL_00a3:  box        "SM$T"
    IL_00a8:  brtrue.s   IL_00b2
    IL_00aa:  ldarg.0
    IL_00ab:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T"
    IL_00b0:  br.s       IL_00b8
    IL_00b2:  ldarg.0
    IL_00b3:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$W0 As SM$T"
    IL_00b8:  ldloca.s   V_1
    IL_00ba:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00bf:  ldloca.s   V_1
    IL_00c1:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00c7:  constrained. "SM$T"
    IL_00cd:  callvirt   "Sub IMoveable.GetName(Integer)"
    IL_00d2:  ldarg.0
    IL_00d3:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T"
    IL_00d8:  initobj    "SM$T"
    IL_00de:  ldarg.0
    IL_00df:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$W0 As SM$T"
    IL_00e4:  initobj    "SM$T"
    IL_00ea:  leave.s    IL_010e
  }
  catch System.Exception
  {
    IL_00ec:  dup
    IL_00ed:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00f2:  stloc.3
    IL_00f3:  ldarg.0
    IL_00f4:  ldc.i4.s   -2
    IL_00f6:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00fb:  ldarg.0
    IL_00fc:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0101:  ldloc.3
    IL_0102:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0107:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_010c:  leave.s    IL_0123
  }
  IL_010e:  ldarg.0
  IL_010f:  ldc.i4.s   -2
  IL_0111:  dup
  IL_0112:  stloc.0
  IL_0113:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0118:  ldarg.0
  IL_0119:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_011e:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0123:  ret
}
]]>,
            <![CDATA[
{
      // Code size      345 (0x159)
  .maxstack  3
  .locals init (Integer V_0,
                SM$T() V_1, //$W0
                Integer V_2, //$W1
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_3,
                SM$T V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse    IL_00a6
    IL_000d:  ldarg.0
    IL_000e:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T()"
    IL_0013:  stloc.1
    IL_0014:  call       "Function Program.GetArrayIndex() As Integer"
    IL_0019:  stloc.2
    IL_001a:  ldarg.0
    IL_001b:  ldloc.1
    IL_001c:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T()"
    IL_0021:  ldarg.0
    IL_0022:  ldloc.2
    IL_0023:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As Integer"
    IL_0028:  ldarg.0
    IL_0029:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T()"
    IL_002e:  ldarg.0
    IL_002f:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As Integer"
    IL_0034:  readonly.
    IL_0036:  ldelema    "SM$T"
    IL_003b:  pop
    IL_003c:  ldloca.s   V_4
    IL_003e:  initobj    "SM$T"
    IL_0044:  ldloc.s    V_4
    IL_0046:  box        "SM$T"
    IL_004b:  brtrue.s   IL_0064
    IL_004d:  ldarg.0
    IL_004e:  ldarg.0
    IL_004f:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T()"
    IL_0054:  ldarg.0
    IL_0055:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As Integer"
    IL_005a:  ldelem     "SM$T"
    IL_005f:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U3 As SM$T"
    IL_0064:  ldarg.0
    IL_0065:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T()"
    IL_006a:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T()) As Integer"
    IL_006f:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0074:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0079:  stloc.3
    IL_007a:  ldloca.s   V_3
    IL_007c:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0081:  brtrue.s   IL_00c2
    IL_0083:  ldarg.0
    IL_0084:  ldc.i4.0
    IL_0085:  dup
    IL_0086:  stloc.0
    IL_0087:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_008c:  ldarg.0
    IL_008d:  ldloc.3
    IL_008e:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0093:  ldarg.0
    IL_0094:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0099:  ldloca.s   V_3
    IL_009b:  ldarg.0
    IL_009c:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_00a1:  leave      IL_0158
    IL_00a6:  ldarg.0
    IL_00a7:  ldc.i4.m1
    IL_00a8:  dup
    IL_00a9:  stloc.0
    IL_00aa:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00af:  ldarg.0
    IL_00b0:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b5:  stloc.3
    IL_00b6:  ldarg.0
    IL_00b7:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00bc:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00c2:  ldloca.s   V_4
    IL_00c4:  initobj    "SM$T"
    IL_00ca:  ldloc.s    V_4
    IL_00cc:  box        "SM$T"
    IL_00d1:  brtrue.s   IL_00db
    IL_00d3:  ldarg.0
    IL_00d4:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U3 As SM$T"
    IL_00d9:  br.s       IL_00ee
    IL_00db:  ldarg.0
    IL_00dc:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T()"
    IL_00e1:  ldarg.0
    IL_00e2:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As Integer"
    IL_00e7:  readonly.
    IL_00e9:  ldelema    "SM$T"
    IL_00ee:  ldloca.s   V_3
    IL_00f0:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00f5:  ldloca.s   V_3
    IL_00f7:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00fd:  constrained. "SM$T"
    IL_0103:  callvirt   "Sub IMoveable.GetName(Integer)"
    IL_0108:  ldarg.0
    IL_0109:  ldnull
    IL_010a:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T()"
    IL_010f:  ldarg.0
    IL_0110:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U3 As SM$T"
    IL_0115:  initobj    "SM$T"
    IL_011b:  ldnull
    IL_011c:  stloc.1
    IL_011d:  leave.s    IL_0143
  }
  catch System.Exception
  {
    IL_011f:  dup
    IL_0120:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0125:  stloc.s    V_5
    IL_0127:  ldarg.0
    IL_0128:  ldc.i4.s   -2
    IL_012a:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_012f:  ldarg.0
    IL_0130:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0135:  ldloc.s    V_5
    IL_0137:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_013c:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0141:  leave.s    IL_0158
  }
  IL_0143:  ldarg.0
  IL_0144:  ldc.i4.s   -2
  IL_0146:  dup
  IL_0147:  stloc.0
  IL_0148:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_014d:  ldarg.0
  IL_014e:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0153:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0158:  ret
}
]]>))
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Theory>
        <CombinatorialData>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Call_Struct_Async_01_ThroughArray_InWith(asRValue As Boolean)

            Dim leftParen As String = ""
            Dim rightParen As String = ""

            If asRValue Then
                leftParen = "("
                rightParen = ")"
            End If

            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Sub GetName(x As Integer)
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Sub GetName(x As Integer) Implements IMoveable.GetName
        Console.WriteLine("Position GetName for item '{0}'", Me.Name)
    End Sub
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = {New Item With {.Name = "1"}}
        Call1(item1).Wait()

        Dim item2 = {New Item With {.Name = "2"}}
        Call2(item2).Wait()

        Dim item3 = {New Item With {.Name = "3"}}
        Call3(item3).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T()) As Task
        With <%= leftParen %>item(GetArrayIndex())<%= rightParen %>
            call .GetName(await GetOffsetAsync(GetOffset(item)))
        End With
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T()) As Task
        With <%= leftParen %>item(GetArrayIndex())<%= rightParen %>
            call .GetName(await GetOffsetAsync(GetOffset(item)))
        End With
    End Function

    Private Shared Async Function Call3(item As Item()) As Task
        With <%= leftParen %>item(GetArrayIndex())<%= rightParen %>
            call .GetName(await GetOffsetAsync(GetOffset(item)))
        End With
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T()) As Integer
        value -= 1
        item(0) = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Function GetArrayIndex() As Integer
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
If(asRValue,
"
Position GetName for item '1'
Position GetName for item '2'
Position GetName for item '3'
",
"
Position GetName for item '-1'
Position GetName for item '-2'
Position GetName for item '-3'
")

            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
If(asRValue,
            <![CDATA[
{
  // Code size      228 (0xe4)
  .maxstack  3
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                System.Exception V_2)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0062
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0011:  call       "Function Program.GetArrayIndex() As Integer"
    IL_0016:  ldelem     "SM$T"
    IL_001b:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$W0 As SM$T"
    IL_0020:  ldarg.0
    IL_0021:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0026:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T()) As Integer"
    IL_002b:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0030:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0035:  stloc.1
    IL_0036:  ldloca.s   V_1
    IL_0038:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_003d:  brtrue.s   IL_007e
    IL_003f:  ldarg.0
    IL_0040:  ldc.i4.0
    IL_0041:  dup
    IL_0042:  stloc.0
    IL_0043:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0048:  ldarg.0
    IL_0049:  ldloc.1
    IL_004a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_004f:  ldarg.0
    IL_0050:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0055:  ldloca.s   V_1
    IL_0057:  ldarg.0
    IL_0058:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_005d:  leave      IL_00e3
    IL_0062:  ldarg.0
    IL_0063:  ldc.i4.m1
    IL_0064:  dup
    IL_0065:  stloc.0
    IL_0066:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_006b:  ldarg.0
    IL_006c:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0071:  stloc.1
    IL_0072:  ldarg.0
    IL_0073:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0078:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007e:  ldarg.0
    IL_007f:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$W0 As SM$T"
    IL_0084:  ldloca.s   V_1
    IL_0086:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_008b:  ldloca.s   V_1
    IL_008d:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0093:  constrained. "SM$T"
    IL_0099:  callvirt   "Sub IMoveable.GetName(Integer)"
    IL_009e:  ldarg.0
    IL_009f:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$W0 As SM$T"
    IL_00a4:  initobj    "SM$T"
    IL_00aa:  leave.s    IL_00ce
  }
  catch System.Exception
  {
    IL_00ac:  dup
    IL_00ad:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00b2:  stloc.2
    IL_00b3:  ldarg.0
    IL_00b4:  ldc.i4.s   -2
    IL_00b6:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00bb:  ldarg.0
    IL_00bc:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00c1:  ldloc.2
    IL_00c2:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00c7:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00cc:  leave.s    IL_00e3
  }
  IL_00ce:  ldarg.0
  IL_00cf:  ldc.i4.s   -2
  IL_00d1:  dup
  IL_00d2:  stloc.0
  IL_00d3:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00d8:  ldarg.0
  IL_00d9:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00de:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_00e3:  ret
}
]]>,
            <![CDATA[
{
  // Code size      265 (0x109)
  .maxstack  3
  .locals init (Integer V_0,
                SM$T() V_1, //$W0
                Integer V_2, //$W1
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_007b
    IL_000a:  ldarg.0
    IL_000b:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0010:  stloc.1
    IL_0011:  call       "Function Program.GetArrayIndex() As Integer"
    IL_0016:  stloc.2
    IL_0017:  ldarg.0
    IL_0018:  ldloc.1
    IL_0019:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T()"
    IL_001e:  ldarg.0
    IL_001f:  ldloc.2
    IL_0020:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As Integer"
    IL_0025:  ldarg.0
    IL_0026:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T()"
    IL_002b:  ldarg.0
    IL_002c:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As Integer"
    IL_0031:  readonly.
    IL_0033:  ldelema    "SM$T"
    IL_0038:  pop
    IL_0039:  ldarg.0
    IL_003a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_003f:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T()) As Integer"
    IL_0044:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0049:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_004e:  stloc.3
    IL_004f:  ldloca.s   V_3
    IL_0051:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0056:  brtrue.s   IL_0097
    IL_0058:  ldarg.0
    IL_0059:  ldc.i4.0
    IL_005a:  dup
    IL_005b:  stloc.0
    IL_005c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0061:  ldarg.0
    IL_0062:  ldloc.3
    IL_0063:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0068:  ldarg.0
    IL_0069:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_006e:  ldloca.s   V_3
    IL_0070:  ldarg.0
    IL_0071:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0076:  leave      IL_0108
    IL_007b:  ldarg.0
    IL_007c:  ldc.i4.m1
    IL_007d:  dup
    IL_007e:  stloc.0
    IL_007f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0084:  ldarg.0
    IL_0085:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008a:  stloc.3
    IL_008b:  ldarg.0
    IL_008c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0091:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0097:  ldarg.0
    IL_0098:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T()"
    IL_009d:  ldarg.0
    IL_009e:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As Integer"
    IL_00a3:  readonly.
    IL_00a5:  ldelema    "SM$T"
    IL_00aa:  ldloca.s   V_3
    IL_00ac:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00b1:  ldloca.s   V_3
    IL_00b3:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b9:  constrained. "SM$T"
    IL_00bf:  callvirt   "Sub IMoveable.GetName(Integer)"
    IL_00c4:  ldarg.0
    IL_00c5:  ldnull
    IL_00c6:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T()"
    IL_00cb:  ldnull
    IL_00cc:  stloc.1
    IL_00cd:  leave.s    IL_00f3
  }
  catch System.Exception
  {
    IL_00cf:  dup
    IL_00d0:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00d5:  stloc.s    V_4
    IL_00d7:  ldarg.0
    IL_00d8:  ldc.i4.s   -2
    IL_00da:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00df:  ldarg.0
    IL_00e0:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00e5:  ldloc.s    V_4
    IL_00e7:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00ec:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00f1:  leave.s    IL_0108
  }
  IL_00f3:  ldarg.0
  IL_00f4:  ldc.i4.s   -2
  IL_00f6:  dup
  IL_00f7:  stloc.0
  IL_00f8:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00fd:  ldarg.0
  IL_00fe:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0103:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0108:  ret
}
]]>))
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Theory>
        <CombinatorialData>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Call_Conditional_Class(asRValue As Boolean)

            Dim leftParen As String = ""
            Dim rightParen As String = ""

            If asRValue Then
                leftParen = "("
                rightParen = ")"
            End If

            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Sub GetName(x As Integer)
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Sub GetName(x As Integer) Implements IMoveable.GetName
        Console.WriteLine("Position GetName for item '{0}'", Me.Name)
    End Sub
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)

        Dim item3 = New Item With {.Name = "3"}
        Call3(item3)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(item As T)
        call <%= leftParen %>item<%= rightParen %>?.GetName(GetOffset(item))
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        call <%= leftParen %>item<%= rightParen %>?.GetName(GetOffset(item))
    End Sub

    Private Shared Sub Call3(item As Item)
        call <%= leftParen %>item<%= rightParen %>?.GetName(GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position GetName for item '1'
Position GetName for item '2'
Position GetName for item '3'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       24 (0x18)
  .maxstack  2
  IL_0000:  ldarg.0
  IL_0001:  box        "T"
  IL_0006:  dup
  IL_0007:  brtrue.s   IL_000b
  IL_0009:  pop
  IL_000a:  ret
  IL_000b:  ldarga.s   V_0
  IL_000d:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0012:  callvirt   "Sub IMoveable.GetName(Integer)"
  IL_0017:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
  // Code size       53 (0x35)
  .maxstack  2
  .locals init (T V_0)
  IL_0000:  ldarg.0
  IL_0001:  box        "T"
  IL_0006:  brfalse.s  IL_0034
  IL_0008:  ldarga.s   V_0
  IL_000a:  ldloca.s   V_0
  IL_000c:  initobj    "T"
  IL_0012:  ldloc.0
  IL_0013:  box        "T"
  IL_0018:  brtrue.s   IL_0022
  IL_001a:  ldobj      "T"
  IL_001f:  stloc.0
  IL_0020:  ldloca.s   V_0
  IL_0022:  ldarga.s   V_0
  IL_0024:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0029:  constrained. "T"
  IL_002f:  callvirt   "Sub IMoveable.GetName(Integer)"
  IL_0034:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Theory>
        <CombinatorialData>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Call_Conditional_Struct(asRValue As Boolean)

            Dim leftParen As String = ""
            Dim rightParen As String = ""

            If asRValue Then
                leftParen = "("
                rightParen = ")"
            End If

            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Sub GetName(x As Integer)
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Sub GetName(x As Integer) Implements IMoveable.GetName
        Console.WriteLine("Position GetName for item '{0}'", Me.Name)
    End Sub
End Structure

Class Program
    Shared Sub Main()
        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)

        Dim item3 = New Item With {.Name = "3"}
        Call3(item3)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        call <%= leftParen %>item<%= rightParen %>?.GetName(GetOffset(item))
    End Sub

    Private Shared Sub Call3(item As Item)
        call <%= leftParen %>item<%= rightParen %>.GetName(GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position GetName for item '-1'
Position GetName for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Theory>
        <CombinatorialData>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Call_Conditional_Class_Ref(asRValue As Boolean)

            Dim leftParen As String = ""
            Dim rightParen As String = ""

            If asRValue Then
                leftParen = "("
                rightParen = ")"
            End If

            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Sub GetName(x As Integer)
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Sub GetName(x As Integer) Implements IMoveable.GetName
        Console.WriteLine("Position GetName for item '{0}'", Me.Name)
    End Sub
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)

        Dim item3 = New Item With {.Name = "3"}
        Call3(item3)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(ByRef item As T)
        call <%= leftParen %>item<%= rightParen %>?.GetName(GetOffset(item))
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        call <%= leftParen %>item<%= rightParen %>?.GetName(GetOffset(item))
    End Sub

    Private Shared Sub Call3(ByRef item As Item)
        call <%= leftParen %>item<%= rightParen %>?.GetName(GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position GetName for item '1'
Position GetName for item '2'
Position GetName for item '3'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       28 (0x1c)
  .maxstack  2
  IL_0000:  ldarg.0
  IL_0001:  ldobj      "T"
  IL_0006:  box        "T"
  IL_000b:  dup
  IL_000c:  brtrue.s   IL_0010
  IL_000e:  pop
  IL_000f:  ret
  IL_0010:  ldarg.0
  IL_0011:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0016:  callvirt   "Sub IMoveable.GetName(Integer)"
  IL_001b:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
  // Code size       53 (0x35)
  .maxstack  2
  .locals init (T V_0)
  IL_0000:  ldarg.0
  IL_0001:  ldloca.s   V_0
  IL_0003:  initobj    "T"
  IL_0009:  ldloc.0
  IL_000a:  box        "T"
  IL_000f:  brtrue.s   IL_0023
  IL_0011:  ldobj      "T"
  IL_0016:  stloc.0
  IL_0017:  ldloca.s   V_0
  IL_0019:  ldloc.0
  IL_001a:  box        "T"
  IL_001f:  brtrue.s   IL_0023
  IL_0021:  pop
  IL_0022:  ret
  IL_0023:  ldarg.0
  IL_0024:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0029:  constrained. "T"
  IL_002f:  callvirt   "Sub IMoveable.GetName(Integer)"
  IL_0034:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Theory>
        <CombinatorialData>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Call_Conditional_Struct_Ref(asRValue As Boolean)

            Dim leftParen As String = ""
            Dim rightParen As String = ""

            If asRValue Then
                leftParen = "("
                rightParen = ")"
            End If

            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Sub GetName(x As Integer)
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Sub GetName(x As Integer) Implements IMoveable.GetName
        Console.WriteLine("Position GetName for item '{0}'", Me.Name)
    End Sub
End Structure

Class Program
    Shared Sub Main()
        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)

        Dim item3 = New Item With {.Name = "3"}
        Call3(item3)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        call <%= leftParen %>item<%= rightParen %>?.GetName(GetOffset(item))
    End Sub

    Private Shared Sub Call3(ByRef item As Item)
        call <%= leftParen %>item<%= rightParen %>.GetName(GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position GetName for item '-1'
Position GetName for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Theory>
        <CombinatorialData>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Call_Conditional_Class_Async_01(asRValue As Boolean)

            Dim leftParen As String = ""
            Dim rightParen As String = ""

            If asRValue Then
                leftParen = "("
                rightParen = ")"
            End If

            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Sub GetName(x As Integer)
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Sub GetName(x As Integer) Implements IMoveable.GetName
        Console.WriteLine("Position GetName for item '{0}'", Me.Name)
    End Sub
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()

        Dim item3 = New Item With {.Name = "3"}
        Call3(item3).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        call <%= leftParen %>item<%= rightParen %>?.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        call <%= leftParen %>item<%= rightParen %>?.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call3(item As Item) As Task
        call <%= leftParen %>item<%= rightParen %>?.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position GetName for item '1'
Position GetName for item '2'
Position GetName for item '3'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput, verify:=Verification.Skipped).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      234 (0xea)
  .maxstack  3
  .locals init (Integer V_0,
                SM$T V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0069
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0011:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0016:  ldarg.0
    IL_0017:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_001c:  ldloca.s   V_1
    IL_001e:  initobj    "SM$T"
    IL_0024:  ldloc.1
    IL_0025:  beq.s      IL_00a4
    IL_0027:  ldarg.0
    IL_0028:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_002d:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0032:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0037:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_003c:  stloc.2
    IL_003d:  ldloca.s   V_2
    IL_003f:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0044:  brtrue.s   IL_0085
    IL_0046:  ldarg.0
    IL_0047:  ldc.i4.0
    IL_0048:  dup
    IL_0049:  stloc.0
    IL_004a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_004f:  ldarg.0
    IL_0050:  ldloc.2
    IL_0051:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0056:  ldarg.0
    IL_0057:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_005c:  ldloca.s   V_2
    IL_005e:  ldarg.0
    IL_005f:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0064:  leave      IL_00e9
    IL_0069:  ldarg.0
    IL_006a:  ldc.i4.m1
    IL_006b:  dup
    IL_006c:  stloc.0
    IL_006d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0072:  ldarg.0
    IL_0073:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0078:  stloc.2
    IL_0079:  ldarg.0
    IL_007a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007f:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0085:  ldarg.0
    IL_0086:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_008b:  box        "SM$T"
    IL_0090:  ldloca.s   V_2
    IL_0092:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0097:  ldloca.s   V_2
    IL_0099:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009f:  callvirt   "Sub IMoveable.GetName(Integer)"
    IL_00a4:  ldarg.0
    IL_00a5:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_00aa:  initobj    "SM$T"
    IL_00b0:  leave.s    IL_00d4
  }
  catch System.Exception
  {
    IL_00b2:  dup
    IL_00b3:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00b8:  stloc.3
    IL_00b9:  ldarg.0
    IL_00ba:  ldc.i4.s   -2
    IL_00bc:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00c1:  ldarg.0
    IL_00c2:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00c7:  ldloc.3
    IL_00c8:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00cd:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00d2:  leave.s    IL_00e9
  }
  IL_00d4:  ldarg.0
  IL_00d5:  ldc.i4.s   -2
  IL_00d7:  dup
  IL_00d8:  stloc.0
  IL_00d9:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00de:  ldarg.0
  IL_00df:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00e4:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_00e9:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      317 (0x13d)
  .maxstack  3
  .locals init (Integer V_0,
                SM$T V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse    IL_00a3
    IL_000d:  ldloca.s   V_1
    IL_000f:  initobj    "SM$T"
    IL_0015:  ldloc.1
    IL_0016:  box        "SM$T"
    IL_001b:  brtrue.s   IL_0029
    IL_001d:  ldarg.0
    IL_001e:  ldarg.0
    IL_001f:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0024:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T"
    IL_0029:  ldloca.s   V_1
    IL_002b:  initobj    "SM$T"
    IL_0031:  ldloc.1
    IL_0032:  box        "SM$T"
    IL_0037:  brtrue.s   IL_0061
    IL_0039:  ldloca.s   V_1
    IL_003b:  initobj    "SM$T"
    IL_0041:  ldloc.1
    IL_0042:  box        "SM$T"
    IL_0047:  brtrue.s   IL_0051
    IL_0049:  ldarg.0
    IL_004a:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T"
    IL_004f:  br.s       IL_0057
    IL_0051:  ldarg.0
    IL_0052:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0057:  box        "SM$T"
    IL_005c:  brfalse    IL_00f7
    IL_0061:  ldarg.0
    IL_0062:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0067:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_006c:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0071:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0076:  stloc.2
    IL_0077:  ldloca.s   V_2
    IL_0079:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_007e:  brtrue.s   IL_00bf
    IL_0080:  ldarg.0
    IL_0081:  ldc.i4.0
    IL_0082:  dup
    IL_0083:  stloc.0
    IL_0084:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0089:  ldarg.0
    IL_008a:  ldloc.2
    IL_008b:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0090:  ldarg.0
    IL_0091:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0096:  ldloca.s   V_2
    IL_0098:  ldarg.0
    IL_0099:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_009e:  leave      IL_013c
    IL_00a3:  ldarg.0
    IL_00a4:  ldc.i4.m1
    IL_00a5:  dup
    IL_00a6:  stloc.0
    IL_00a7:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00ac:  ldarg.0
    IL_00ad:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b2:  stloc.2
    IL_00b3:  ldarg.0
    IL_00b4:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b9:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00bf:  ldloca.s   V_1
    IL_00c1:  initobj    "SM$T"
    IL_00c7:  ldloc.1
    IL_00c8:  box        "SM$T"
    IL_00cd:  brtrue.s   IL_00d7
    IL_00cf:  ldarg.0
    IL_00d0:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T"
    IL_00d5:  br.s       IL_00dd
    IL_00d7:  ldarg.0
    IL_00d8:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00dd:  ldloca.s   V_2
    IL_00df:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00e4:  ldloca.s   V_2
    IL_00e6:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00ec:  constrained. "SM$T"
    IL_00f2:  callvirt   "Sub IMoveable.GetName(Integer)"
    IL_00f7:  ldarg.0
    IL_00f8:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T"
    IL_00fd:  initobj    "SM$T"
    IL_0103:  leave.s    IL_0127
  }
  catch System.Exception
  {
    IL_0105:  dup
    IL_0106:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_010b:  stloc.3
    IL_010c:  ldarg.0
    IL_010d:  ldc.i4.s   -2
    IL_010f:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0114:  ldarg.0
    IL_0115:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_011a:  ldloc.3
    IL_011b:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0120:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0125:  leave.s    IL_013c
  }
  IL_0127:  ldarg.0
  IL_0128:  ldc.i4.s   -2
  IL_012a:  dup
  IL_012b:  stloc.0
  IL_012c:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0131:  ldarg.0
  IL_0132:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0137:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_013c:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput, verify:=Verification.Skipped).VerifyDiagnostics()
        End Sub

        <Theory>
        <CombinatorialData>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Call_Conditional_Struct_Async_01(asRValue As Boolean)

            Dim leftParen As String = ""
            Dim rightParen As String = ""

            If asRValue Then
                leftParen = "("
                rightParen = ")"
            End If

            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Sub GetName(x As Integer)
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Sub GetName(x As Integer) Implements IMoveable.GetName
        Console.WriteLine("Position GetName for item '{0}'", Me.Name)
    End Sub
End Structure

Class Program
    Shared Sub Main()
        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()

        Dim item3 = New Item With {.Name = "3"}
        Call3(item3).Wait()
    End Sub

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        call <%= leftParen %>item<%= rightParen %>?.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call3(item As Item) As Task
        call <%= leftParen %>item<%= rightParen %>.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position GetName for item '-1'
Position GetName for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Call_Conditional_Class_Async_02()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Sub GetName(x As Integer)
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Sub GetName(x As Integer) Implements IMoveable.GetName
        Console.WriteLine("Position GetName for item '{0}'", Me.Name)
    End Sub
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        await Task.Yield()
        item?.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        await Task.Yield()
        item?.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position GetName for item '1'
Position GetName for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput, verify:=Verification.Skipped).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      350 (0x15e)
  .maxstack  3
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter V_1,
                System.Runtime.CompilerServices.YieldAwaitable V_2,
                SM$T V_3,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004b
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_00da
    IL_0011:  call       "Function System.Threading.Tasks.Task.Yield() As System.Runtime.CompilerServices.YieldAwaitable"
    IL_0016:  stloc.2
    IL_0017:  ldloca.s   V_2
    IL_0019:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.GetAwaiter() As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_001e:  stloc.1
    IL_001f:  ldloca.s   V_1
    IL_0021:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.get_IsCompleted() As Boolean"
    IL_0026:  brtrue.s   IL_0067
    IL_0028:  ldarg.0
    IL_0029:  ldc.i4.0
    IL_002a:  dup
    IL_002b:  stloc.0
    IL_002c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0031:  ldarg.0
    IL_0032:  ldloc.1
    IL_0033:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0038:  ldarg.0
    IL_0039:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003e:  ldloca.s   V_1
    IL_0040:  ldarg.0
    IL_0041:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0046:  leave      IL_015d
    IL_004b:  ldarg.0
    IL_004c:  ldc.i4.m1
    IL_004d:  dup
    IL_004e:  stloc.0
    IL_004f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0054:  ldarg.0
    IL_0055:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_005a:  stloc.1
    IL_005b:  ldarg.0
    IL_005c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0061:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0067:  ldloca.s   V_1
    IL_0069:  call       "Sub System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.GetResult()"
    IL_006e:  ldloca.s   V_1
    IL_0070:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0076:  ldarg.0
    IL_0077:  ldarg.0
    IL_0078:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_007d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0082:  ldarg.0
    IL_0083:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0088:  ldloca.s   V_3
    IL_008a:  initobj    "SM$T"
    IL_0090:  ldloc.3
    IL_0091:  beq        IL_0116
    IL_0096:  ldarg.0
    IL_0097:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_009c:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_00a1:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_00a6:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00ab:  stloc.s    V_4
    IL_00ad:  ldloca.s   V_4
    IL_00af:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_00b4:  brtrue.s   IL_00f7
    IL_00b6:  ldarg.0
    IL_00b7:  ldc.i4.1
    IL_00b8:  dup
    IL_00b9:  stloc.0
    IL_00ba:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00bf:  ldarg.0
    IL_00c0:  ldloc.s    V_4
    IL_00c2:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00c7:  ldarg.0
    IL_00c8:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00cd:  ldloca.s   V_4
    IL_00cf:  ldarg.0
    IL_00d0:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_00d5:  leave      IL_015d
    IL_00da:  ldarg.0
    IL_00db:  ldc.i4.m1
    IL_00dc:  dup
    IL_00dd:  stloc.0
    IL_00de:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00e3:  ldarg.0
    IL_00e4:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00e9:  stloc.s    V_4
    IL_00eb:  ldarg.0
    IL_00ec:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00f1:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00f7:  ldarg.0
    IL_00f8:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_00fd:  box        "SM$T"
    IL_0102:  ldloca.s   V_4
    IL_0104:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0109:  ldloca.s   V_4
    IL_010b:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0111:  callvirt   "Sub IMoveable.GetName(Integer)"
    IL_0116:  ldarg.0
    IL_0117:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_011c:  initobj    "SM$T"
    IL_0122:  leave.s    IL_0148
  }
  catch System.Exception
  {
    IL_0124:  dup
    IL_0125:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_012a:  stloc.s    V_5
    IL_012c:  ldarg.0
    IL_012d:  ldc.i4.s   -2
    IL_012f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0134:  ldarg.0
    IL_0135:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_013a:  ldloc.s    V_5
    IL_013c:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0141:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0146:  leave.s    IL_015d
  }
  IL_0148:  ldarg.0
  IL_0149:  ldc.i4.s   -2
  IL_014b:  dup
  IL_014c:  stloc.0
  IL_014d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0152:  ldarg.0
  IL_0153:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0158:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_015d:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      427 (0x1ab)
  .maxstack  3
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter V_1,
                System.Runtime.CompilerServices.YieldAwaitable V_2,
                SM$T V_3,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004b
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_010e
    IL_0011:  call       "Function System.Threading.Tasks.Task.Yield() As System.Runtime.CompilerServices.YieldAwaitable"
    IL_0016:  stloc.2
    IL_0017:  ldloca.s   V_2
    IL_0019:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.GetAwaiter() As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_001e:  stloc.1
    IL_001f:  ldloca.s   V_1
    IL_0021:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.get_IsCompleted() As Boolean"
    IL_0026:  brtrue.s   IL_0067
    IL_0028:  ldarg.0
    IL_0029:  ldc.i4.0
    IL_002a:  dup
    IL_002b:  stloc.0
    IL_002c:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0031:  ldarg.0
    IL_0032:  ldloc.1
    IL_0033:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0038:  ldarg.0
    IL_0039:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003e:  ldloca.s   V_1
    IL_0040:  ldarg.0
    IL_0041:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_0046:  leave      IL_01aa
    IL_004b:  ldarg.0
    IL_004c:  ldc.i4.m1
    IL_004d:  dup
    IL_004e:  stloc.0
    IL_004f:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0054:  ldarg.0
    IL_0055:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_005a:  stloc.1
    IL_005b:  ldarg.0
    IL_005c:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0061:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0067:  ldloca.s   V_1
    IL_0069:  call       "Sub System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.GetResult()"
    IL_006e:  ldloca.s   V_1
    IL_0070:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0076:  ldloca.s   V_3
    IL_0078:  initobj    "SM$T"
    IL_007e:  ldloc.3
    IL_007f:  box        "SM$T"
    IL_0084:  brtrue.s   IL_0092
    IL_0086:  ldarg.0
    IL_0087:  ldarg.0
    IL_0088:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_008d:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T"
    IL_0092:  ldloca.s   V_3
    IL_0094:  initobj    "SM$T"
    IL_009a:  ldloc.3
    IL_009b:  box        "SM$T"
    IL_00a0:  brtrue.s   IL_00ca
    IL_00a2:  ldloca.s   V_3
    IL_00a4:  initobj    "SM$T"
    IL_00aa:  ldloc.3
    IL_00ab:  box        "SM$T"
    IL_00b0:  brtrue.s   IL_00ba
    IL_00b2:  ldarg.0
    IL_00b3:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T"
    IL_00b8:  br.s       IL_00c0
    IL_00ba:  ldarg.0
    IL_00bb:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00c0:  box        "SM$T"
    IL_00c5:  brfalse    IL_0163
    IL_00ca:  ldarg.0
    IL_00cb:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00d0:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_00d5:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_00da:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00df:  stloc.s    V_4
    IL_00e1:  ldloca.s   V_4
    IL_00e3:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_00e8:  brtrue.s   IL_012b
    IL_00ea:  ldarg.0
    IL_00eb:  ldc.i4.1
    IL_00ec:  dup
    IL_00ed:  stloc.0
    IL_00ee:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00f3:  ldarg.0
    IL_00f4:  ldloc.s    V_4
    IL_00f6:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00fb:  ldarg.0
    IL_00fc:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0101:  ldloca.s   V_4
    IL_0103:  ldarg.0
    IL_0104:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_0109:  leave      IL_01aa
    IL_010e:  ldarg.0
    IL_010f:  ldc.i4.m1
    IL_0110:  dup
    IL_0111:  stloc.0
    IL_0112:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0117:  ldarg.0
    IL_0118:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_011d:  stloc.s    V_4
    IL_011f:  ldarg.0
    IL_0120:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0125:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_012b:  ldloca.s   V_3
    IL_012d:  initobj    "SM$T"
    IL_0133:  ldloc.3
    IL_0134:  box        "SM$T"
    IL_0139:  brtrue.s   IL_0143
    IL_013b:  ldarg.0
    IL_013c:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T"
    IL_0141:  br.s       IL_0149
    IL_0143:  ldarg.0
    IL_0144:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0149:  ldloca.s   V_4
    IL_014b:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0150:  ldloca.s   V_4
    IL_0152:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0158:  constrained. "SM$T"
    IL_015e:  callvirt   "Sub IMoveable.GetName(Integer)"
    IL_0163:  ldarg.0
    IL_0164:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T"
    IL_0169:  initobj    "SM$T"
    IL_016f:  leave.s    IL_0195
  }
  catch System.Exception
  {
    IL_0171:  dup
    IL_0172:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0177:  stloc.s    V_5
    IL_0179:  ldarg.0
    IL_017a:  ldc.i4.s   -2
    IL_017c:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0181:  ldarg.0
    IL_0182:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0187:  ldloc.s    V_5
    IL_0189:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_018e:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0193:  leave.s    IL_01aa
  }
  IL_0195:  ldarg.0
  IL_0196:  ldc.i4.s   -2
  IL_0198:  dup
  IL_0199:  stloc.0
  IL_019a:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_019f:  ldarg.0
  IL_01a0:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_01a5:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_01aa:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput, verify:=Verification.Skipped).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Call_Conditional_Struct_Async_02()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Sub GetName(x As Integer)
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Sub GetName(x As Integer) Implements IMoveable.GetName
        Console.WriteLine("Position GetName for item '{0}'", Me.Name)
    End Sub
End Structure

Class Program
    Shared Sub Main()
        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        await Task.Yield()
        item?.GetName(await GetOffsetAsync(GetOffset(item)))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position GetName for item '-1'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Property_Class()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(item As T)
        item.Position += GetOffset(item)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        item.Position += GetOffset(item)
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       35 (0x23)
  .maxstack  3
  .locals init (T V_0)
  IL_0000:  ldarg.0
  IL_0001:  dup
  IL_0002:  stloc.0
  IL_0003:  box        "T"
  IL_0008:  ldloca.s   V_0
  IL_000a:  constrained. "T"
  IL_0010:  callvirt   "Function IMoveable.get_Position() As Integer"
  IL_0015:  ldarga.s   V_0
  IL_0017:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_001c:  add.ovf
  IL_001d:  callvirt   "Sub IMoveable.set_Position(Integer)"
  IL_0022:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
  // Code size       59 (0x3b)
  .maxstack  3
  .locals init (T V_0)
  IL_0000:  ldarga.s   V_0
  IL_0002:  ldloca.s   V_0
  IL_0004:  initobj    "T"
  IL_000a:  ldloc.0
  IL_000b:  box        "T"
  IL_0010:  brtrue.s   IL_001a
  IL_0012:  ldobj      "T"
  IL_0017:  stloc.0
  IL_0018:  ldloca.s   V_0
  IL_001a:  ldarga.s   V_0
  IL_001c:  constrained. "T"
  IL_0022:  callvirt   "Function IMoveable.get_Position() As Integer"
  IL_0027:  ldarga.s   V_0
  IL_0029:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_002e:  add.ovf
  IL_002f:  constrained. "T"
  IL_0035:  callvirt   "Sub IMoveable.set_Position(Integer)"
  IL_003a:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Property_Struct()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(item As T)
        item.Position += GetOffset(item)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        item.Position += GetOffset(item)
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '-1'
Position get for item '2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       35 (0x23)
  .maxstack  3
  IL_0000:  ldarga.s   V_0
  IL_0002:  ldarga.s   V_0
  IL_0004:  constrained. "T"
  IL_000a:  callvirt   "Function IMoveable.get_Position() As Integer"
  IL_000f:  ldarga.s   V_0
  IL_0011:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0016:  add.ovf
  IL_0017:  constrained. "T"
  IL_001d:  callvirt   "Sub IMoveable.set_Position(Integer)"
  IL_0022:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Property_Class_Ref()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(ByRef item As T)
        item.Position += GetOffset(item)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        item.Position += GetOffset(item)
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       39 (0x27)
  .maxstack  3
  .locals init (T V_0)
  IL_0000:  ldarg.0
  IL_0001:  ldobj      "T"
  IL_0006:  dup
  IL_0007:  stloc.0
  IL_0008:  box        "T"
  IL_000d:  ldloca.s   V_0
  IL_000f:  constrained. "T"
  IL_0015:  callvirt   "Function IMoveable.get_Position() As Integer"
  IL_001a:  ldarg.0
  IL_001b:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0020:  add.ovf
  IL_0021:  callvirt   "Sub IMoveable.set_Position(Integer)"
  IL_0026:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
  // Code size       56 (0x38)
  .maxstack  3
  .locals init (T V_0)
  IL_0000:  ldarg.0
  IL_0001:  ldloca.s   V_0
  IL_0003:  initobj    "T"
  IL_0009:  ldloc.0
  IL_000a:  box        "T"
  IL_000f:  brtrue.s   IL_0019
  IL_0011:  ldobj      "T"
  IL_0016:  stloc.0
  IL_0017:  ldloca.s   V_0
  IL_0019:  ldarg.0
  IL_001a:  constrained. "T"
  IL_0020:  callvirt   "Function IMoveable.get_Position() As Integer"
  IL_0025:  ldarg.0
  IL_0026:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_002b:  add.ovf
  IL_002c:  constrained. "T"
  IL_0032:  callvirt   "Sub IMoveable.set_Position(Integer)"
  IL_0037:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Property_Struct_Ref()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(ByRef item As T)
        item.Position += GetOffset(item)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        item.Position += GetOffset(item)
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '-1'
Position get for item '2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       32 (0x20)
  .maxstack  3
  IL_0000:  ldarg.0
  IL_0001:  ldarg.0
  IL_0002:  constrained. "T"
  IL_0008:  callvirt   "Function IMoveable.get_Position() As Integer"
  IL_000d:  ldarg.0
  IL_000e:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0013:  add.ovf
  IL_0014:  constrained. "T"
  IL_001a:  callvirt   "Sub IMoveable.set_Position(Integer)"
  IL_001f:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Property_Class_Async_01()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        item.Position += await GetOffsetAsync(GetOffset(item))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        item.Position += await GetOffsetAsync(GetOffset(item))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      256 (0x100)
  .maxstack  4
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                SM$T V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0078
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldarg.0
    IL_000d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0012:  dup
    IL_0013:  stloc.2
    IL_0014:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0019:  ldloc.2
    IL_001a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_001f:  ldarg.0
    IL_0020:  ldarg.0
    IL_0021:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0026:  constrained. "SM$T"
    IL_002c:  callvirt   "Function IMoveable.get_Position() As Integer"
    IL_0031:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_0036:  ldarg.0
    IL_0037:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_003c:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0041:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0046:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_004b:  stloc.1
    IL_004c:  ldloca.s   V_1
    IL_004e:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0053:  brtrue.s   IL_0094
    IL_0055:  ldarg.0
    IL_0056:  ldc.i4.0
    IL_0057:  dup
    IL_0058:  stloc.0
    IL_0059:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_005e:  ldarg.0
    IL_005f:  ldloc.1
    IL_0060:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0065:  ldarg.0
    IL_0066:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_006b:  ldloca.s   V_1
    IL_006d:  ldarg.0
    IL_006e:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0073:  leave      IL_00ff
    IL_0078:  ldarg.0
    IL_0079:  ldc.i4.m1
    IL_007a:  dup
    IL_007b:  stloc.0
    IL_007c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0081:  ldarg.0
    IL_0082:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0087:  stloc.1
    IL_0088:  ldarg.0
    IL_0089:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008e:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0094:  ldarg.0
    IL_0095:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_009a:  box        "SM$T"
    IL_009f:  ldarg.0
    IL_00a0:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_00a5:  ldloca.s   V_1
    IL_00a7:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00ac:  ldloca.s   V_1
    IL_00ae:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b4:  add.ovf
    IL_00b5:  callvirt   "Sub IMoveable.set_Position(Integer)"
    IL_00ba:  ldarg.0
    IL_00bb:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_00c0:  initobj    "SM$T"
    IL_00c6:  leave.s    IL_00ea
  }
  catch System.Exception
  {
    IL_00c8:  dup
    IL_00c9:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00ce:  stloc.3
    IL_00cf:  ldarg.0
    IL_00d0:  ldc.i4.s   -2
    IL_00d2:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00d7:  ldarg.0
    IL_00d8:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00dd:  ldloc.3
    IL_00de:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00e3:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00e8:  leave.s    IL_00ff
  }
  IL_00ea:  ldarg.0
  IL_00eb:  ldc.i4.s   -2
  IL_00ed:  dup
  IL_00ee:  stloc.0
  IL_00ef:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00f4:  ldarg.0
  IL_00f5:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00fa:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_00ff:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      288 (0x120)
  .maxstack  4
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                SM$T V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_007f
    IL_000a:  ldloca.s   V_2
    IL_000c:  initobj    "SM$T"
    IL_0012:  ldloc.2
    IL_0013:  box        "SM$T"
    IL_0018:  brtrue.s   IL_0026
    IL_001a:  ldarg.0
    IL_001b:  ldarg.0
    IL_001c:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0021:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T"
    IL_0026:  ldarg.0
    IL_0027:  ldarg.0
    IL_0028:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_002d:  constrained. "SM$T"
    IL_0033:  callvirt   "Function IMoveable.get_Position() As Integer"
    IL_0038:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As Integer"
    IL_003d:  ldarg.0
    IL_003e:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0043:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0048:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_004d:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0052:  stloc.1
    IL_0053:  ldloca.s   V_1
    IL_0055:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_005a:  brtrue.s   IL_009b
    IL_005c:  ldarg.0
    IL_005d:  ldc.i4.0
    IL_005e:  dup
    IL_005f:  stloc.0
    IL_0060:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0065:  ldarg.0
    IL_0066:  ldloc.1
    IL_0067:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_006c:  ldarg.0
    IL_006d:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0072:  ldloca.s   V_1
    IL_0074:  ldarg.0
    IL_0075:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_007a:  leave      IL_011f
    IL_007f:  ldarg.0
    IL_0080:  ldc.i4.m1
    IL_0081:  dup
    IL_0082:  stloc.0
    IL_0083:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0088:  ldarg.0
    IL_0089:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008e:  stloc.1
    IL_008f:  ldarg.0
    IL_0090:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0095:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009b:  ldloca.s   V_2
    IL_009d:  initobj    "SM$T"
    IL_00a3:  ldloc.2
    IL_00a4:  box        "SM$T"
    IL_00a9:  brtrue.s   IL_00b3
    IL_00ab:  ldarg.0
    IL_00ac:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T"
    IL_00b1:  br.s       IL_00b9
    IL_00b3:  ldarg.0
    IL_00b4:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00b9:  ldarg.0
    IL_00ba:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As Integer"
    IL_00bf:  ldloca.s   V_1
    IL_00c1:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00c6:  ldloca.s   V_1
    IL_00c8:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00ce:  add.ovf
    IL_00cf:  constrained. "SM$T"
    IL_00d5:  callvirt   "Sub IMoveable.set_Position(Integer)"
    IL_00da:  ldarg.0
    IL_00db:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T"
    IL_00e0:  initobj    "SM$T"
    IL_00e6:  leave.s    IL_010a
  }
  catch System.Exception
  {
    IL_00e8:  dup
    IL_00e9:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00ee:  stloc.3
    IL_00ef:  ldarg.0
    IL_00f0:  ldc.i4.s   -2
    IL_00f2:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00f7:  ldarg.0
    IL_00f8:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00fd:  ldloc.3
    IL_00fe:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0103:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0108:  leave.s    IL_011f
  }
  IL_010a:  ldarg.0
  IL_010b:  ldc.i4.s   -2
  IL_010d:  dup
  IL_010e:  stloc.0
  IL_010f:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0114:  ldarg.0
  IL_0115:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_011a:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_011f:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Property_Struct_Async_01()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        item.Position += await GetOffsetAsync(GetOffset(item))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        item.Position += await GetOffsetAsync(GetOffset(item))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '-1'
Position get for item '2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      221 (0xdd)
  .maxstack  4
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                System.Exception V_2)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0060
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0011:  constrained. "SM$T"
    IL_0017:  callvirt   "Function IMoveable.get_Position() As Integer"
    IL_001c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_0021:  ldarg.0
    IL_0022:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0027:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_002c:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0031:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0036:  stloc.1
    IL_0037:  ldloca.s   V_1
    IL_0039:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_003e:  brtrue.s   IL_007c
    IL_0040:  ldarg.0
    IL_0041:  ldc.i4.0
    IL_0042:  dup
    IL_0043:  stloc.0
    IL_0044:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0049:  ldarg.0
    IL_004a:  ldloc.1
    IL_004b:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0050:  ldarg.0
    IL_0051:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0056:  ldloca.s   V_1
    IL_0058:  ldarg.0
    IL_0059:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_005e:  leave.s    IL_00dc
    IL_0060:  ldarg.0
    IL_0061:  ldc.i4.m1
    IL_0062:  dup
    IL_0063:  stloc.0
    IL_0064:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0069:  ldarg.0
    IL_006a:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_006f:  stloc.1
    IL_0070:  ldarg.0
    IL_0071:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0076:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007c:  ldarg.0
    IL_007d:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0082:  ldarg.0
    IL_0083:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_0088:  ldloca.s   V_1
    IL_008a:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_008f:  ldloca.s   V_1
    IL_0091:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0097:  add.ovf
    IL_0098:  constrained. "SM$T"
    IL_009e:  callvirt   "Sub IMoveable.set_Position(Integer)"
    IL_00a3:  leave.s    IL_00c7
  }
  catch System.Exception
  {
    IL_00a5:  dup
    IL_00a6:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00ab:  stloc.2
    IL_00ac:  ldarg.0
    IL_00ad:  ldc.i4.s   -2
    IL_00af:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00b4:  ldarg.0
    IL_00b5:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00ba:  ldloc.2
    IL_00bb:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00c0:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00c5:  leave.s    IL_00dc
  }
  IL_00c7:  ldarg.0
  IL_00c8:  ldc.i4.s   -2
  IL_00ca:  dup
  IL_00cb:  stloc.0
  IL_00cc:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00d1:  ldarg.0
  IL_00d2:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00d7:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_00dc:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Property_Class_Async_02()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        await Task.Yield()
        item.Position += await GetOffsetAsync(GetOffset(item))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        await Task.Yield()
        item.Position += await GetOffsetAsync(GetOffset(item))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      368 (0x170)
  .maxstack  4
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter V_1,
                System.Runtime.CompilerServices.YieldAwaitable V_2,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_3,
                SM$T V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004b
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_00e6
    IL_0011:  call       "Function System.Threading.Tasks.Task.Yield() As System.Runtime.CompilerServices.YieldAwaitable"
    IL_0016:  stloc.2
    IL_0017:  ldloca.s   V_2
    IL_0019:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.GetAwaiter() As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_001e:  stloc.1
    IL_001f:  ldloca.s   V_1
    IL_0021:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.get_IsCompleted() As Boolean"
    IL_0026:  brtrue.s   IL_0067
    IL_0028:  ldarg.0
    IL_0029:  ldc.i4.0
    IL_002a:  dup
    IL_002b:  stloc.0
    IL_002c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0031:  ldarg.0
    IL_0032:  ldloc.1
    IL_0033:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0038:  ldarg.0
    IL_0039:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003e:  ldloca.s   V_1
    IL_0040:  ldarg.0
    IL_0041:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0046:  leave      IL_016f
    IL_004b:  ldarg.0
    IL_004c:  ldc.i4.m1
    IL_004d:  dup
    IL_004e:  stloc.0
    IL_004f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0054:  ldarg.0
    IL_0055:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_005a:  stloc.1
    IL_005b:  ldarg.0
    IL_005c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0061:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0067:  ldloca.s   V_1
    IL_0069:  call       "Sub System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.GetResult()"
    IL_006e:  ldloca.s   V_1
    IL_0070:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0076:  ldarg.0
    IL_0077:  ldarg.0
    IL_0078:  ldarg.0
    IL_0079:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_007e:  dup
    IL_007f:  stloc.s    V_4
    IL_0081:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0086:  ldloc.s    V_4
    IL_0088:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_008d:  ldarg.0
    IL_008e:  ldarg.0
    IL_008f:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0094:  constrained. "SM$T"
    IL_009a:  callvirt   "Function IMoveable.get_Position() As Integer"
    IL_009f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_00a4:  ldarg.0
    IL_00a5:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_00aa:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_00af:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_00b4:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b9:  stloc.3
    IL_00ba:  ldloca.s   V_3
    IL_00bc:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_00c1:  brtrue.s   IL_0102
    IL_00c3:  ldarg.0
    IL_00c4:  ldc.i4.1
    IL_00c5:  dup
    IL_00c6:  stloc.0
    IL_00c7:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00cc:  ldarg.0
    IL_00cd:  ldloc.3
    IL_00ce:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d3:  ldarg.0
    IL_00d4:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00d9:  ldloca.s   V_3
    IL_00db:  ldarg.0
    IL_00dc:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_00e1:  leave      IL_016f
    IL_00e6:  ldarg.0
    IL_00e7:  ldc.i4.m1
    IL_00e8:  dup
    IL_00e9:  stloc.0
    IL_00ea:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00ef:  ldarg.0
    IL_00f0:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00f5:  stloc.3
    IL_00f6:  ldarg.0
    IL_00f7:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00fc:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0102:  ldarg.0
    IL_0103:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_0108:  box        "SM$T"
    IL_010d:  ldarg.0
    IL_010e:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_0113:  ldloca.s   V_3
    IL_0115:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_011a:  ldloca.s   V_3
    IL_011c:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0122:  add.ovf
    IL_0123:  callvirt   "Sub IMoveable.set_Position(Integer)"
    IL_0128:  ldarg.0
    IL_0129:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_012e:  initobj    "SM$T"
    IL_0134:  leave.s    IL_015a
  }
  catch System.Exception
  {
    IL_0136:  dup
    IL_0137:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_013c:  stloc.s    V_5
    IL_013e:  ldarg.0
    IL_013f:  ldc.i4.s   -2
    IL_0141:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0146:  ldarg.0
    IL_0147:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_014c:  ldloc.s    V_5
    IL_014e:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0153:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0158:  leave.s    IL_016f
  }
  IL_015a:  ldarg.0
  IL_015b:  ldc.i4.s   -2
  IL_015d:  dup
  IL_015e:  stloc.0
  IL_015f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0164:  ldarg.0
  IL_0165:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_016a:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_016f:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      400 (0x190)
  .maxstack  4
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter V_1,
                System.Runtime.CompilerServices.YieldAwaitable V_2,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_3,
                SM$T V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004b
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_00ec
    IL_0011:  call       "Function System.Threading.Tasks.Task.Yield() As System.Runtime.CompilerServices.YieldAwaitable"
    IL_0016:  stloc.2
    IL_0017:  ldloca.s   V_2
    IL_0019:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.GetAwaiter() As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_001e:  stloc.1
    IL_001f:  ldloca.s   V_1
    IL_0021:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.get_IsCompleted() As Boolean"
    IL_0026:  brtrue.s   IL_0067
    IL_0028:  ldarg.0
    IL_0029:  ldc.i4.0
    IL_002a:  dup
    IL_002b:  stloc.0
    IL_002c:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0031:  ldarg.0
    IL_0032:  ldloc.1
    IL_0033:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0038:  ldarg.0
    IL_0039:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003e:  ldloca.s   V_1
    IL_0040:  ldarg.0
    IL_0041:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_0046:  leave      IL_018f
    IL_004b:  ldarg.0
    IL_004c:  ldc.i4.m1
    IL_004d:  dup
    IL_004e:  stloc.0
    IL_004f:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0054:  ldarg.0
    IL_0055:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_005a:  stloc.1
    IL_005b:  ldarg.0
    IL_005c:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0061:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0067:  ldloca.s   V_1
    IL_0069:  call       "Sub System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.GetResult()"
    IL_006e:  ldloca.s   V_1
    IL_0070:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0076:  ldloca.s   V_4
    IL_0078:  initobj    "SM$T"
    IL_007e:  ldloc.s    V_4
    IL_0080:  box        "SM$T"
    IL_0085:  brtrue.s   IL_0093
    IL_0087:  ldarg.0
    IL_0088:  ldarg.0
    IL_0089:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_008e:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T"
    IL_0093:  ldarg.0
    IL_0094:  ldarg.0
    IL_0095:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_009a:  constrained. "SM$T"
    IL_00a0:  callvirt   "Function IMoveable.get_Position() As Integer"
    IL_00a5:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As Integer"
    IL_00aa:  ldarg.0
    IL_00ab:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00b0:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_00b5:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_00ba:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00bf:  stloc.3
    IL_00c0:  ldloca.s   V_3
    IL_00c2:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_00c7:  brtrue.s   IL_0108
    IL_00c9:  ldarg.0
    IL_00ca:  ldc.i4.1
    IL_00cb:  dup
    IL_00cc:  stloc.0
    IL_00cd:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00d2:  ldarg.0
    IL_00d3:  ldloc.3
    IL_00d4:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d9:  ldarg.0
    IL_00da:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00df:  ldloca.s   V_3
    IL_00e1:  ldarg.0
    IL_00e2:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_00e7:  leave      IL_018f
    IL_00ec:  ldarg.0
    IL_00ed:  ldc.i4.m1
    IL_00ee:  dup
    IL_00ef:  stloc.0
    IL_00f0:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00f5:  ldarg.0
    IL_00f6:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00fb:  stloc.3
    IL_00fc:  ldarg.0
    IL_00fd:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0102:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0108:  ldloca.s   V_4
    IL_010a:  initobj    "SM$T"
    IL_0110:  ldloc.s    V_4
    IL_0112:  box        "SM$T"
    IL_0117:  brtrue.s   IL_0121
    IL_0119:  ldarg.0
    IL_011a:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T"
    IL_011f:  br.s       IL_0127
    IL_0121:  ldarg.0
    IL_0122:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0127:  ldarg.0
    IL_0128:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As Integer"
    IL_012d:  ldloca.s   V_3
    IL_012f:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0134:  ldloca.s   V_3
    IL_0136:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_013c:  add.ovf
    IL_013d:  constrained. "SM$T"
    IL_0143:  callvirt   "Sub IMoveable.set_Position(Integer)"
    IL_0148:  ldarg.0
    IL_0149:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T"
    IL_014e:  initobj    "SM$T"
    IL_0154:  leave.s    IL_017a
  }
  catch System.Exception
  {
    IL_0156:  dup
    IL_0157:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_015c:  stloc.s    V_5
    IL_015e:  ldarg.0
    IL_015f:  ldc.i4.s   -2
    IL_0161:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0166:  ldarg.0
    IL_0167:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_016c:  ldloc.s    V_5
    IL_016e:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0173:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0178:  leave.s    IL_018f
  }
  IL_017a:  ldarg.0
  IL_017b:  ldc.i4.s   -2
  IL_017d:  dup
  IL_017e:  stloc.0
  IL_017f:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0184:  ldarg.0
  IL_0185:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_018a:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_018f:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Property_Struct_Async_02()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        await Task.Yield()
        item.Position += await GetOffsetAsync(GetOffset(item))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        await Task.Yield()
        item.Position += await GetOffsetAsync(GetOffset(item))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '-1'
Position get for item '2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      331 (0x14b)
  .maxstack  4
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter V_1,
                System.Runtime.CompilerServices.YieldAwaitable V_2,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004b
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_00cc
    IL_0011:  call       "Function System.Threading.Tasks.Task.Yield() As System.Runtime.CompilerServices.YieldAwaitable"
    IL_0016:  stloc.2
    IL_0017:  ldloca.s   V_2
    IL_0019:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.GetAwaiter() As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_001e:  stloc.1
    IL_001f:  ldloca.s   V_1
    IL_0021:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.get_IsCompleted() As Boolean"
    IL_0026:  brtrue.s   IL_0067
    IL_0028:  ldarg.0
    IL_0029:  ldc.i4.0
    IL_002a:  dup
    IL_002b:  stloc.0
    IL_002c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0031:  ldarg.0
    IL_0032:  ldloc.1
    IL_0033:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0038:  ldarg.0
    IL_0039:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003e:  ldloca.s   V_1
    IL_0040:  ldarg.0
    IL_0041:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0046:  leave      IL_014a
    IL_004b:  ldarg.0
    IL_004c:  ldc.i4.m1
    IL_004d:  dup
    IL_004e:  stloc.0
    IL_004f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0054:  ldarg.0
    IL_0055:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_005a:  stloc.1
    IL_005b:  ldarg.0
    IL_005c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0061:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0067:  ldloca.s   V_1
    IL_0069:  call       "Sub System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.GetResult()"
    IL_006e:  ldloca.s   V_1
    IL_0070:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0076:  ldarg.0
    IL_0077:  ldarg.0
    IL_0078:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_007d:  constrained. "SM$T"
    IL_0083:  callvirt   "Function IMoveable.get_Position() As Integer"
    IL_0088:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_008d:  ldarg.0
    IL_008e:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0093:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0098:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_009d:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a2:  stloc.3
    IL_00a3:  ldloca.s   V_3
    IL_00a5:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_00aa:  brtrue.s   IL_00e8
    IL_00ac:  ldarg.0
    IL_00ad:  ldc.i4.1
    IL_00ae:  dup
    IL_00af:  stloc.0
    IL_00b0:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00b5:  ldarg.0
    IL_00b6:  ldloc.3
    IL_00b7:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00bc:  ldarg.0
    IL_00bd:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00c2:  ldloca.s   V_3
    IL_00c4:  ldarg.0
    IL_00c5:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_00ca:  leave.s    IL_014a
    IL_00cc:  ldarg.0
    IL_00cd:  ldc.i4.m1
    IL_00ce:  dup
    IL_00cf:  stloc.0
    IL_00d0:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00d5:  ldarg.0
    IL_00d6:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00db:  stloc.3
    IL_00dc:  ldarg.0
    IL_00dd:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00e2:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00e8:  ldarg.0
    IL_00e9:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_00ee:  ldarg.0
    IL_00ef:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_00f4:  ldloca.s   V_3
    IL_00f6:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00fb:  ldloca.s   V_3
    IL_00fd:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0103:  add.ovf
    IL_0104:  constrained. "SM$T"
    IL_010a:  callvirt   "Sub IMoveable.set_Position(Integer)"
    IL_010f:  leave.s    IL_0135
  }
  catch System.Exception
  {
    IL_0111:  dup
    IL_0112:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0117:  stloc.s    V_4
    IL_0119:  ldarg.0
    IL_011a:  ldc.i4.s   -2
    IL_011c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0121:  ldarg.0
    IL_0122:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0127:  ldloc.s    V_4
    IL_0129:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_012e:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0133:  leave.s    IL_014a
  }
  IL_0135:  ldarg.0
  IL_0136:  ldc.i4.s   -2
  IL_0138:  dup
  IL_0139:  stloc.0
  IL_013a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_013f:  ldarg.0
  IL_0140:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0145:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_014a:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Index()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(item As T)
        item.Position(GetOffset(item)) += 1
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        item.Position(GetOffset(item)) += 1
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       39 (0x27)
  .maxstack  4
  .locals init (T V_0,
                Integer V_1)
  IL_0000:  ldarg.0
  IL_0001:  dup
  IL_0002:  stloc.0
  IL_0003:  box        "T"
  IL_0008:  ldarga.s   V_0
  IL_000a:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_000f:  dup
  IL_0010:  stloc.1
  IL_0011:  ldloca.s   V_0
  IL_0013:  ldloc.1
  IL_0014:  constrained. "T"
  IL_001a:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_001f:  ldc.i4.1
  IL_0020:  add.ovf
  IL_0021:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0026:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
  // Code size       81 (0x51)
  .maxstack  4
  .locals init (T V_0,
                Integer V_1,
                T V_2)
  IL_0000:  ldloca.s   V_2
  IL_0002:  initobj    "T"
  IL_0008:  ldloc.2
  IL_0009:  box        "T"
  IL_000e:  brtrue.s   IL_0016
  IL_0010:  ldarg.0
  IL_0011:  stloc.0
  IL_0012:  ldloca.s   V_0
  IL_0014:  br.s       IL_0018
  IL_0016:  ldarga.s   V_0
  IL_0018:  ldarga.s   V_0
  IL_001a:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_001f:  dup
  IL_0020:  stloc.1
  IL_0021:  ldloca.s   V_2
  IL_0023:  initobj    "T"
  IL_0029:  ldloc.2
  IL_002a:  box        "T"
  IL_002f:  brtrue.s   IL_0035
  IL_0031:  ldloca.s   V_0
  IL_0033:  br.s       IL_0037
  IL_0035:  ldarga.s   V_0
  IL_0037:  ldloc.1
  IL_0038:  constrained. "T"
  IL_003e:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0043:  ldc.i4.1
  IL_0044:  add.ovf
  IL_0045:  constrained. "T"
  IL_004b:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0050:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Index()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(item As T)
        item.Position(GetOffset(item)) += 1
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        item.Position(GetOffset(item)) += 1
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-1'
Position get for item '-2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       39 (0x27)
  .maxstack  4
  .locals init (Integer V_0)
  IL_0000:  ldarga.s   V_0
  IL_0002:  ldarga.s   V_0
  IL_0004:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0009:  dup
  IL_000a:  stloc.0
  IL_000b:  ldarga.s   V_0
  IL_000d:  ldloc.0
  IL_000e:  constrained. "T"
  IL_0014:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0019:  ldc.i4.1
  IL_001a:  add.ovf
  IL_001b:  constrained. "T"
  IL_0021:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0026:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Index_Ref()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(ByRef item As T)
        item.Position(GetOffset(item)) += 1
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        item.Position(GetOffset(item)) += 1
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       43 (0x2b)
  .maxstack  4
  .locals init (T V_0,
                Integer V_1)
  IL_0000:  ldarg.0
  IL_0001:  ldobj      "T"
  IL_0006:  dup
  IL_0007:  stloc.0
  IL_0008:  box        "T"
  IL_000d:  ldarg.0
  IL_000e:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0013:  dup
  IL_0014:  stloc.1
  IL_0015:  ldloca.s   V_0
  IL_0017:  ldloc.1
  IL_0018:  constrained. "T"
  IL_001e:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0023:  ldc.i4.1
  IL_0024:  add.ovf
  IL_0025:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_002a:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
      // Code size       83 (0x53)
  .maxstack  4
  .locals init (T V_0,
                Integer V_1,
                T V_2)
  IL_0000:  ldloca.s   V_2
  IL_0002:  initobj    "T"
  IL_0008:  ldloc.2
  IL_0009:  box        "T"
  IL_000e:  brtrue.s   IL_001b
  IL_0010:  ldarg.0
  IL_0011:  ldobj      "T"
  IL_0016:  stloc.0
  IL_0017:  ldloca.s   V_0
  IL_0019:  br.s       IL_001c
  IL_001b:  ldarg.0
  IL_001c:  ldarg.0
  IL_001d:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0022:  dup
  IL_0023:  stloc.1
  IL_0024:  ldloca.s   V_2
  IL_0026:  initobj    "T"
  IL_002c:  ldloc.2
  IL_002d:  box        "T"
  IL_0032:  brtrue.s   IL_0038
  IL_0034:  ldloca.s   V_0
  IL_0036:  br.s       IL_0039
  IL_0038:  ldarg.0
  IL_0039:  ldloc.1
  IL_003a:  constrained. "T"
  IL_0040:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0045:  ldc.i4.1
  IL_0046:  add.ovf
  IL_0047:  constrained. "T"
  IL_004d:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0052:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Index_Ref()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(ByRef item As T)
        item.Position(GetOffset(item)) += 1
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        item.Position(GetOffset(item)) += 1
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-1'
Position get for item '-2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       36 (0x24)
  .maxstack  4
  .locals init (Integer V_0)
  IL_0000:  ldarg.0
  IL_0001:  ldarg.0
  IL_0002:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0007:  dup
  IL_0008:  stloc.0
  IL_0009:  ldarg.0
  IL_000a:  ldloc.0
  IL_000b:  constrained. "T"
  IL_0011:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0016:  ldc.i4.1
  IL_0017:  add.ovf
  IL_0018:  constrained. "T"
  IL_001e:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0023:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Index_Async_01()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        item.Position(await GetOffsetAsync(GetOffset(item))) += 1
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        item.Position(await GetOffsetAsync(GetOffset(item))) += 1
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      250 (0xfa)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0061
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldarg.0
    IL_000d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0012:  dup
    IL_0013:  stloc.3
    IL_0014:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0019:  ldloc.3
    IL_001a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_001f:  ldarg.0
    IL_0020:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0025:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_002a:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_002f:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0034:  stloc.2
    IL_0035:  ldloca.s   V_2
    IL_0037:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_003c:  brtrue.s   IL_007d
    IL_003e:  ldarg.0
    IL_003f:  ldc.i4.0
    IL_0040:  dup
    IL_0041:  stloc.0
    IL_0042:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0047:  ldarg.0
    IL_0048:  ldloc.2
    IL_0049:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_004e:  ldarg.0
    IL_004f:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0054:  ldloca.s   V_2
    IL_0056:  ldarg.0
    IL_0057:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_005c:  leave      IL_00f9
    IL_0061:  ldarg.0
    IL_0062:  ldc.i4.m1
    IL_0063:  dup
    IL_0064:  stloc.0
    IL_0065:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_006a:  ldarg.0
    IL_006b:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0070:  stloc.2
    IL_0071:  ldarg.0
    IL_0072:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0077:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007d:  ldarg.0
    IL_007e:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0083:  box        "SM$T"
    IL_0088:  ldloca.s   V_2
    IL_008a:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_008f:  ldloca.s   V_2
    IL_0091:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0097:  dup
    IL_0098:  stloc.1
    IL_0099:  ldarg.0
    IL_009a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_009f:  ldloc.1
    IL_00a0:  constrained. "SM$T"
    IL_00a6:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_00ab:  ldc.i4.1
    IL_00ac:  add.ovf
    IL_00ad:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00b2:  ldarg.0
    IL_00b3:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_00b8:  initobj    "SM$T"
    IL_00be:  leave.s    IL_00e4
  }
  catch System.Exception
  {
    IL_00c0:  dup
    IL_00c1:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00c6:  stloc.s    V_4
    IL_00c8:  ldarg.0
    IL_00c9:  ldc.i4.s   -2
    IL_00cb:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00d0:  ldarg.0
    IL_00d1:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00d6:  ldloc.s    V_4
    IL_00d8:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00dd:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00e2:  leave.s    IL_00f9
  }
  IL_00e4:  ldarg.0
  IL_00e5:  ldc.i4.s   -2
  IL_00e7:  dup
  IL_00e8:  stloc.0
  IL_00e9:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00ee:  ldarg.0
  IL_00ef:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00f4:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_00f9:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      294 (0x126)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0068
    IL_000a:  ldloca.s   V_3
    IL_000c:  initobj    "SM$T"
    IL_0012:  ldloc.3
    IL_0013:  box        "SM$T"
    IL_0018:  brtrue.s   IL_0026
    IL_001a:  ldarg.0
    IL_001b:  ldarg.0
    IL_001c:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0021:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_0026:  ldarg.0
    IL_0027:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_002c:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0031:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0036:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_003b:  stloc.2
    IL_003c:  ldloca.s   V_2
    IL_003e:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0043:  brtrue.s   IL_0084
    IL_0045:  ldarg.0
    IL_0046:  ldc.i4.0
    IL_0047:  dup
    IL_0048:  stloc.0
    IL_0049:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_004e:  ldarg.0
    IL_004f:  ldloc.2
    IL_0050:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0055:  ldarg.0
    IL_0056:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_005b:  ldloca.s   V_2
    IL_005d:  ldarg.0
    IL_005e:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_0063:  leave      IL_0125
    IL_0068:  ldarg.0
    IL_0069:  ldc.i4.m1
    IL_006a:  dup
    IL_006b:  stloc.0
    IL_006c:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0071:  ldarg.0
    IL_0072:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0077:  stloc.2
    IL_0078:  ldarg.0
    IL_0079:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007e:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0084:  ldloca.s   V_3
    IL_0086:  initobj    "SM$T"
    IL_008c:  ldloc.3
    IL_008d:  box        "SM$T"
    IL_0092:  brtrue.s   IL_009c
    IL_0094:  ldarg.0
    IL_0095:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_009a:  br.s       IL_00a2
    IL_009c:  ldarg.0
    IL_009d:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00a2:  ldloca.s   V_2
    IL_00a4:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00a9:  ldloca.s   V_2
    IL_00ab:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b1:  dup
    IL_00b2:  stloc.1
    IL_00b3:  ldloca.s   V_3
    IL_00b5:  initobj    "SM$T"
    IL_00bb:  ldloc.3
    IL_00bc:  box        "SM$T"
    IL_00c1:  brtrue.s   IL_00cb
    IL_00c3:  ldarg.0
    IL_00c4:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_00c9:  br.s       IL_00d1
    IL_00cb:  ldarg.0
    IL_00cc:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00d1:  ldloc.1
    IL_00d2:  constrained. "SM$T"
    IL_00d8:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_00dd:  ldc.i4.1
    IL_00de:  add.ovf
    IL_00df:  constrained. "SM$T"
    IL_00e5:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00ea:  leave.s    IL_0110
  }
  catch System.Exception
  {
    IL_00ec:  dup
    IL_00ed:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00f2:  stloc.s    V_4
    IL_00f4:  ldarg.0
    IL_00f5:  ldc.i4.s   -2
    IL_00f7:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00fc:  ldarg.0
    IL_00fd:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0102:  ldloc.s    V_4
    IL_0104:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0109:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_010e:  leave.s    IL_0125
  }
  IL_0110:  ldarg.0
  IL_0111:  ldc.i4.s   -2
  IL_0113:  dup
  IL_0114:  stloc.0
  IL_0115:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_011a:  ldarg.0
  IL_011b:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0120:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0125:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Index_Async_01()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        item.Position(await GetOffsetAsync(GetOffset(item))) += 1
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        item.Position(await GetOffsetAsync(GetOffset(item))) += 1
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-1'
Position get for item '-2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      216 (0xd8)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004c
    IL_000a:  ldarg.0
    IL_000b:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0010:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0015:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_001a:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_001f:  stloc.2
    IL_0020:  ldloca.s   V_2
    IL_0022:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0027:  brtrue.s   IL_0068
    IL_0029:  ldarg.0
    IL_002a:  ldc.i4.0
    IL_002b:  dup
    IL_002c:  stloc.0
    IL_002d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0032:  ldarg.0
    IL_0033:  ldloc.2
    IL_0034:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0039:  ldarg.0
    IL_003a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003f:  ldloca.s   V_2
    IL_0041:  ldarg.0
    IL_0042:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0047:  leave      IL_00d7
    IL_004c:  ldarg.0
    IL_004d:  ldc.i4.m1
    IL_004e:  dup
    IL_004f:  stloc.0
    IL_0050:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0055:  ldarg.0
    IL_0056:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_005b:  stloc.2
    IL_005c:  ldarg.0
    IL_005d:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0062:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0068:  ldarg.0
    IL_0069:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_006e:  ldloca.s   V_2
    IL_0070:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0075:  ldloca.s   V_2
    IL_0077:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007d:  dup
    IL_007e:  stloc.1
    IL_007f:  ldarg.0
    IL_0080:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0085:  ldloc.1
    IL_0086:  constrained. "SM$T"
    IL_008c:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_0091:  ldc.i4.1
    IL_0092:  add.ovf
    IL_0093:  constrained. "SM$T"
    IL_0099:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_009e:  leave.s    IL_00c2
  }
  catch System.Exception
  {
    IL_00a0:  dup
    IL_00a1:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00a6:  stloc.3
    IL_00a7:  ldarg.0
    IL_00a8:  ldc.i4.s   -2
    IL_00aa:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00af:  ldarg.0
    IL_00b0:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00b5:  ldloc.3
    IL_00b6:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00bb:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00c0:  leave.s    IL_00d7
  }
  IL_00c2:  ldarg.0
  IL_00c3:  ldc.i4.s   -2
  IL_00c5:  dup
  IL_00c6:  stloc.0
  IL_00c7:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00cc:  ldarg.0
  IL_00cd:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00d2:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_00d7:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Index_Async_02()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x as Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x as Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        await Task.Yield()
        item.Position(await GetOffsetAsync(GetOffset(item))) += 1
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        await Task.Yield()
        item.Position(await GetOffsetAsync(GetOffset(item))) += 1
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      363 (0x16b)
  .maxstack  4
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter V_1,
                System.Runtime.CompilerServices.YieldAwaitable V_2,
                Integer V_3,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_4,
                SM$T V_5,
                System.Exception V_6)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004b
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_00d1
    IL_0011:  call       "Function System.Threading.Tasks.Task.Yield() As System.Runtime.CompilerServices.YieldAwaitable"
    IL_0016:  stloc.2
    IL_0017:  ldloca.s   V_2
    IL_0019:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.GetAwaiter() As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_001e:  stloc.1
    IL_001f:  ldloca.s   V_1
    IL_0021:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.get_IsCompleted() As Boolean"
    IL_0026:  brtrue.s   IL_0067
    IL_0028:  ldarg.0
    IL_0029:  ldc.i4.0
    IL_002a:  dup
    IL_002b:  stloc.0
    IL_002c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0031:  ldarg.0
    IL_0032:  ldloc.1
    IL_0033:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0038:  ldarg.0
    IL_0039:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003e:  ldloca.s   V_1
    IL_0040:  ldarg.0
    IL_0041:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0046:  leave      IL_016a
    IL_004b:  ldarg.0
    IL_004c:  ldc.i4.m1
    IL_004d:  dup
    IL_004e:  stloc.0
    IL_004f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0054:  ldarg.0
    IL_0055:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_005a:  stloc.1
    IL_005b:  ldarg.0
    IL_005c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0061:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0067:  ldloca.s   V_1
    IL_0069:  call       "Sub System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.GetResult()"
    IL_006e:  ldloca.s   V_1
    IL_0070:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0076:  ldarg.0
    IL_0077:  ldarg.0
    IL_0078:  ldarg.0
    IL_0079:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_007e:  dup
    IL_007f:  stloc.s    V_5
    IL_0081:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0086:  ldloc.s    V_5
    IL_0088:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_008d:  ldarg.0
    IL_008e:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0093:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0098:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_009d:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a2:  stloc.s    V_4
    IL_00a4:  ldloca.s   V_4
    IL_00a6:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_00ab:  brtrue.s   IL_00ee
    IL_00ad:  ldarg.0
    IL_00ae:  ldc.i4.1
    IL_00af:  dup
    IL_00b0:  stloc.0
    IL_00b1:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00b6:  ldarg.0
    IL_00b7:  ldloc.s    V_4
    IL_00b9:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00be:  ldarg.0
    IL_00bf:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00c4:  ldloca.s   V_4
    IL_00c6:  ldarg.0
    IL_00c7:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_00cc:  leave      IL_016a
    IL_00d1:  ldarg.0
    IL_00d2:  ldc.i4.m1
    IL_00d3:  dup
    IL_00d4:  stloc.0
    IL_00d5:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00da:  ldarg.0
    IL_00db:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00e0:  stloc.s    V_4
    IL_00e2:  ldarg.0
    IL_00e3:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00e8:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00ee:  ldarg.0
    IL_00ef:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_00f4:  box        "SM$T"
    IL_00f9:  ldloca.s   V_4
    IL_00fb:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0100:  ldloca.s   V_4
    IL_0102:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0108:  dup
    IL_0109:  stloc.3
    IL_010a:  ldarg.0
    IL_010b:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0110:  ldloc.3
    IL_0111:  constrained. "SM$T"
    IL_0117:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_011c:  ldc.i4.1
    IL_011d:  add.ovf
    IL_011e:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_0123:  ldarg.0
    IL_0124:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0129:  initobj    "SM$T"
    IL_012f:  leave.s    IL_0155
  }
  catch System.Exception
  {
    IL_0131:  dup
    IL_0132:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0137:  stloc.s    V_6
    IL_0139:  ldarg.0
    IL_013a:  ldc.i4.s   -2
    IL_013c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0141:  ldarg.0
    IL_0142:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0147:  ldloc.s    V_6
    IL_0149:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_014e:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0153:  leave.s    IL_016a
  }
  IL_0155:  ldarg.0
  IL_0156:  ldc.i4.s   -2
  IL_0158:  dup
  IL_0159:  stloc.0
  IL_015a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_015f:  ldarg.0
  IL_0160:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0165:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_016a:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      408 (0x198)
  .maxstack  4
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter V_1,
                System.Runtime.CompilerServices.YieldAwaitable V_2,
                Integer V_3,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_4,
                SM$T V_5,
                System.Exception V_6)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004b
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_00d7
    IL_0011:  call       "Function System.Threading.Tasks.Task.Yield() As System.Runtime.CompilerServices.YieldAwaitable"
    IL_0016:  stloc.2
    IL_0017:  ldloca.s   V_2
    IL_0019:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.GetAwaiter() As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_001e:  stloc.1
    IL_001f:  ldloca.s   V_1
    IL_0021:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.get_IsCompleted() As Boolean"
    IL_0026:  brtrue.s   IL_0067
    IL_0028:  ldarg.0
    IL_0029:  ldc.i4.0
    IL_002a:  dup
    IL_002b:  stloc.0
    IL_002c:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0031:  ldarg.0
    IL_0032:  ldloc.1
    IL_0033:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0038:  ldarg.0
    IL_0039:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003e:  ldloca.s   V_1
    IL_0040:  ldarg.0
    IL_0041:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_0046:  leave      IL_0197
    IL_004b:  ldarg.0
    IL_004c:  ldc.i4.m1
    IL_004d:  dup
    IL_004e:  stloc.0
    IL_004f:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0054:  ldarg.0
    IL_0055:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_005a:  stloc.1
    IL_005b:  ldarg.0
    IL_005c:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0061:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0067:  ldloca.s   V_1
    IL_0069:  call       "Sub System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.GetResult()"
    IL_006e:  ldloca.s   V_1
    IL_0070:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0076:  ldloca.s   V_5
    IL_0078:  initobj    "SM$T"
    IL_007e:  ldloc.s    V_5
    IL_0080:  box        "SM$T"
    IL_0085:  brtrue.s   IL_0093
    IL_0087:  ldarg.0
    IL_0088:  ldarg.0
    IL_0089:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_008e:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_0093:  ldarg.0
    IL_0094:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0099:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_009e:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_00a3:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a8:  stloc.s    V_4
    IL_00aa:  ldloca.s   V_4
    IL_00ac:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_00b1:  brtrue.s   IL_00f4
    IL_00b3:  ldarg.0
    IL_00b4:  ldc.i4.1
    IL_00b5:  dup
    IL_00b6:  stloc.0
    IL_00b7:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00bc:  ldarg.0
    IL_00bd:  ldloc.s    V_4
    IL_00bf:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00c4:  ldarg.0
    IL_00c5:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00ca:  ldloca.s   V_4
    IL_00cc:  ldarg.0
    IL_00cd:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_00d2:  leave      IL_0197
    IL_00d7:  ldarg.0
    IL_00d8:  ldc.i4.m1
    IL_00d9:  dup
    IL_00da:  stloc.0
    IL_00db:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00e0:  ldarg.0
    IL_00e1:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00e6:  stloc.s    V_4
    IL_00e8:  ldarg.0
    IL_00e9:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00ee:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00f4:  ldloca.s   V_5
    IL_00f6:  initobj    "SM$T"
    IL_00fc:  ldloc.s    V_5
    IL_00fe:  box        "SM$T"
    IL_0103:  brtrue.s   IL_010d
    IL_0105:  ldarg.0
    IL_0106:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_010b:  br.s       IL_0113
    IL_010d:  ldarg.0
    IL_010e:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0113:  ldloca.s   V_4
    IL_0115:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_011a:  ldloca.s   V_4
    IL_011c:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0122:  dup
    IL_0123:  stloc.3
    IL_0124:  ldloca.s   V_5
    IL_0126:  initobj    "SM$T"
    IL_012c:  ldloc.s    V_5
    IL_012e:  box        "SM$T"
    IL_0133:  brtrue.s   IL_013d
    IL_0135:  ldarg.0
    IL_0136:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_013b:  br.s       IL_0143
    IL_013d:  ldarg.0
    IL_013e:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0143:  ldloc.3
    IL_0144:  constrained. "SM$T"
    IL_014a:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_014f:  ldc.i4.1
    IL_0150:  add.ovf
    IL_0151:  constrained. "SM$T"
    IL_0157:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_015c:  leave.s    IL_0182
  }
  catch System.Exception
  {
    IL_015e:  dup
    IL_015f:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0164:  stloc.s    V_6
    IL_0166:  ldarg.0
    IL_0167:  ldc.i4.s   -2
    IL_0169:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_016e:  ldarg.0
    IL_016f:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0174:  ldloc.s    V_6
    IL_0176:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_017b:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0180:  leave.s    IL_0197
  }
  IL_0182:  ldarg.0
  IL_0183:  ldc.i4.s   -2
  IL_0185:  dup
  IL_0186:  stloc.0
  IL_0187:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_018c:  ldarg.0
  IL_018d:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0192:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0197:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Index_Async_02()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x as Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x as Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        await Task.Yield()
        item.Position(await GetOffsetAsync(GetOffset(item))) += 1
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        await Task.Yield()
        item.Position(await GetOffsetAsync(GetOffset(item))) += 1
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-1'
Position get for item '-2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      329 (0x149)
  .maxstack  4
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter V_1,
                System.Runtime.CompilerServices.YieldAwaitable V_2,
                Integer V_3,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004b
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_00ba
    IL_0011:  call       "Function System.Threading.Tasks.Task.Yield() As System.Runtime.CompilerServices.YieldAwaitable"
    IL_0016:  stloc.2
    IL_0017:  ldloca.s   V_2
    IL_0019:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.GetAwaiter() As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_001e:  stloc.1
    IL_001f:  ldloca.s   V_1
    IL_0021:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.get_IsCompleted() As Boolean"
    IL_0026:  brtrue.s   IL_0067
    IL_0028:  ldarg.0
    IL_0029:  ldc.i4.0
    IL_002a:  dup
    IL_002b:  stloc.0
    IL_002c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0031:  ldarg.0
    IL_0032:  ldloc.1
    IL_0033:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0038:  ldarg.0
    IL_0039:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003e:  ldloca.s   V_1
    IL_0040:  ldarg.0
    IL_0041:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0046:  leave      IL_0148
    IL_004b:  ldarg.0
    IL_004c:  ldc.i4.m1
    IL_004d:  dup
    IL_004e:  stloc.0
    IL_004f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0054:  ldarg.0
    IL_0055:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_005a:  stloc.1
    IL_005b:  ldarg.0
    IL_005c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0061:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0067:  ldloca.s   V_1
    IL_0069:  call       "Sub System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.GetResult()"
    IL_006e:  ldloca.s   V_1
    IL_0070:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0076:  ldarg.0
    IL_0077:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_007c:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0081:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0086:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008b:  stloc.s    V_4
    IL_008d:  ldloca.s   V_4
    IL_008f:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0094:  brtrue.s   IL_00d7
    IL_0096:  ldarg.0
    IL_0097:  ldc.i4.1
    IL_0098:  dup
    IL_0099:  stloc.0
    IL_009a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_009f:  ldarg.0
    IL_00a0:  ldloc.s    V_4
    IL_00a2:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a7:  ldarg.0
    IL_00a8:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00ad:  ldloca.s   V_4
    IL_00af:  ldarg.0
    IL_00b0:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_00b5:  leave      IL_0148
    IL_00ba:  ldarg.0
    IL_00bb:  ldc.i4.m1
    IL_00bc:  dup
    IL_00bd:  stloc.0
    IL_00be:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00c3:  ldarg.0
    IL_00c4:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00c9:  stloc.s    V_4
    IL_00cb:  ldarg.0
    IL_00cc:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d1:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d7:  ldarg.0
    IL_00d8:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_00dd:  ldloca.s   V_4
    IL_00df:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00e4:  ldloca.s   V_4
    IL_00e6:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00ec:  dup
    IL_00ed:  stloc.3
    IL_00ee:  ldarg.0
    IL_00ef:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_00f4:  ldloc.3
    IL_00f5:  constrained. "SM$T"
    IL_00fb:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_0100:  ldc.i4.1
    IL_0101:  add.ovf
    IL_0102:  constrained. "SM$T"
    IL_0108:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_010d:  leave.s    IL_0133
  }
  catch System.Exception
  {
    IL_010f:  dup
    IL_0110:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0115:  stloc.s    V_5
    IL_0117:  ldarg.0
    IL_0118:  ldc.i4.s   -2
    IL_011a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_011f:  ldarg.0
    IL_0120:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0125:  ldloc.s    V_5
    IL_0127:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_012c:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0131:  leave.s    IL_0148
  }
  IL_0133:  ldarg.0
  IL_0134:  ldc.i4.s   -2
  IL_0136:  dup
  IL_0137:  stloc.0
  IL_0138:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_013d:  ldarg.0
  IL_013e:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0143:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0148:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Value()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(item As T)
        item.Position(1) += GetOffset(item)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        item.Position(1) += GetOffset(item)
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       37 (0x25)
  .maxstack  4
  .locals init (T V_0)
  IL_0000:  ldarg.0
  IL_0001:  dup
  IL_0002:  stloc.0
  IL_0003:  box        "T"
  IL_0008:  ldc.i4.1
  IL_0009:  ldloca.s   V_0
  IL_000b:  ldc.i4.1
  IL_000c:  constrained. "T"
  IL_0012:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0017:  ldarga.s   V_0
  IL_0019:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_001e:  add.ovf
  IL_001f:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0024:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
  // Code size       61 (0x3d)
  .maxstack  4
  .locals init (T V_0)
  IL_0000:  ldarga.s   V_0
  IL_0002:  ldloca.s   V_0
  IL_0004:  initobj    "T"
  IL_000a:  ldloc.0
  IL_000b:  box        "T"
  IL_0010:  brtrue.s   IL_001a
  IL_0012:  ldobj      "T"
  IL_0017:  stloc.0
  IL_0018:  ldloca.s   V_0
  IL_001a:  ldc.i4.1
  IL_001b:  ldarga.s   V_0
  IL_001d:  ldc.i4.1
  IL_001e:  constrained. "T"
  IL_0024:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0029:  ldarga.s   V_0
  IL_002b:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0030:  add.ovf
  IL_0031:  constrained. "T"
  IL_0037:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_003c:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Value()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(item As T)
        item.Position(1) += GetOffset(item)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        item.Position(1) += GetOffset(item)
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '-1'
Position get for item '2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       37 (0x25)
  .maxstack  4
  IL_0000:  ldarga.s   V_0
  IL_0002:  ldc.i4.1
  IL_0003:  ldarga.s   V_0
  IL_0005:  ldc.i4.1
  IL_0006:  constrained. "T"
  IL_000c:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0011:  ldarga.s   V_0
  IL_0013:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0018:  add.ovf
  IL_0019:  constrained. "T"
  IL_001f:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0024:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Value_Ref()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(ByRef item As T)
        item.Position(1) += GetOffset(item)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        item.Position(1) += GetOffset(item)
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       41 (0x29)
  .maxstack  4
  .locals init (T V_0)
  IL_0000:  ldarg.0
  IL_0001:  ldobj      "T"
  IL_0006:  dup
  IL_0007:  stloc.0
  IL_0008:  box        "T"
  IL_000d:  ldc.i4.1
  IL_000e:  ldloca.s   V_0
  IL_0010:  ldc.i4.1
  IL_0011:  constrained. "T"
  IL_0017:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_001c:  ldarg.0
  IL_001d:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0022:  add.ovf
  IL_0023:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0028:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
  // Code size       58 (0x3a)
  .maxstack  4
  .locals init (T V_0)
  IL_0000:  ldarg.0
  IL_0001:  ldloca.s   V_0
  IL_0003:  initobj    "T"
  IL_0009:  ldloc.0
  IL_000a:  box        "T"
  IL_000f:  brtrue.s   IL_0019
  IL_0011:  ldobj      "T"
  IL_0016:  stloc.0
  IL_0017:  ldloca.s   V_0
  IL_0019:  ldc.i4.1
  IL_001a:  ldarg.0
  IL_001b:  ldc.i4.1
  IL_001c:  constrained. "T"
  IL_0022:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0027:  ldarg.0
  IL_0028:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_002d:  add.ovf
  IL_002e:  constrained. "T"
  IL_0034:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0039:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Value_Ref()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(ByRef item As T)
        item.Position(1) += GetOffset(item)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        item.Position(1) += GetOffset(item)
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '-1'
Position get for item '2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       34 (0x22)
  .maxstack  4
  IL_0000:  ldarg.0
  IL_0001:  ldc.i4.1
  IL_0002:  ldarg.0
  IL_0003:  ldc.i4.1
  IL_0004:  constrained. "T"
  IL_000a:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_000f:  ldarg.0
  IL_0010:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0015:  add.ovf
  IL_0016:  constrained. "T"
  IL_001c:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0021:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Value_Async_01()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        item.Position(1) += await GetOffsetAsync(GetOffset(item))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        item.Position(1) += await GetOffsetAsync(GetOffset(item))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      258 (0x102)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                SM$T V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0079
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldarg.0
    IL_000d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0012:  dup
    IL_0013:  stloc.2
    IL_0014:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0019:  ldloc.2
    IL_001a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_001f:  ldarg.0
    IL_0020:  ldarg.0
    IL_0021:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0026:  ldc.i4.1
    IL_0027:  constrained. "SM$T"
    IL_002d:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_0032:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_0037:  ldarg.0
    IL_0038:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_003d:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0042:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0047:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_004c:  stloc.1
    IL_004d:  ldloca.s   V_1
    IL_004f:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0054:  brtrue.s   IL_0095
    IL_0056:  ldarg.0
    IL_0057:  ldc.i4.0
    IL_0058:  dup
    IL_0059:  stloc.0
    IL_005a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_005f:  ldarg.0
    IL_0060:  ldloc.1
    IL_0061:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0066:  ldarg.0
    IL_0067:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_006c:  ldloca.s   V_1
    IL_006e:  ldarg.0
    IL_006f:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0074:  leave      IL_0101
    IL_0079:  ldarg.0
    IL_007a:  ldc.i4.m1
    IL_007b:  dup
    IL_007c:  stloc.0
    IL_007d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0082:  ldarg.0
    IL_0083:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0088:  stloc.1
    IL_0089:  ldarg.0
    IL_008a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008f:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0095:  ldarg.0
    IL_0096:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_009b:  box        "SM$T"
    IL_00a0:  ldc.i4.1
    IL_00a1:  ldarg.0
    IL_00a2:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_00a7:  ldloca.s   V_1
    IL_00a9:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00ae:  ldloca.s   V_1
    IL_00b0:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b6:  add.ovf
    IL_00b7:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00bc:  ldarg.0
    IL_00bd:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_00c2:  initobj    "SM$T"
    IL_00c8:  leave.s    IL_00ec
  }
  catch System.Exception
  {
    IL_00ca:  dup
    IL_00cb:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00d0:  stloc.3
    IL_00d1:  ldarg.0
    IL_00d2:  ldc.i4.s   -2
    IL_00d4:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00d9:  ldarg.0
    IL_00da:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00df:  ldloc.3
    IL_00e0:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00e5:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00ea:  leave.s    IL_0101
  }
  IL_00ec:  ldarg.0
  IL_00ed:  ldc.i4.s   -2
  IL_00ef:  dup
  IL_00f0:  stloc.0
  IL_00f1:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00f6:  ldarg.0
  IL_00f7:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00fc:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0101:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      290 (0x122)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                SM$T V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0080
    IL_000a:  ldloca.s   V_2
    IL_000c:  initobj    "SM$T"
    IL_0012:  ldloc.2
    IL_0013:  box        "SM$T"
    IL_0018:  brtrue.s   IL_0026
    IL_001a:  ldarg.0
    IL_001b:  ldarg.0
    IL_001c:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0021:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T"
    IL_0026:  ldarg.0
    IL_0027:  ldarg.0
    IL_0028:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_002d:  ldc.i4.1
    IL_002e:  constrained. "SM$T"
    IL_0034:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_0039:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As Integer"
    IL_003e:  ldarg.0
    IL_003f:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0044:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0049:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_004e:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0053:  stloc.1
    IL_0054:  ldloca.s   V_1
    IL_0056:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_005b:  brtrue.s   IL_009c
    IL_005d:  ldarg.0
    IL_005e:  ldc.i4.0
    IL_005f:  dup
    IL_0060:  stloc.0
    IL_0061:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0066:  ldarg.0
    IL_0067:  ldloc.1
    IL_0068:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_006d:  ldarg.0
    IL_006e:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0073:  ldloca.s   V_1
    IL_0075:  ldarg.0
    IL_0076:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_007b:  leave      IL_0121
    IL_0080:  ldarg.0
    IL_0081:  ldc.i4.m1
    IL_0082:  dup
    IL_0083:  stloc.0
    IL_0084:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0089:  ldarg.0
    IL_008a:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008f:  stloc.1
    IL_0090:  ldarg.0
    IL_0091:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0096:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009c:  ldloca.s   V_2
    IL_009e:  initobj    "SM$T"
    IL_00a4:  ldloc.2
    IL_00a5:  box        "SM$T"
    IL_00aa:  brtrue.s   IL_00b4
    IL_00ac:  ldarg.0
    IL_00ad:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T"
    IL_00b2:  br.s       IL_00ba
    IL_00b4:  ldarg.0
    IL_00b5:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00ba:  ldc.i4.1
    IL_00bb:  ldarg.0
    IL_00bc:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As Integer"
    IL_00c1:  ldloca.s   V_1
    IL_00c3:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00c8:  ldloca.s   V_1
    IL_00ca:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d0:  add.ovf
    IL_00d1:  constrained. "SM$T"
    IL_00d7:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00dc:  ldarg.0
    IL_00dd:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T"
    IL_00e2:  initobj    "SM$T"
    IL_00e8:  leave.s    IL_010c
  }
  catch System.Exception
  {
    IL_00ea:  dup
    IL_00eb:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00f0:  stloc.3
    IL_00f1:  ldarg.0
    IL_00f2:  ldc.i4.s   -2
    IL_00f4:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00f9:  ldarg.0
    IL_00fa:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00ff:  ldloc.3
    IL_0100:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0105:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_010a:  leave.s    IL_0121
  }
  IL_010c:  ldarg.0
  IL_010d:  ldc.i4.s   -2
  IL_010f:  dup
  IL_0110:  stloc.0
  IL_0111:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0116:  ldarg.0
  IL_0117:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_011c:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0121:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Value_Async_01()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        item.Position(1) += await GetOffsetAsync(GetOffset(item))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        item.Position(1) += await GetOffsetAsync(GetOffset(item))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '-1'
Position get for item '2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      223 (0xdf)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                System.Exception V_2)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0061
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0011:  ldc.i4.1
    IL_0012:  constrained. "SM$T"
    IL_0018:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_001d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_0022:  ldarg.0
    IL_0023:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0028:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_002d:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0032:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0037:  stloc.1
    IL_0038:  ldloca.s   V_1
    IL_003a:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_003f:  brtrue.s   IL_007d
    IL_0041:  ldarg.0
    IL_0042:  ldc.i4.0
    IL_0043:  dup
    IL_0044:  stloc.0
    IL_0045:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_004a:  ldarg.0
    IL_004b:  ldloc.1
    IL_004c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0051:  ldarg.0
    IL_0052:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0057:  ldloca.s   V_1
    IL_0059:  ldarg.0
    IL_005a:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_005f:  leave.s    IL_00de
    IL_0061:  ldarg.0
    IL_0062:  ldc.i4.m1
    IL_0063:  dup
    IL_0064:  stloc.0
    IL_0065:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_006a:  ldarg.0
    IL_006b:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0070:  stloc.1
    IL_0071:  ldarg.0
    IL_0072:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0077:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007d:  ldarg.0
    IL_007e:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0083:  ldc.i4.1
    IL_0084:  ldarg.0
    IL_0085:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_008a:  ldloca.s   V_1
    IL_008c:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0091:  ldloca.s   V_1
    IL_0093:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0099:  add.ovf
    IL_009a:  constrained. "SM$T"
    IL_00a0:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00a5:  leave.s    IL_00c9
  }
  catch System.Exception
  {
    IL_00a7:  dup
    IL_00a8:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00ad:  stloc.2
    IL_00ae:  ldarg.0
    IL_00af:  ldc.i4.s   -2
    IL_00b1:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00b6:  ldarg.0
    IL_00b7:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00bc:  ldloc.2
    IL_00bd:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00c2:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00c7:  leave.s    IL_00de
  }
  IL_00c9:  ldarg.0
  IL_00ca:  ldc.i4.s   -2
  IL_00cc:  dup
  IL_00cd:  stloc.0
  IL_00ce:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00d3:  ldarg.0
  IL_00d4:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00d9:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_00de:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_IndexAndValue()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(item As T)
        item.Position(GetOffset(item)) += GetOffset(item)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        item.Position(GetOffset(item)) += GetOffset(item)
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       45 (0x2d)
  .maxstack  4
  .locals init (T V_0,
                Integer V_1)
  IL_0000:  ldarg.0
  IL_0001:  dup
  IL_0002:  stloc.0
  IL_0003:  box        "T"
  IL_0008:  ldarga.s   V_0
  IL_000a:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_000f:  dup
  IL_0010:  stloc.1
  IL_0011:  ldloca.s   V_0
  IL_0013:  ldloc.1
  IL_0014:  constrained. "T"
  IL_001a:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_001f:  ldarga.s   V_0
  IL_0021:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0026:  add.ovf
  IL_0027:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_002c:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
  // Code size       87 (0x57)
  .maxstack  4
  .locals init (T V_0,
                Integer V_1,
                T V_2)
  IL_0000:  ldloca.s   V_2
  IL_0002:  initobj    "T"
  IL_0008:  ldloc.2
  IL_0009:  box        "T"
  IL_000e:  brtrue.s   IL_0016
  IL_0010:  ldarg.0
  IL_0011:  stloc.0
  IL_0012:  ldloca.s   V_0
  IL_0014:  br.s       IL_0018
  IL_0016:  ldarga.s   V_0
  IL_0018:  ldarga.s   V_0
  IL_001a:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_001f:  dup
  IL_0020:  stloc.1
  IL_0021:  ldloca.s   V_2
  IL_0023:  initobj    "T"
  IL_0029:  ldloc.2
  IL_002a:  box        "T"
  IL_002f:  brtrue.s   IL_0035
  IL_0031:  ldloca.s   V_0
  IL_0033:  br.s       IL_0037
  IL_0035:  ldarga.s   V_0
  IL_0037:  ldloc.1
  IL_0038:  constrained. "T"
  IL_003e:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0043:  ldarga.s   V_0
  IL_0045:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_004a:  add.ovf
  IL_004b:  constrained. "T"
  IL_0051:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0056:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_IndexAndValue()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(item As T)
        item.Position(GetOffset(item)) += GetOffset(item)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        item.Position(GetOffset(item)) += GetOffset(item)
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-2'
Position get for item '-3'
Position set for item '-4'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       45 (0x2d)
  .maxstack  4
  .locals init (Integer V_0)
  IL_0000:  ldarga.s   V_0
  IL_0002:  ldarga.s   V_0
  IL_0004:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0009:  dup
  IL_000a:  stloc.0
  IL_000b:  ldarga.s   V_0
  IL_000d:  ldloc.0
  IL_000e:  constrained. "T"
  IL_0014:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0019:  ldarga.s   V_0
  IL_001b:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0020:  add.ovf
  IL_0021:  constrained. "T"
  IL_0027:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_002c:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_IndexAndValue_Ref()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(ByRef item As T)
        item.Position(GetOffset(item)) += GetOffset(item)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        item.Position(GetOffset(item)) += GetOffset(item)
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       48 (0x30)
  .maxstack  4
  .locals init (T V_0,
                Integer V_1)
  IL_0000:  ldarg.0
  IL_0001:  ldobj      "T"
  IL_0006:  dup
  IL_0007:  stloc.0
  IL_0008:  box        "T"
  IL_000d:  ldarg.0
  IL_000e:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0013:  dup
  IL_0014:  stloc.1
  IL_0015:  ldloca.s   V_0
  IL_0017:  ldloc.1
  IL_0018:  constrained. "T"
  IL_001e:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0023:  ldarg.0
  IL_0024:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0029:  add.ovf
  IL_002a:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_002f:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
  // Code size       88 (0x58)
  .maxstack  4
  .locals init (T V_0,
                Integer V_1,
                T V_2)
  IL_0000:  ldloca.s   V_2
  IL_0002:  initobj    "T"
  IL_0008:  ldloc.2
  IL_0009:  box        "T"
  IL_000e:  brtrue.s   IL_001b
  IL_0010:  ldarg.0
  IL_0011:  ldobj      "T"
  IL_0016:  stloc.0
  IL_0017:  ldloca.s   V_0
  IL_0019:  br.s       IL_001c
  IL_001b:  ldarg.0
  IL_001c:  ldarg.0
  IL_001d:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0022:  dup
  IL_0023:  stloc.1
  IL_0024:  ldloca.s   V_2
  IL_0026:  initobj    "T"
  IL_002c:  ldloc.2
  IL_002d:  box        "T"
  IL_0032:  brtrue.s   IL_0038
  IL_0034:  ldloca.s   V_0
  IL_0036:  br.s       IL_0039
  IL_0038:  ldarg.0
  IL_0039:  ldloc.1
  IL_003a:  constrained. "T"
  IL_0040:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0045:  ldarg.0
  IL_0046:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_004b:  add.ovf
  IL_004c:  constrained. "T"
  IL_0052:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0057:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_IndexAndValue_Ref()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(ByRef item As T)
        item.Position(GetOffset(item)) += GetOffset(item)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        item.Position(GetOffset(item)) += GetOffset(item)
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-2'
Position get for item '-3'
Position set for item '-4'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       41 (0x29)
  .maxstack  4
  .locals init (Integer V_0)
  IL_0000:  ldarg.0
  IL_0001:  ldarg.0
  IL_0002:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0007:  dup
  IL_0008:  stloc.0
  IL_0009:  ldarg.0
  IL_000a:  ldloc.0
  IL_000b:  constrained. "T"
  IL_0011:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0016:  ldarg.0
  IL_0017:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_001c:  add.ovf
  IL_001d:  constrained. "T"
  IL_0023:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0028:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_IndexAndValue_Async_01()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        item.Position(GetOffset(item)) += await GetOffsetAsync(GetOffset(item))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        item.Position(GetOffset(item)) += await GetOffsetAsync(GetOffset(item))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      298 (0x12a)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                SM$T V_2,
                Integer V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse    IL_009a
    IL_000d:  ldarg.0
    IL_000e:  ldarg.0
    IL_000f:  ldarg.0
    IL_0010:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0015:  dup
    IL_0016:  stloc.2
    IL_0017:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_001c:  ldloc.2
    IL_001d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_0022:  ldarg.0
    IL_0023:  ldarg.0
    IL_0024:  ldarg.0
    IL_0025:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_002a:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_002f:  dup
    IL_0030:  stloc.3
    IL_0031:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S1 As Integer"
    IL_0036:  ldloc.3
    IL_0037:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U3 As Integer"
    IL_003c:  ldarg.0
    IL_003d:  ldarg.0
    IL_003e:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0043:  box        "SM$T"
    IL_0048:  ldarg.0
    IL_0049:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S1 As Integer"
    IL_004e:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_0053:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_0058:  ldarg.0
    IL_0059:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_005e:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0063:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0068:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_006d:  stloc.1
    IL_006e:  ldloca.s   V_1
    IL_0070:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0075:  brtrue.s   IL_00b6
    IL_0077:  ldarg.0
    IL_0078:  ldc.i4.0
    IL_0079:  dup
    IL_007a:  stloc.0
    IL_007b:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0080:  ldarg.0
    IL_0081:  ldloc.1
    IL_0082:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0087:  ldarg.0
    IL_0088:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_008d:  ldloca.s   V_1
    IL_008f:  ldarg.0
    IL_0090:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0095:  leave      IL_0129
    IL_009a:  ldarg.0
    IL_009b:  ldc.i4.m1
    IL_009c:  dup
    IL_009d:  stloc.0
    IL_009e:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00a3:  ldarg.0
    IL_00a4:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a9:  stloc.1
    IL_00aa:  ldarg.0
    IL_00ab:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b0:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b6:  ldarg.0
    IL_00b7:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_00bc:  box        "SM$T"
    IL_00c1:  ldarg.0
    IL_00c2:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U3 As Integer"
    IL_00c7:  ldarg.0
    IL_00c8:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_00cd:  ldloca.s   V_1
    IL_00cf:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00d4:  ldloca.s   V_1
    IL_00d6:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00dc:  add.ovf
    IL_00dd:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00e2:  ldarg.0
    IL_00e3:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_00e8:  initobj    "SM$T"
    IL_00ee:  leave.s    IL_0114
  }
  catch System.Exception
  {
    IL_00f0:  dup
    IL_00f1:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00f6:  stloc.s    V_4
    IL_00f8:  ldarg.0
    IL_00f9:  ldc.i4.s   -2
    IL_00fb:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0100:  ldarg.0
    IL_0101:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0106:  ldloc.s    V_4
    IL_0108:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_010d:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0112:  leave.s    IL_0129
  }
  IL_0114:  ldarg.0
  IL_0115:  ldc.i4.s   -2
  IL_0117:  dup
  IL_0118:  stloc.0
  IL_0119:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_011e:  ldarg.0
  IL_011f:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0124:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0129:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      343 (0x157)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                SM$T V_2,
                Integer V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse    IL_00ba
    IL_000d:  ldloca.s   V_2
    IL_000f:  initobj    "SM$T"
    IL_0015:  ldloc.2
    IL_0016:  box        "SM$T"
    IL_001b:  brtrue.s   IL_0029
    IL_001d:  ldarg.0
    IL_001e:  ldarg.0
    IL_001f:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0024:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_0029:  ldarg.0
    IL_002a:  ldarg.0
    IL_002b:  ldarg.0
    IL_002c:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0031:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0036:  dup
    IL_0037:  stloc.3
    IL_0038:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S1 As Integer"
    IL_003d:  ldloc.3
    IL_003e:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As Integer"
    IL_0043:  ldarg.0
    IL_0044:  ldloca.s   V_2
    IL_0046:  initobj    "SM$T"
    IL_004c:  ldloc.2
    IL_004d:  box        "SM$T"
    IL_0052:  brtrue.s   IL_005c
    IL_0054:  ldarg.0
    IL_0055:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_005a:  br.s       IL_0062
    IL_005c:  ldarg.0
    IL_005d:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0062:  ldarg.0
    IL_0063:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S1 As Integer"
    IL_0068:  constrained. "SM$T"
    IL_006e:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_0073:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As Integer"
    IL_0078:  ldarg.0
    IL_0079:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_007e:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0083:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0088:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008d:  stloc.1
    IL_008e:  ldloca.s   V_1
    IL_0090:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0095:  brtrue.s   IL_00d6
    IL_0097:  ldarg.0
    IL_0098:  ldc.i4.0
    IL_0099:  dup
    IL_009a:  stloc.0
    IL_009b:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00a0:  ldarg.0
    IL_00a1:  ldloc.1
    IL_00a2:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a7:  ldarg.0
    IL_00a8:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00ad:  ldloca.s   V_1
    IL_00af:  ldarg.0
    IL_00b0:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_00b5:  leave      IL_0156
    IL_00ba:  ldarg.0
    IL_00bb:  ldc.i4.m1
    IL_00bc:  dup
    IL_00bd:  stloc.0
    IL_00be:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00c3:  ldarg.0
    IL_00c4:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00c9:  stloc.1
    IL_00ca:  ldarg.0
    IL_00cb:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d0:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d6:  ldloca.s   V_2
    IL_00d8:  initobj    "SM$T"
    IL_00de:  ldloc.2
    IL_00df:  box        "SM$T"
    IL_00e4:  brtrue.s   IL_00ee
    IL_00e6:  ldarg.0
    IL_00e7:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_00ec:  br.s       IL_00f4
    IL_00ee:  ldarg.0
    IL_00ef:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00f4:  ldarg.0
    IL_00f5:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As Integer"
    IL_00fa:  ldarg.0
    IL_00fb:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As Integer"
    IL_0100:  ldloca.s   V_1
    IL_0102:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0107:  ldloca.s   V_1
    IL_0109:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_010f:  add.ovf
    IL_0110:  constrained. "SM$T"
    IL_0116:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_011b:  leave.s    IL_0141
  }
  catch System.Exception
  {
    IL_011d:  dup
    IL_011e:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0123:  stloc.s    V_4
    IL_0125:  ldarg.0
    IL_0126:  ldc.i4.s   -2
    IL_0128:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_012d:  ldarg.0
    IL_012e:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0133:  ldloc.s    V_4
    IL_0135:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_013a:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_013f:  leave.s    IL_0156
  }
  IL_0141:  ldarg.0
  IL_0142:  ldc.i4.s   -2
  IL_0144:  dup
  IL_0145:  stloc.0
  IL_0146:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_014b:  ldarg.0
  IL_014c:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0151:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0156:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_IndexAndValue_Async_01()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        item.Position(GetOffset(item)) += await GetOffsetAsync(GetOffset(item))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        item.Position(GetOffset(item)) += await GetOffsetAsync(GetOffset(item))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-2'
Position get for item '-3'
Position set for item '-4'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      262 (0x106)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                Integer V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0083
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldarg.0
    IL_000d:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0012:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0017:  dup
    IL_0018:  stloc.2
    IL_0019:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As Integer"
    IL_001e:  ldloc.2
    IL_001f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As Integer"
    IL_0024:  ldarg.0
    IL_0025:  ldarg.0
    IL_0026:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_002b:  ldarg.0
    IL_002c:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As Integer"
    IL_0031:  constrained. "SM$T"
    IL_0037:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_003c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_0041:  ldarg.0
    IL_0042:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0047:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_004c:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0051:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0056:  stloc.1
    IL_0057:  ldloca.s   V_1
    IL_0059:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_005e:  brtrue.s   IL_009f
    IL_0060:  ldarg.0
    IL_0061:  ldc.i4.0
    IL_0062:  dup
    IL_0063:  stloc.0
    IL_0064:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0069:  ldarg.0
    IL_006a:  ldloc.1
    IL_006b:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0070:  ldarg.0
    IL_0071:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0076:  ldloca.s   V_1
    IL_0078:  ldarg.0
    IL_0079:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_007e:  leave      IL_0105
    IL_0083:  ldarg.0
    IL_0084:  ldc.i4.m1
    IL_0085:  dup
    IL_0086:  stloc.0
    IL_0087:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_008c:  ldarg.0
    IL_008d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0092:  stloc.1
    IL_0093:  ldarg.0
    IL_0094:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0099:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009f:  ldarg.0
    IL_00a0:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_00a5:  ldarg.0
    IL_00a6:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As Integer"
    IL_00ab:  ldarg.0
    IL_00ac:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_00b1:  ldloca.s   V_1
    IL_00b3:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00b8:  ldloca.s   V_1
    IL_00ba:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00c0:  add.ovf
    IL_00c1:  constrained. "SM$T"
    IL_00c7:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00cc:  leave.s    IL_00f0
  }
  catch System.Exception
  {
    IL_00ce:  dup
    IL_00cf:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00d4:  stloc.3
    IL_00d5:  ldarg.0
    IL_00d6:  ldc.i4.s   -2
    IL_00d8:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00dd:  ldarg.0
    IL_00de:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00e3:  ldloc.3
    IL_00e4:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00e9:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00ee:  leave.s    IL_0105
  }
  IL_00f0:  ldarg.0
  IL_00f1:  ldc.i4.s   -2
  IL_00f3:  dup
  IL_00f4:  stloc.0
  IL_00f5:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00fa:  ldarg.0
  IL_00fb:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0100:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0105:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_IndexAndValue_Async_03()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        item.Position(await GetOffsetAsync(GetOffset(item))) += GetOffset(item)
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        item.Position(await GetOffsetAsync(GetOffset(item))) += GetOffset(item)
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      260 (0x104)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0061
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldarg.0
    IL_000d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0012:  dup
    IL_0013:  stloc.3
    IL_0014:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0019:  ldloc.3
    IL_001a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_001f:  ldarg.0
    IL_0020:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0025:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_002a:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_002f:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0034:  stloc.2
    IL_0035:  ldloca.s   V_2
    IL_0037:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_003c:  brtrue.s   IL_007d
    IL_003e:  ldarg.0
    IL_003f:  ldc.i4.0
    IL_0040:  dup
    IL_0041:  stloc.0
    IL_0042:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0047:  ldarg.0
    IL_0048:  ldloc.2
    IL_0049:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_004e:  ldarg.0
    IL_004f:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0054:  ldloca.s   V_2
    IL_0056:  ldarg.0
    IL_0057:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_005c:  leave      IL_0103
    IL_0061:  ldarg.0
    IL_0062:  ldc.i4.m1
    IL_0063:  dup
    IL_0064:  stloc.0
    IL_0065:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_006a:  ldarg.0
    IL_006b:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0070:  stloc.2
    IL_0071:  ldarg.0
    IL_0072:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0077:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007d:  ldarg.0
    IL_007e:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0083:  box        "SM$T"
    IL_0088:  ldloca.s   V_2
    IL_008a:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_008f:  ldloca.s   V_2
    IL_0091:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0097:  dup
    IL_0098:  stloc.1
    IL_0099:  ldarg.0
    IL_009a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_009f:  ldloc.1
    IL_00a0:  constrained. "SM$T"
    IL_00a6:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_00ab:  ldarg.0
    IL_00ac:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_00b1:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_00b6:  add.ovf
    IL_00b7:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00bc:  ldarg.0
    IL_00bd:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_00c2:  initobj    "SM$T"
    IL_00c8:  leave.s    IL_00ee
  }
  catch System.Exception
  {
    IL_00ca:  dup
    IL_00cb:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00d0:  stloc.s    V_4
    IL_00d2:  ldarg.0
    IL_00d3:  ldc.i4.s   -2
    IL_00d5:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00da:  ldarg.0
    IL_00db:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00e0:  ldloc.s    V_4
    IL_00e2:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00e7:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00ec:  leave.s    IL_0103
  }
  IL_00ee:  ldarg.0
  IL_00ef:  ldc.i4.s   -2
  IL_00f1:  dup
  IL_00f2:  stloc.0
  IL_00f3:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00f8:  ldarg.0
  IL_00f9:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00fe:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0103:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      304 (0x130)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0068
    IL_000a:  ldloca.s   V_3
    IL_000c:  initobj    "SM$T"
    IL_0012:  ldloc.3
    IL_0013:  box        "SM$T"
    IL_0018:  brtrue.s   IL_0026
    IL_001a:  ldarg.0
    IL_001b:  ldarg.0
    IL_001c:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0021:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_0026:  ldarg.0
    IL_0027:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_002c:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0031:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0036:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_003b:  stloc.2
    IL_003c:  ldloca.s   V_2
    IL_003e:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0043:  brtrue.s   IL_0084
    IL_0045:  ldarg.0
    IL_0046:  ldc.i4.0
    IL_0047:  dup
    IL_0048:  stloc.0
    IL_0049:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_004e:  ldarg.0
    IL_004f:  ldloc.2
    IL_0050:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0055:  ldarg.0
    IL_0056:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_005b:  ldloca.s   V_2
    IL_005d:  ldarg.0
    IL_005e:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_0063:  leave      IL_012f
    IL_0068:  ldarg.0
    IL_0069:  ldc.i4.m1
    IL_006a:  dup
    IL_006b:  stloc.0
    IL_006c:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0071:  ldarg.0
    IL_0072:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0077:  stloc.2
    IL_0078:  ldarg.0
    IL_0079:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007e:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0084:  ldloca.s   V_3
    IL_0086:  initobj    "SM$T"
    IL_008c:  ldloc.3
    IL_008d:  box        "SM$T"
    IL_0092:  brtrue.s   IL_009c
    IL_0094:  ldarg.0
    IL_0095:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_009a:  br.s       IL_00a2
    IL_009c:  ldarg.0
    IL_009d:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00a2:  ldloca.s   V_2
    IL_00a4:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00a9:  ldloca.s   V_2
    IL_00ab:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b1:  dup
    IL_00b2:  stloc.1
    IL_00b3:  ldloca.s   V_3
    IL_00b5:  initobj    "SM$T"
    IL_00bb:  ldloc.3
    IL_00bc:  box        "SM$T"
    IL_00c1:  brtrue.s   IL_00cb
    IL_00c3:  ldarg.0
    IL_00c4:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_00c9:  br.s       IL_00d1
    IL_00cb:  ldarg.0
    IL_00cc:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00d1:  ldloc.1
    IL_00d2:  constrained. "SM$T"
    IL_00d8:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_00dd:  ldarg.0
    IL_00de:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00e3:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_00e8:  add.ovf
    IL_00e9:  constrained. "SM$T"
    IL_00ef:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00f4:  leave.s    IL_011a
  }
  catch System.Exception
  {
    IL_00f6:  dup
    IL_00f7:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00fc:  stloc.s    V_4
    IL_00fe:  ldarg.0
    IL_00ff:  ldc.i4.s   -2
    IL_0101:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0106:  ldarg.0
    IL_0107:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_010c:  ldloc.s    V_4
    IL_010e:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0113:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0118:  leave.s    IL_012f
  }
  IL_011a:  ldarg.0
  IL_011b:  ldc.i4.s   -2
  IL_011d:  dup
  IL_011e:  stloc.0
  IL_011f:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0124:  ldarg.0
  IL_0125:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_012a:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_012f:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_IndexAndValue_Async_03()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        item.Position(await GetOffsetAsync(GetOffset(item))) += GetOffset(item)
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        item.Position(await GetOffsetAsync(GetOffset(item))) += GetOffset(item)
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-2'
Position get for item '-3'
Position set for item '-4'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      226 (0xe2)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004c
    IL_000a:  ldarg.0
    IL_000b:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0010:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0015:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_001a:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_001f:  stloc.2
    IL_0020:  ldloca.s   V_2
    IL_0022:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0027:  brtrue.s   IL_0068
    IL_0029:  ldarg.0
    IL_002a:  ldc.i4.0
    IL_002b:  dup
    IL_002c:  stloc.0
    IL_002d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0032:  ldarg.0
    IL_0033:  ldloc.2
    IL_0034:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0039:  ldarg.0
    IL_003a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003f:  ldloca.s   V_2
    IL_0041:  ldarg.0
    IL_0042:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0047:  leave      IL_00e1
    IL_004c:  ldarg.0
    IL_004d:  ldc.i4.m1
    IL_004e:  dup
    IL_004f:  stloc.0
    IL_0050:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0055:  ldarg.0
    IL_0056:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_005b:  stloc.2
    IL_005c:  ldarg.0
    IL_005d:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0062:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0068:  ldarg.0
    IL_0069:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_006e:  ldloca.s   V_2
    IL_0070:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0075:  ldloca.s   V_2
    IL_0077:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007d:  dup
    IL_007e:  stloc.1
    IL_007f:  ldarg.0
    IL_0080:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0085:  ldloc.1
    IL_0086:  constrained. "SM$T"
    IL_008c:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_0091:  ldarg.0
    IL_0092:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0097:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_009c:  add.ovf
    IL_009d:  constrained. "SM$T"
    IL_00a3:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00a8:  leave.s    IL_00cc
  }
  catch System.Exception
  {
    IL_00aa:  dup
    IL_00ab:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00b0:  stloc.3
    IL_00b1:  ldarg.0
    IL_00b2:  ldc.i4.s   -2
    IL_00b4:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00b9:  ldarg.0
    IL_00ba:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00bf:  ldloc.3
    IL_00c0:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00c5:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00ca:  leave.s    IL_00e1
  }
  IL_00cc:  ldarg.0
  IL_00cd:  ldc.i4.s   -2
  IL_00cf:  dup
  IL_00d0:  stloc.0
  IL_00d1:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00d6:  ldarg.0
  IL_00d7:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00dc:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_00e1:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_IndexAndValue_Async_04()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        item.Position(await GetOffsetAsync(GetOffset(item))) += await GetOffsetAsync(GetOffset(item))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        item.Position(await GetOffsetAsync(GetOffset(item))) += await GetOffsetAsync(GetOffset(item))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      402 (0x192)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T V_3,
                Integer V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0068
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_0102
    IL_0011:  ldarg.0
    IL_0012:  ldarg.0
    IL_0013:  ldarg.0
    IL_0014:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0019:  dup
    IL_001a:  stloc.3
    IL_001b:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0020:  ldloc.3
    IL_0021:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_0026:  ldarg.0
    IL_0027:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_002c:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0031:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0036:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_003b:  stloc.1
    IL_003c:  ldloca.s   V_1
    IL_003e:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0043:  brtrue.s   IL_0084
    IL_0045:  ldarg.0
    IL_0046:  ldc.i4.0
    IL_0047:  dup
    IL_0048:  stloc.0
    IL_0049:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_004e:  ldarg.0
    IL_004f:  ldloc.1
    IL_0050:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0055:  ldarg.0
    IL_0056:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_005b:  ldloca.s   V_1
    IL_005d:  ldarg.0
    IL_005e:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0063:  leave      IL_0191
    IL_0068:  ldarg.0
    IL_0069:  ldc.i4.m1
    IL_006a:  dup
    IL_006b:  stloc.0
    IL_006c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0071:  ldarg.0
    IL_0072:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0077:  stloc.1
    IL_0078:  ldarg.0
    IL_0079:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007e:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0084:  ldarg.0
    IL_0085:  ldarg.0
    IL_0086:  ldloca.s   V_1
    IL_0088:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_008d:  ldloca.s   V_1
    IL_008f:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0095:  dup
    IL_0096:  stloc.s    V_4
    IL_0098:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S1 As Integer"
    IL_009d:  ldloc.s    V_4
    IL_009f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U3 As Integer"
    IL_00a4:  ldarg.0
    IL_00a5:  ldarg.0
    IL_00a6:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_00ab:  box        "SM$T"
    IL_00b0:  ldarg.0
    IL_00b1:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S1 As Integer"
    IL_00b6:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_00bb:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_00c0:  ldarg.0
    IL_00c1:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_00c6:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_00cb:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_00d0:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d5:  stloc.2
    IL_00d6:  ldloca.s   V_2
    IL_00d8:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_00dd:  brtrue.s   IL_011e
    IL_00df:  ldarg.0
    IL_00e0:  ldc.i4.1
    IL_00e1:  dup
    IL_00e2:  stloc.0
    IL_00e3:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00e8:  ldarg.0
    IL_00e9:  ldloc.2
    IL_00ea:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00ef:  ldarg.0
    IL_00f0:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00f5:  ldloca.s   V_2
    IL_00f7:  ldarg.0
    IL_00f8:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_00fd:  leave      IL_0191
    IL_0102:  ldarg.0
    IL_0103:  ldc.i4.m1
    IL_0104:  dup
    IL_0105:  stloc.0
    IL_0106:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_010b:  ldarg.0
    IL_010c:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0111:  stloc.2
    IL_0112:  ldarg.0
    IL_0113:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0118:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_011e:  ldarg.0
    IL_011f:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_0124:  box        "SM$T"
    IL_0129:  ldarg.0
    IL_012a:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U3 As Integer"
    IL_012f:  ldarg.0
    IL_0130:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_0135:  ldloca.s   V_2
    IL_0137:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_013c:  ldloca.s   V_2
    IL_013e:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0144:  add.ovf
    IL_0145:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_014a:  ldarg.0
    IL_014b:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_0150:  initobj    "SM$T"
    IL_0156:  leave.s    IL_017c
  }
  catch System.Exception
  {
    IL_0158:  dup
    IL_0159:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_015e:  stloc.s    V_5
    IL_0160:  ldarg.0
    IL_0161:  ldc.i4.s   -2
    IL_0163:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0168:  ldarg.0
    IL_0169:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_016e:  ldloc.s    V_5
    IL_0170:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0175:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_017a:  leave.s    IL_0191
  }
  IL_017c:  ldarg.0
  IL_017d:  ldc.i4.s   -2
  IL_017f:  dup
  IL_0180:  stloc.0
  IL_0181:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0186:  ldarg.0
  IL_0187:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_018c:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0191:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      447 (0x1bf)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T V_3,
                Integer V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_006f
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_0122
    IL_0011:  ldloca.s   V_3
    IL_0013:  initobj    "SM$T"
    IL_0019:  ldloc.3
    IL_001a:  box        "SM$T"
    IL_001f:  brtrue.s   IL_002d
    IL_0021:  ldarg.0
    IL_0022:  ldarg.0
    IL_0023:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0028:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_002d:  ldarg.0
    IL_002e:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0033:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0038:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_003d:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0042:  stloc.1
    IL_0043:  ldloca.s   V_1
    IL_0045:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_004a:  brtrue.s   IL_008b
    IL_004c:  ldarg.0
    IL_004d:  ldc.i4.0
    IL_004e:  dup
    IL_004f:  stloc.0
    IL_0050:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0055:  ldarg.0
    IL_0056:  ldloc.1
    IL_0057:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_005c:  ldarg.0
    IL_005d:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0062:  ldloca.s   V_1
    IL_0064:  ldarg.0
    IL_0065:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_006a:  leave      IL_01be
    IL_006f:  ldarg.0
    IL_0070:  ldc.i4.m1
    IL_0071:  dup
    IL_0072:  stloc.0
    IL_0073:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0078:  ldarg.0
    IL_0079:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007e:  stloc.1
    IL_007f:  ldarg.0
    IL_0080:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0085:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008b:  ldarg.0
    IL_008c:  ldarg.0
    IL_008d:  ldloca.s   V_1
    IL_008f:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0094:  ldloca.s   V_1
    IL_0096:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009c:  dup
    IL_009d:  stloc.s    V_4
    IL_009f:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S1 As Integer"
    IL_00a4:  ldloc.s    V_4
    IL_00a6:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As Integer"
    IL_00ab:  ldarg.0
    IL_00ac:  ldloca.s   V_3
    IL_00ae:  initobj    "SM$T"
    IL_00b4:  ldloc.3
    IL_00b5:  box        "SM$T"
    IL_00ba:  brtrue.s   IL_00c4
    IL_00bc:  ldarg.0
    IL_00bd:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_00c2:  br.s       IL_00ca
    IL_00c4:  ldarg.0
    IL_00c5:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00ca:  ldarg.0
    IL_00cb:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S1 As Integer"
    IL_00d0:  constrained. "SM$T"
    IL_00d6:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_00db:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As Integer"
    IL_00e0:  ldarg.0
    IL_00e1:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00e6:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_00eb:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_00f0:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00f5:  stloc.2
    IL_00f6:  ldloca.s   V_2
    IL_00f8:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_00fd:  brtrue.s   IL_013e
    IL_00ff:  ldarg.0
    IL_0100:  ldc.i4.1
    IL_0101:  dup
    IL_0102:  stloc.0
    IL_0103:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0108:  ldarg.0
    IL_0109:  ldloc.2
    IL_010a:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_010f:  ldarg.0
    IL_0110:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0115:  ldloca.s   V_2
    IL_0117:  ldarg.0
    IL_0118:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_011d:  leave      IL_01be
    IL_0122:  ldarg.0
    IL_0123:  ldc.i4.m1
    IL_0124:  dup
    IL_0125:  stloc.0
    IL_0126:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_012b:  ldarg.0
    IL_012c:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0131:  stloc.2
    IL_0132:  ldarg.0
    IL_0133:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0138:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_013e:  ldloca.s   V_3
    IL_0140:  initobj    "SM$T"
    IL_0146:  ldloc.3
    IL_0147:  box        "SM$T"
    IL_014c:  brtrue.s   IL_0156
    IL_014e:  ldarg.0
    IL_014f:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_0154:  br.s       IL_015c
    IL_0156:  ldarg.0
    IL_0157:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_015c:  ldarg.0
    IL_015d:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As Integer"
    IL_0162:  ldarg.0
    IL_0163:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As Integer"
    IL_0168:  ldloca.s   V_2
    IL_016a:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_016f:  ldloca.s   V_2
    IL_0171:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0177:  add.ovf
    IL_0178:  constrained. "SM$T"
    IL_017e:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_0183:  leave.s    IL_01a9
  }
  catch System.Exception
  {
    IL_0185:  dup
    IL_0186:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_018b:  stloc.s    V_5
    IL_018d:  ldarg.0
    IL_018e:  ldc.i4.s   -2
    IL_0190:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0195:  ldarg.0
    IL_0196:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_019b:  ldloc.s    V_5
    IL_019d:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_01a2:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_01a7:  leave.s    IL_01be
  }
  IL_01a9:  ldarg.0
  IL_01aa:  ldc.i4.s   -2
  IL_01ac:  dup
  IL_01ad:  stloc.0
  IL_01ae:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_01b3:  ldarg.0
  IL_01b4:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_01b9:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_01be:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_IndexAndValue_Async_04()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        item.Position(await GetOffsetAsync(GetOffset(item))) += await GetOffsetAsync(GetOffset(item))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        item.Position(await GetOffsetAsync(GetOffset(item))) += await GetOffsetAsync(GetOffset(item))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-2'
Position get for item '-3'
Position set for item '-4'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      369 (0x171)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                Integer V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0053
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_00ec
    IL_0011:  ldarg.0
    IL_0012:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0017:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_001c:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0021:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0026:  stloc.1
    IL_0027:  ldloca.s   V_1
    IL_0029:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_002e:  brtrue.s   IL_006f
    IL_0030:  ldarg.0
    IL_0031:  ldc.i4.0
    IL_0032:  dup
    IL_0033:  stloc.0
    IL_0034:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0039:  ldarg.0
    IL_003a:  ldloc.1
    IL_003b:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0040:  ldarg.0
    IL_0041:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0046:  ldloca.s   V_1
    IL_0048:  ldarg.0
    IL_0049:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_004e:  leave      IL_0170
    IL_0053:  ldarg.0
    IL_0054:  ldc.i4.m1
    IL_0055:  dup
    IL_0056:  stloc.0
    IL_0057:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_005c:  ldarg.0
    IL_005d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0062:  stloc.1
    IL_0063:  ldarg.0
    IL_0064:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0069:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_006f:  ldarg.0
    IL_0070:  ldarg.0
    IL_0071:  ldloca.s   V_1
    IL_0073:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0078:  ldloca.s   V_1
    IL_007a:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0080:  dup
    IL_0081:  stloc.3
    IL_0082:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As Integer"
    IL_0087:  ldloc.3
    IL_0088:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As Integer"
    IL_008d:  ldarg.0
    IL_008e:  ldarg.0
    IL_008f:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0094:  ldarg.0
    IL_0095:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As Integer"
    IL_009a:  constrained. "SM$T"
    IL_00a0:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_00a5:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_00aa:  ldarg.0
    IL_00ab:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_00b0:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_00b5:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_00ba:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00bf:  stloc.2
    IL_00c0:  ldloca.s   V_2
    IL_00c2:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_00c7:  brtrue.s   IL_0108
    IL_00c9:  ldarg.0
    IL_00ca:  ldc.i4.1
    IL_00cb:  dup
    IL_00cc:  stloc.0
    IL_00cd:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00d2:  ldarg.0
    IL_00d3:  ldloc.2
    IL_00d4:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d9:  ldarg.0
    IL_00da:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00df:  ldloca.s   V_2
    IL_00e1:  ldarg.0
    IL_00e2:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_00e7:  leave      IL_0170
    IL_00ec:  ldarg.0
    IL_00ed:  ldc.i4.m1
    IL_00ee:  dup
    IL_00ef:  stloc.0
    IL_00f0:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00f5:  ldarg.0
    IL_00f6:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00fb:  stloc.2
    IL_00fc:  ldarg.0
    IL_00fd:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0102:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0108:  ldarg.0
    IL_0109:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_010e:  ldarg.0
    IL_010f:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As Integer"
    IL_0114:  ldarg.0
    IL_0115:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_011a:  ldloca.s   V_2
    IL_011c:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0121:  ldloca.s   V_2
    IL_0123:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0129:  add.ovf
    IL_012a:  constrained. "SM$T"
    IL_0130:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_0135:  leave.s    IL_015b
  }
  catch System.Exception
  {
    IL_0137:  dup
    IL_0138:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_013d:  stloc.s    V_4
    IL_013f:  ldarg.0
    IL_0140:  ldc.i4.s   -2
    IL_0142:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0147:  ldarg.0
    IL_0148:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_014d:  ldloc.s    V_4
    IL_014f:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0154:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0159:  leave.s    IL_0170
  }
  IL_015b:  ldarg.0
  IL_015c:  ldc.i4.s   -2
  IL_015e:  dup
  IL_015f:  stloc.0
  IL_0160:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0165:  ldarg.0
  IL_0166:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_016b:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0170:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Index_InWith()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(item As T)
        With item
            .Position(GetOffset(item)) += 1
        End With
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        With item
            .Position(GetOffset(item)) += 1
        End With
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       39 (0x27)
  .maxstack  4
  .locals init (T V_0,
                Integer V_1)
  IL_0000:  ldarg.0
  IL_0001:  dup
  IL_0002:  stloc.0
  IL_0003:  box        "T"
  IL_0008:  ldarga.s   V_0
  IL_000a:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_000f:  dup
  IL_0010:  stloc.1
  IL_0011:  ldloca.s   V_0
  IL_0013:  ldloc.1
  IL_0014:  constrained. "T"
  IL_001a:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_001f:  ldc.i4.1
  IL_0020:  add.ovf
  IL_0021:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0026:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
      // Code size       81 (0x51)
  .maxstack  4
  .locals init (T V_0,
                Integer V_1,
                T V_2)
  IL_0000:  ldloca.s   V_2
  IL_0002:  initobj    "T"
  IL_0008:  ldloc.2
  IL_0009:  box        "T"
  IL_000e:  brtrue.s   IL_0016
  IL_0010:  ldarg.0
  IL_0011:  stloc.0
  IL_0012:  ldloca.s   V_0
  IL_0014:  br.s       IL_0018
  IL_0016:  ldarga.s   V_0
  IL_0018:  ldarga.s   V_0
  IL_001a:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_001f:  dup
  IL_0020:  stloc.1
  IL_0021:  ldloca.s   V_2
  IL_0023:  initobj    "T"
  IL_0029:  ldloc.2
  IL_002a:  box        "T"
  IL_002f:  brtrue.s   IL_0035
  IL_0031:  ldloca.s   V_0
  IL_0033:  br.s       IL_0037
  IL_0035:  ldarga.s   V_0
  IL_0037:  ldloc.1
  IL_0038:  constrained. "T"
  IL_003e:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0043:  ldc.i4.1
  IL_0044:  add.ovf
  IL_0045:  constrained. "T"
  IL_004b:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0050:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Index_InWith()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(item As T)
        With item
            .Position(GetOffset(item)) += 1
        End With
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        With item
            .Position(GetOffset(item)) += 1
        End With
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-1'
Position get for item '-2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       39 (0x27)
  .maxstack  4
  .locals init (Integer V_0)
  IL_0000:  ldarga.s   V_0
  IL_0002:  ldarga.s   V_0
  IL_0004:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0009:  dup
  IL_000a:  stloc.0
  IL_000b:  ldarga.s   V_0
  IL_000d:  ldloc.0
  IL_000e:  constrained. "T"
  IL_0014:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0019:  ldc.i4.1
  IL_001a:  add.ovf
  IL_001b:  constrained. "T"
  IL_0021:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0026:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Index_Ref_InWith()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(ByRef item As T)
        With item
            .Position(GetOffset(item)) += 1
        End With
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        With item
            .Position(GetOffset(item)) += 1
        End With
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       43 (0x2b)
  .maxstack  4
  .locals init (T V_0,
                Integer V_1)
  IL_0000:  ldarg.0
  IL_0001:  ldobj      "T"
  IL_0006:  dup
  IL_0007:  stloc.0
  IL_0008:  box        "T"
  IL_000d:  ldarg.0
  IL_000e:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0013:  dup
  IL_0014:  stloc.1
  IL_0015:  ldloca.s   V_0
  IL_0017:  ldloc.1
  IL_0018:  constrained. "T"
  IL_001e:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0023:  ldc.i4.1
  IL_0024:  add.ovf
  IL_0025:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_002a:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
      // Code size       83 (0x53)
  .maxstack  4
  .locals init (T V_0,
                Integer V_1,
                T V_2)
  IL_0000:  ldloca.s   V_2
  IL_0002:  initobj    "T"
  IL_0008:  ldloc.2
  IL_0009:  box        "T"
  IL_000e:  brtrue.s   IL_001b
  IL_0010:  ldarg.0
  IL_0011:  ldobj      "T"
  IL_0016:  stloc.0
  IL_0017:  ldloca.s   V_0
  IL_0019:  br.s       IL_001c
  IL_001b:  ldarg.0
  IL_001c:  ldarg.0
  IL_001d:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0022:  dup
  IL_0023:  stloc.1
  IL_0024:  ldloca.s   V_2
  IL_0026:  initobj    "T"
  IL_002c:  ldloc.2
  IL_002d:  box        "T"
  IL_0032:  brtrue.s   IL_0038
  IL_0034:  ldloca.s   V_0
  IL_0036:  br.s       IL_0039
  IL_0038:  ldarg.0
  IL_0039:  ldloc.1
  IL_003a:  constrained. "T"
  IL_0040:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0045:  ldc.i4.1
  IL_0046:  add.ovf
  IL_0047:  constrained. "T"
  IL_004d:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0052:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Index_Ref_InWith()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(ByRef item As T)
        With item
            .Position(GetOffset(item)) += 1
        End With
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        With item
            .Position(GetOffset(item)) += 1
        End With
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-1'
Position get for item '-2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       36 (0x24)
  .maxstack  4
  .locals init (Integer V_0)
  IL_0000:  ldarg.0
  IL_0001:  ldarg.0
  IL_0002:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0007:  dup
  IL_0008:  stloc.0
  IL_0009:  ldarg.0
  IL_000a:  ldloc.0
  IL_000b:  constrained. "T"
  IL_0011:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0016:  ldc.i4.1
  IL_0017:  add.ovf
  IL_0018:  constrained. "T"
  IL_001e:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0023:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Index_Async_01_InWith()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        With item
            .Position(await GetOffsetAsync(GetOffset(item))) += 1
        End With
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        With item
            .Position(await GetOffsetAsync(GetOffset(item))) += 1
        End With
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      262 (0x106)
  .maxstack  4
  .locals init (Integer V_0,
                SM$T V_1, //$W0
                Integer V_2,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_3,
                SM$T V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0065
    IL_000a:  ldarg.0
    IL_000b:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0010:  stloc.1
    IL_0011:  ldarg.0
    IL_0012:  ldarg.0
    IL_0013:  ldloc.1
    IL_0014:  dup
    IL_0015:  stloc.s    V_4
    IL_0017:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_001c:  ldloc.s    V_4
    IL_001e:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0023:  ldarg.0
    IL_0024:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0029:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_002e:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0033:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0038:  stloc.3
    IL_0039:  ldloca.s   V_3
    IL_003b:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0040:  brtrue.s   IL_0081
    IL_0042:  ldarg.0
    IL_0043:  ldc.i4.0
    IL_0044:  dup
    IL_0045:  stloc.0
    IL_0046:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_004b:  ldarg.0
    IL_004c:  ldloc.3
    IL_004d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0052:  ldarg.0
    IL_0053:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0058:  ldloca.s   V_3
    IL_005a:  ldarg.0
    IL_005b:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0060:  leave      IL_0105
    IL_0065:  ldarg.0
    IL_0066:  ldc.i4.m1
    IL_0067:  dup
    IL_0068:  stloc.0
    IL_0069:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_006e:  ldarg.0
    IL_006f:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0074:  stloc.3
    IL_0075:  ldarg.0
    IL_0076:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007b:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0081:  ldarg.0
    IL_0082:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0087:  box        "SM$T"
    IL_008c:  ldloca.s   V_3
    IL_008e:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0093:  ldloca.s   V_3
    IL_0095:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009b:  dup
    IL_009c:  stloc.2
    IL_009d:  ldarg.0
    IL_009e:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_00a3:  ldloc.2
    IL_00a4:  constrained. "SM$T"
    IL_00aa:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_00af:  ldc.i4.1
    IL_00b0:  add.ovf
    IL_00b1:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00b6:  ldarg.0
    IL_00b7:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_00bc:  initobj    "SM$T"
    IL_00c2:  ldloca.s   V_1
    IL_00c4:  initobj    "SM$T"
    IL_00ca:  leave.s    IL_00f0
  }
  catch System.Exception
  {
    IL_00cc:  dup
    IL_00cd:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00d2:  stloc.s    V_5
    IL_00d4:  ldarg.0
    IL_00d5:  ldc.i4.s   -2
    IL_00d7:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00dc:  ldarg.0
    IL_00dd:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00e2:  ldloc.s    V_5
    IL_00e4:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00e9:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00ee:  leave.s    IL_0105
  }
  IL_00f0:  ldarg.0
  IL_00f1:  ldc.i4.s   -2
  IL_00f3:  dup
  IL_00f4:  stloc.0
  IL_00f5:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00fa:  ldarg.0
  IL_00fb:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0100:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0105:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      294 (0x126)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0068
    IL_000a:  ldloca.s   V_3
    IL_000c:  initobj    "SM$T"
    IL_0012:  ldloc.3
    IL_0013:  box        "SM$T"
    IL_0018:  brtrue.s   IL_0026
    IL_001a:  ldarg.0
    IL_001b:  ldarg.0
    IL_001c:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0021:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_0026:  ldarg.0
    IL_0027:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_002c:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0031:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0036:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_003b:  stloc.2
    IL_003c:  ldloca.s   V_2
    IL_003e:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0043:  brtrue.s   IL_0084
    IL_0045:  ldarg.0
    IL_0046:  ldc.i4.0
    IL_0047:  dup
    IL_0048:  stloc.0
    IL_0049:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_004e:  ldarg.0
    IL_004f:  ldloc.2
    IL_0050:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0055:  ldarg.0
    IL_0056:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_005b:  ldloca.s   V_2
    IL_005d:  ldarg.0
    IL_005e:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_0063:  leave      IL_0125
    IL_0068:  ldarg.0
    IL_0069:  ldc.i4.m1
    IL_006a:  dup
    IL_006b:  stloc.0
    IL_006c:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0071:  ldarg.0
    IL_0072:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0077:  stloc.2
    IL_0078:  ldarg.0
    IL_0079:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007e:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0084:  ldloca.s   V_3
    IL_0086:  initobj    "SM$T"
    IL_008c:  ldloc.3
    IL_008d:  box        "SM$T"
    IL_0092:  brtrue.s   IL_009c
    IL_0094:  ldarg.0
    IL_0095:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_009a:  br.s       IL_00a2
    IL_009c:  ldarg.0
    IL_009d:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00a2:  ldloca.s   V_2
    IL_00a4:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00a9:  ldloca.s   V_2
    IL_00ab:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b1:  dup
    IL_00b2:  stloc.1
    IL_00b3:  ldloca.s   V_3
    IL_00b5:  initobj    "SM$T"
    IL_00bb:  ldloc.3
    IL_00bc:  box        "SM$T"
    IL_00c1:  brtrue.s   IL_00cb
    IL_00c3:  ldarg.0
    IL_00c4:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_00c9:  br.s       IL_00d1
    IL_00cb:  ldarg.0
    IL_00cc:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00d1:  ldloc.1
    IL_00d2:  constrained. "SM$T"
    IL_00d8:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_00dd:  ldc.i4.1
    IL_00de:  add.ovf
    IL_00df:  constrained. "SM$T"
    IL_00e5:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00ea:  leave.s    IL_0110
  }
  catch System.Exception
  {
    IL_00ec:  dup
    IL_00ed:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00f2:  stloc.s    V_4
    IL_00f4:  ldarg.0
    IL_00f5:  ldc.i4.s   -2
    IL_00f7:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00fc:  ldarg.0
    IL_00fd:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0102:  ldloc.s    V_4
    IL_0104:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0109:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_010e:  leave.s    IL_0125
  }
  IL_0110:  ldarg.0
  IL_0111:  ldc.i4.s   -2
  IL_0113:  dup
  IL_0114:  stloc.0
  IL_0115:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_011a:  ldarg.0
  IL_011b:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0120:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0125:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Index_Async_01_InWith()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        With item
            .Position(await GetOffsetAsync(GetOffset(item))) += 1
        End With
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        With item
            .Position(await GetOffsetAsync(GetOffset(item))) += 1
        End With
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-1'
Position get for item '-2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      216 (0xd8)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004c
    IL_000a:  ldarg.0
    IL_000b:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0010:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0015:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_001a:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_001f:  stloc.2
    IL_0020:  ldloca.s   V_2
    IL_0022:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0027:  brtrue.s   IL_0068
    IL_0029:  ldarg.0
    IL_002a:  ldc.i4.0
    IL_002b:  dup
    IL_002c:  stloc.0
    IL_002d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0032:  ldarg.0
    IL_0033:  ldloc.2
    IL_0034:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0039:  ldarg.0
    IL_003a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003f:  ldloca.s   V_2
    IL_0041:  ldarg.0
    IL_0042:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0047:  leave      IL_00d7
    IL_004c:  ldarg.0
    IL_004d:  ldc.i4.m1
    IL_004e:  dup
    IL_004f:  stloc.0
    IL_0050:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0055:  ldarg.0
    IL_0056:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_005b:  stloc.2
    IL_005c:  ldarg.0
    IL_005d:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0062:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0068:  ldarg.0
    IL_0069:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_006e:  ldloca.s   V_2
    IL_0070:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0075:  ldloca.s   V_2
    IL_0077:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007d:  dup
    IL_007e:  stloc.1
    IL_007f:  ldarg.0
    IL_0080:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0085:  ldloc.1
    IL_0086:  constrained. "SM$T"
    IL_008c:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_0091:  ldc.i4.1
    IL_0092:  add.ovf
    IL_0093:  constrained. "SM$T"
    IL_0099:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_009e:  leave.s    IL_00c2
  }
  catch System.Exception
  {
    IL_00a0:  dup
    IL_00a1:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00a6:  stloc.3
    IL_00a7:  ldarg.0
    IL_00a8:  ldc.i4.s   -2
    IL_00aa:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00af:  ldarg.0
    IL_00b0:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00b5:  ldloc.3
    IL_00b6:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00bb:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00c0:  leave.s    IL_00d7
  }
  IL_00c2:  ldarg.0
  IL_00c3:  ldc.i4.s   -2
  IL_00c5:  dup
  IL_00c6:  stloc.0
  IL_00c7:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00cc:  ldarg.0
  IL_00cd:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00d2:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_00d7:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Value_InWith()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(item As T)
        With item
            .Position(1) += GetOffset(item)
        End With
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        With item
            .Position(1) += GetOffset(item)
        End With
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       37 (0x25)
  .maxstack  4
  .locals init (T V_0)
  IL_0000:  ldarg.0
  IL_0001:  dup
  IL_0002:  stloc.0
  IL_0003:  box        "T"
  IL_0008:  ldc.i4.1
  IL_0009:  ldloca.s   V_0
  IL_000b:  ldc.i4.1
  IL_000c:  constrained. "T"
  IL_0012:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0017:  ldarga.s   V_0
  IL_0019:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_001e:  add.ovf
  IL_001f:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0024:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
      // Code size       61 (0x3d)
  .maxstack  4
  .locals init (T V_0)
  IL_0000:  ldarga.s   V_0
  IL_0002:  ldloca.s   V_0
  IL_0004:  initobj    "T"
  IL_000a:  ldloc.0
  IL_000b:  box        "T"
  IL_0010:  brtrue.s   IL_001a
  IL_0012:  ldobj      "T"
  IL_0017:  stloc.0
  IL_0018:  ldloca.s   V_0
  IL_001a:  ldc.i4.1
  IL_001b:  ldarga.s   V_0
  IL_001d:  ldc.i4.1
  IL_001e:  constrained. "T"
  IL_0024:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0029:  ldarga.s   V_0
  IL_002b:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0030:  add.ovf
  IL_0031:  constrained. "T"
  IL_0037:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_003c:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Value_InWith()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(item As T)
        With item
            .Position(1) += GetOffset(item)
        End With
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        With item
            .Position(1) += GetOffset(item)
        End With
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '-1'
Position get for item '2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       37 (0x25)
  .maxstack  4
  IL_0000:  ldarga.s   V_0
  IL_0002:  ldc.i4.1
  IL_0003:  ldarga.s   V_0
  IL_0005:  ldc.i4.1
  IL_0006:  constrained. "T"
  IL_000c:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0011:  ldarga.s   V_0
  IL_0013:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0018:  add.ovf
  IL_0019:  constrained. "T"
  IL_001f:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0024:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Value_Ref_InWith()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(ByRef item As T)
        With item
            .Position(1) += GetOffset(item)
        End With
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        With item
            .Position(1) += GetOffset(item)
        End With
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       41 (0x29)
  .maxstack  4
  .locals init (T V_0)
  IL_0000:  ldarg.0
  IL_0001:  ldobj      "T"
  IL_0006:  dup
  IL_0007:  stloc.0
  IL_0008:  box        "T"
  IL_000d:  ldc.i4.1
  IL_000e:  ldloca.s   V_0
  IL_0010:  ldc.i4.1
  IL_0011:  constrained. "T"
  IL_0017:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_001c:  ldarg.0
  IL_001d:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0022:  add.ovf
  IL_0023:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0028:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
      // Code size       58 (0x3a)
  .maxstack  4
  .locals init (T V_0)
  IL_0000:  ldarg.0
  IL_0001:  ldloca.s   V_0
  IL_0003:  initobj    "T"
  IL_0009:  ldloc.0
  IL_000a:  box        "T"
  IL_000f:  brtrue.s   IL_0019
  IL_0011:  ldobj      "T"
  IL_0016:  stloc.0
  IL_0017:  ldloca.s   V_0
  IL_0019:  ldc.i4.1
  IL_001a:  ldarg.0
  IL_001b:  ldc.i4.1
  IL_001c:  constrained. "T"
  IL_0022:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0027:  ldarg.0
  IL_0028:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_002d:  add.ovf
  IL_002e:  constrained. "T"
  IL_0034:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0039:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Value_Ref_InWith()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(ByRef item As T)
        With item
            .Position(1) += GetOffset(item)
        End With
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        With item
            .Position(1) += GetOffset(item)
        End With
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '-1'
Position get for item '2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       34 (0x22)
  .maxstack  4
  IL_0000:  ldarg.0
  IL_0001:  ldc.i4.1
  IL_0002:  ldarg.0
  IL_0003:  ldc.i4.1
  IL_0004:  constrained. "T"
  IL_000a:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_000f:  ldarg.0
  IL_0010:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0015:  add.ovf
  IL_0016:  constrained. "T"
  IL_001c:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0021:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Value_Async_01_InWith()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        With item
            .Position(1) += await GetOffsetAsync(GetOffset(item))
        End With
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        With item
            .Position(1) += await GetOffsetAsync(GetOffset(item))
        End With
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      270 (0x10e)
  .maxstack  5
  .locals init (Integer V_0,
                SM$T V_1, //$W0
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_007b
    IL_000a:  ldarg.0
    IL_000b:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0010:  stloc.1
    IL_0011:  ldarg.0
    IL_0012:  ldarg.0
    IL_0013:  ldloc.1
    IL_0014:  dup
    IL_0015:  stloc.3
    IL_0016:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_001b:  ldloc.3
    IL_001c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_0021:  ldarg.0
    IL_0022:  ldarg.0
    IL_0023:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0028:  ldc.i4.1
    IL_0029:  constrained. "SM$T"
    IL_002f:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_0034:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_0039:  ldarg.0
    IL_003a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_003f:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0044:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0049:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_004e:  stloc.2
    IL_004f:  ldloca.s   V_2
    IL_0051:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0056:  brtrue.s   IL_0097
    IL_0058:  ldarg.0
    IL_0059:  ldc.i4.0
    IL_005a:  dup
    IL_005b:  stloc.0
    IL_005c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0061:  ldarg.0
    IL_0062:  ldloc.2
    IL_0063:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0068:  ldarg.0
    IL_0069:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_006e:  ldloca.s   V_2
    IL_0070:  ldarg.0
    IL_0071:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0076:  leave      IL_010d
    IL_007b:  ldarg.0
    IL_007c:  ldc.i4.m1
    IL_007d:  dup
    IL_007e:  stloc.0
    IL_007f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0084:  ldarg.0
    IL_0085:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008a:  stloc.2
    IL_008b:  ldarg.0
    IL_008c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0091:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0097:  ldarg.0
    IL_0098:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_009d:  box        "SM$T"
    IL_00a2:  ldc.i4.1
    IL_00a3:  ldarg.0
    IL_00a4:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_00a9:  ldloca.s   V_2
    IL_00ab:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00b0:  ldloca.s   V_2
    IL_00b2:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b8:  add.ovf
    IL_00b9:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00be:  ldarg.0
    IL_00bf:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_00c4:  initobj    "SM$T"
    IL_00ca:  ldloca.s   V_1
    IL_00cc:  initobj    "SM$T"
    IL_00d2:  leave.s    IL_00f8
  }
  catch System.Exception
  {
    IL_00d4:  dup
    IL_00d5:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00da:  stloc.s    V_4
    IL_00dc:  ldarg.0
    IL_00dd:  ldc.i4.s   -2
    IL_00df:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00e4:  ldarg.0
    IL_00e5:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00ea:  ldloc.s    V_4
    IL_00ec:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00f1:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00f6:  leave.s    IL_010d
  }
  IL_00f8:  ldarg.0
  IL_00f9:  ldc.i4.s   -2
  IL_00fb:  dup
  IL_00fc:  stloc.0
  IL_00fd:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0102:  ldarg.0
  IL_0103:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0108:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_010d:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      290 (0x122)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                SM$T V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0080
    IL_000a:  ldloca.s   V_2
    IL_000c:  initobj    "SM$T"
    IL_0012:  ldloc.2
    IL_0013:  box        "SM$T"
    IL_0018:  brtrue.s   IL_0026
    IL_001a:  ldarg.0
    IL_001b:  ldarg.0
    IL_001c:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0021:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T"
    IL_0026:  ldarg.0
    IL_0027:  ldarg.0
    IL_0028:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_002d:  ldc.i4.1
    IL_002e:  constrained. "SM$T"
    IL_0034:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_0039:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As Integer"
    IL_003e:  ldarg.0
    IL_003f:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0044:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0049:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_004e:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0053:  stloc.1
    IL_0054:  ldloca.s   V_1
    IL_0056:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_005b:  brtrue.s   IL_009c
    IL_005d:  ldarg.0
    IL_005e:  ldc.i4.0
    IL_005f:  dup
    IL_0060:  stloc.0
    IL_0061:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0066:  ldarg.0
    IL_0067:  ldloc.1
    IL_0068:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_006d:  ldarg.0
    IL_006e:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0073:  ldloca.s   V_1
    IL_0075:  ldarg.0
    IL_0076:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_007b:  leave      IL_0121
    IL_0080:  ldarg.0
    IL_0081:  ldc.i4.m1
    IL_0082:  dup
    IL_0083:  stloc.0
    IL_0084:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0089:  ldarg.0
    IL_008a:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008f:  stloc.1
    IL_0090:  ldarg.0
    IL_0091:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0096:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009c:  ldloca.s   V_2
    IL_009e:  initobj    "SM$T"
    IL_00a4:  ldloc.2
    IL_00a5:  box        "SM$T"
    IL_00aa:  brtrue.s   IL_00b4
    IL_00ac:  ldarg.0
    IL_00ad:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T"
    IL_00b2:  br.s       IL_00ba
    IL_00b4:  ldarg.0
    IL_00b5:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00ba:  ldc.i4.1
    IL_00bb:  ldarg.0
    IL_00bc:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As Integer"
    IL_00c1:  ldloca.s   V_1
    IL_00c3:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00c8:  ldloca.s   V_1
    IL_00ca:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d0:  add.ovf
    IL_00d1:  constrained. "SM$T"
    IL_00d7:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00dc:  ldarg.0
    IL_00dd:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T"
    IL_00e2:  initobj    "SM$T"
    IL_00e8:  leave.s    IL_010c
  }
  catch System.Exception
  {
    IL_00ea:  dup
    IL_00eb:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00f0:  stloc.3
    IL_00f1:  ldarg.0
    IL_00f2:  ldc.i4.s   -2
    IL_00f4:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00f9:  ldarg.0
    IL_00fa:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00ff:  ldloc.3
    IL_0100:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0105:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_010a:  leave.s    IL_0121
  }
  IL_010c:  ldarg.0
  IL_010d:  ldc.i4.s   -2
  IL_010f:  dup
  IL_0110:  stloc.0
  IL_0111:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0116:  ldarg.0
  IL_0117:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_011c:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0121:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Value_Async_01_InWith()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        With item
            .Position(1) += await GetOffsetAsync(GetOffset(item))
        End With
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        With item
            .Position(1) += await GetOffsetAsync(GetOffset(item))
        End With
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '-1'
Position get for item '2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      223 (0xdf)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                System.Exception V_2)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0061
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0011:  ldc.i4.1
    IL_0012:  constrained. "SM$T"
    IL_0018:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_001d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_0022:  ldarg.0
    IL_0023:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0028:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_002d:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0032:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0037:  stloc.1
    IL_0038:  ldloca.s   V_1
    IL_003a:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_003f:  brtrue.s   IL_007d
    IL_0041:  ldarg.0
    IL_0042:  ldc.i4.0
    IL_0043:  dup
    IL_0044:  stloc.0
    IL_0045:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_004a:  ldarg.0
    IL_004b:  ldloc.1
    IL_004c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0051:  ldarg.0
    IL_0052:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0057:  ldloca.s   V_1
    IL_0059:  ldarg.0
    IL_005a:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_005f:  leave.s    IL_00de
    IL_0061:  ldarg.0
    IL_0062:  ldc.i4.m1
    IL_0063:  dup
    IL_0064:  stloc.0
    IL_0065:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_006a:  ldarg.0
    IL_006b:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0070:  stloc.1
    IL_0071:  ldarg.0
    IL_0072:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0077:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007d:  ldarg.0
    IL_007e:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0083:  ldc.i4.1
    IL_0084:  ldarg.0
    IL_0085:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_008a:  ldloca.s   V_1
    IL_008c:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0091:  ldloca.s   V_1
    IL_0093:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0099:  add.ovf
    IL_009a:  constrained. "SM$T"
    IL_00a0:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00a5:  leave.s    IL_00c9
  }
  catch System.Exception
  {
    IL_00a7:  dup
    IL_00a8:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00ad:  stloc.2
    IL_00ae:  ldarg.0
    IL_00af:  ldc.i4.s   -2
    IL_00b1:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00b6:  ldarg.0
    IL_00b7:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00bc:  ldloc.2
    IL_00bd:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00c2:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00c7:  leave.s    IL_00de
  }
  IL_00c9:  ldarg.0
  IL_00ca:  ldc.i4.s   -2
  IL_00cc:  dup
  IL_00cd:  stloc.0
  IL_00ce:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00d3:  ldarg.0
  IL_00d4:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00d9:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_00de:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Index_ThroughArray()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Item2
    Inherits Item
End Class

Class Program
    Shared Sub Main()
        Dim item1 = { New Item2 With {.Name = "1"} }
        Call1(DirectCast(item1, Item()))

        Dim item2 = { New Item2 With {.Name = "2"} }
        Call2(DirectCast(item2, Item()))
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(item As T())
        item(GetArrayIndex()).Position(GetOffset(item)) += 1
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T())
        item(GetArrayIndex()).Position(GetOffset(item)) += 1
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(item As T()) As Integer
        value -= 1
        item(0) = DirectCast(DirectCast(New Item2 With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Function GetArrayIndex() As Integer
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       48 (0x30)
  .maxstack  4
  .locals init (T V_0,
                Integer V_1)
  IL_0000:  ldarg.0
  IL_0001:  call       "Function Program.GetArrayIndex() As Integer"
  IL_0006:  ldelem     "T"
  IL_000b:  dup
  IL_000c:  stloc.0
  IL_000d:  box        "T"
  IL_0012:  ldarg.0
  IL_0013:  call       "Function Program.GetOffset(Of T)(T()) As Integer"
  IL_0018:  dup
  IL_0019:  stloc.1
  IL_001a:  ldloca.s   V_0
  IL_001c:  ldloc.1
  IL_001d:  constrained. "T"
  IL_0023:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0028:  ldc.i4.1
  IL_0029:  add.ovf
  IL_002a:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_002f:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
      // Code size      120 (0x78)
  .maxstack  4
  .locals init (T() V_0,
                Integer V_1,
                T V_2,
                Integer V_3,
                T V_4)
  IL_0000:  ldarg.0
  IL_0001:  dup
  IL_0002:  stloc.0
  IL_0003:  call       "Function Program.GetArrayIndex() As Integer"
  IL_0008:  dup
  IL_0009:  stloc.1
  IL_000a:  readonly.
  IL_000c:  ldelema    "T"
  IL_0011:  pop
  IL_0012:  ldloca.s   V_4
  IL_0014:  initobj    "T"
  IL_001a:  ldloc.s    V_4
  IL_001c:  box        "T"
  IL_0021:  brtrue.s   IL_002f
  IL_0023:  ldloc.0
  IL_0024:  ldloc.1
  IL_0025:  ldelem     "T"
  IL_002a:  stloc.2
  IL_002b:  ldloca.s   V_2
  IL_002d:  br.s       IL_0038
  IL_002f:  ldloc.0
  IL_0030:  ldloc.1
  IL_0031:  readonly.
  IL_0033:  ldelema    "T"
  IL_0038:  ldarg.0
  IL_0039:  call       "Function Program.GetOffset(Of T)(T()) As Integer"
  IL_003e:  dup
  IL_003f:  stloc.3
  IL_0040:  ldloca.s   V_4
  IL_0042:  initobj    "T"
  IL_0048:  ldloc.s    V_4
  IL_004a:  box        "T"
  IL_004f:  brtrue.s   IL_0055
  IL_0051:  ldloca.s   V_2
  IL_0053:  br.s       IL_005e
  IL_0055:  ldloc.0
  IL_0056:  ldloc.1
  IL_0057:  readonly.
  IL_0059:  ldelema    "T"
  IL_005e:  ldloc.3
  IL_005f:  constrained. "T"
  IL_0065:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_006a:  ldc.i4.1
  IL_006b:  add.ovf
  IL_006c:  constrained. "T"
  IL_0072:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0077:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Index_ThroughArray()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = { New Item With {.Name = "1"} }
        Call1(item1)

        Dim item2 = { New Item With {.Name = "2"} }
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(item As T())
        item(GetArrayIndex()).Position(GetOffset(item)) += 1
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T())
        item(GetArrayIndex()).Position(GetOffset(item)) += 1
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(item As T()) As Integer
        value -= 1
        item(0) = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Function GetArrayIndex() As Integer
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-1'
Position get for item '-2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       60 (0x3c)
  .maxstack  4
  .locals init (T() V_0,
                Integer V_1,
                Integer V_2)
  IL_0000:  ldarg.0
  IL_0001:  dup
  IL_0002:  stloc.0
  IL_0003:  call       "Function Program.GetArrayIndex() As Integer"
  IL_0008:  dup
  IL_0009:  stloc.1
  IL_000a:  readonly.
  IL_000c:  ldelema    "T"
  IL_0011:  ldarg.0
  IL_0012:  call       "Function Program.GetOffset(Of T)(T()) As Integer"
  IL_0017:  dup
  IL_0018:  stloc.2
  IL_0019:  ldloc.0
  IL_001a:  ldloc.1
  IL_001b:  readonly.
  IL_001d:  ldelema    "T"
  IL_0022:  ldloc.2
  IL_0023:  constrained. "T"
  IL_0029:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_002e:  ldc.i4.1
  IL_002f:  add.ovf
  IL_0030:  constrained. "T"
  IL_0036:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_003b:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Index_Async_01_ThroughArray()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Item2
    Inherits Item
End Class

Class Program
    Shared Sub Main()
        Dim item1 = { New Item2 With {.Name = "1"} }
        Call1(DirectCast(item1, Item())).Wait()

        Dim item2 = { New Item2 With {.Name = "2"} }
        Call2(DirectCast(item2, Item())).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T()) As Task
        item(GetArrayIndex()).Position(await GetOffsetAsync(GetOffset(item))) += 1
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T()) As Task
        item(GetArrayIndex()).Position(await GetOffsetAsync(GetOffset(item))) += 1
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T()) As Integer
        value -= 1
        item(0) = DirectCast(DirectCast(New Item2 With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Function GetArrayIndex() As Integer
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      260 (0x104)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_006b
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldarg.0
    IL_000d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0012:  call       "Function Program.GetArrayIndex() As Integer"
    IL_0017:  ldelem     "SM$T"
    IL_001c:  dup
    IL_001d:  stloc.3
    IL_001e:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0023:  ldloc.3
    IL_0024:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0029:  ldarg.0
    IL_002a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_002f:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T()) As Integer"
    IL_0034:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0039:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_003e:  stloc.2
    IL_003f:  ldloca.s   V_2
    IL_0041:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0046:  brtrue.s   IL_0087
    IL_0048:  ldarg.0
    IL_0049:  ldc.i4.0
    IL_004a:  dup
    IL_004b:  stloc.0
    IL_004c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0051:  ldarg.0
    IL_0052:  ldloc.2
    IL_0053:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0058:  ldarg.0
    IL_0059:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_005e:  ldloca.s   V_2
    IL_0060:  ldarg.0
    IL_0061:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0066:  leave      IL_0103
    IL_006b:  ldarg.0
    IL_006c:  ldc.i4.m1
    IL_006d:  dup
    IL_006e:  stloc.0
    IL_006f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0074:  ldarg.0
    IL_0075:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007a:  stloc.2
    IL_007b:  ldarg.0
    IL_007c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0081:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0087:  ldarg.0
    IL_0088:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_008d:  box        "SM$T"
    IL_0092:  ldloca.s   V_2
    IL_0094:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0099:  ldloca.s   V_2
    IL_009b:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a1:  dup
    IL_00a2:  stloc.1
    IL_00a3:  ldarg.0
    IL_00a4:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_00a9:  ldloc.1
    IL_00aa:  constrained. "SM$T"
    IL_00b0:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_00b5:  ldc.i4.1
    IL_00b6:  add.ovf
    IL_00b7:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00bc:  ldarg.0
    IL_00bd:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_00c2:  initobj    "SM$T"
    IL_00c8:  leave.s    IL_00ee
  }
  catch System.Exception
  {
    IL_00ca:  dup
    IL_00cb:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00d0:  stloc.s    V_4
    IL_00d2:  ldarg.0
    IL_00d3:  ldc.i4.s   -2
    IL_00d5:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00da:  ldarg.0
    IL_00db:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00e0:  ldloc.s    V_4
    IL_00e2:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00e7:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00ec:  leave.s    IL_0103
  }
  IL_00ee:  ldarg.0
  IL_00ef:  ldc.i4.s   -2
  IL_00f1:  dup
  IL_00f2:  stloc.0
  IL_00f3:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00f8:  ldarg.0
  IL_00f9:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00fe:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0103:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      429 (0x1ad)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T() V_3,
                Integer V_4,
                SM$T V_5,
                System.Exception V_6)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse    IL_00cc
    IL_000d:  ldarg.0
    IL_000e:  ldarg.0
    IL_000f:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T()"
    IL_0014:  dup
    IL_0015:  stloc.3
    IL_0016:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T()"
    IL_001b:  ldloc.3
    IL_001c:  ldarg.0
    IL_001d:  call       "Function Program.GetArrayIndex() As Integer"
    IL_0022:  dup
    IL_0023:  stloc.s    V_4
    IL_0025:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S1 As Integer"
    IL_002a:  ldloc.s    V_4
    IL_002c:  readonly.
    IL_002e:  ldelema    "SM$T"
    IL_0033:  pop
    IL_0034:  ldloca.s   V_5
    IL_0036:  initobj    "SM$T"
    IL_003c:  ldloc.s    V_5
    IL_003e:  box        "SM$T"
    IL_0043:  brfalse.s  IL_0073
    IL_0045:  ldarg.0
    IL_0046:  ldarg.0
    IL_0047:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T()"
    IL_004c:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T()"
    IL_0051:  ldarg.0
    IL_0052:  ldarg.0
    IL_0053:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S1 As Integer"
    IL_0058:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As Integer"
    IL_005d:  ldarg.0
    IL_005e:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T()"
    IL_0063:  ldarg.0
    IL_0064:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As Integer"
    IL_0069:  readonly.
    IL_006b:  ldelema    "SM$T"
    IL_0070:  pop
    IL_0071:  br.s       IL_008a
    IL_0073:  ldarg.0
    IL_0074:  ldarg.0
    IL_0075:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T()"
    IL_007a:  ldarg.0
    IL_007b:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S1 As Integer"
    IL_0080:  ldelem     "SM$T"
    IL_0085:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S2 As SM$T"
    IL_008a:  ldarg.0
    IL_008b:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T()"
    IL_0090:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T()) As Integer"
    IL_0095:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_009a:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009f:  stloc.2
    IL_00a0:  ldloca.s   V_2
    IL_00a2:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_00a7:  brtrue.s   IL_00e8
    IL_00a9:  ldarg.0
    IL_00aa:  ldc.i4.0
    IL_00ab:  dup
    IL_00ac:  stloc.0
    IL_00ad:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00b2:  ldarg.0
    IL_00b3:  ldloc.2
    IL_00b4:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b9:  ldarg.0
    IL_00ba:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00bf:  ldloca.s   V_2
    IL_00c1:  ldarg.0
    IL_00c2:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_00c7:  leave      IL_01ac
    IL_00cc:  ldarg.0
    IL_00cd:  ldc.i4.m1
    IL_00ce:  dup
    IL_00cf:  stloc.0
    IL_00d0:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00d5:  ldarg.0
    IL_00d6:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00db:  stloc.2
    IL_00dc:  ldarg.0
    IL_00dd:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00e2:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00e8:  ldloca.s   V_5
    IL_00ea:  initobj    "SM$T"
    IL_00f0:  ldloc.s    V_5
    IL_00f2:  box        "SM$T"
    IL_00f7:  brtrue.s   IL_0101
    IL_00f9:  ldarg.0
    IL_00fa:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S2 As SM$T"
    IL_00ff:  br.s       IL_0114
    IL_0101:  ldarg.0
    IL_0102:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T()"
    IL_0107:  ldarg.0
    IL_0108:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As Integer"
    IL_010d:  readonly.
    IL_010f:  ldelema    "SM$T"
    IL_0114:  ldloca.s   V_2
    IL_0116:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_011b:  ldloca.s   V_2
    IL_011d:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0123:  dup
    IL_0124:  stloc.1
    IL_0125:  ldloca.s   V_5
    IL_0127:  initobj    "SM$T"
    IL_012d:  ldloc.s    V_5
    IL_012f:  box        "SM$T"
    IL_0134:  brtrue.s   IL_013e
    IL_0136:  ldarg.0
    IL_0137:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S2 As SM$T"
    IL_013c:  br.s       IL_0151
    IL_013e:  ldarg.0
    IL_013f:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T()"
    IL_0144:  ldarg.0
    IL_0145:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S1 As Integer"
    IL_014a:  readonly.
    IL_014c:  ldelema    "SM$T"
    IL_0151:  ldloc.1
    IL_0152:  constrained. "SM$T"
    IL_0158:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_015d:  ldc.i4.1
    IL_015e:  add.ovf
    IL_015f:  constrained. "SM$T"
    IL_0165:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_016a:  ldarg.0
    IL_016b:  ldnull
    IL_016c:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T()"
    IL_0171:  leave.s    IL_0197
  }
  catch System.Exception
  {
    IL_0173:  dup
    IL_0174:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0179:  stloc.s    V_6
    IL_017b:  ldarg.0
    IL_017c:  ldc.i4.s   -2
    IL_017e:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0183:  ldarg.0
    IL_0184:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0189:  ldloc.s    V_6
    IL_018b:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0190:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0195:  leave.s    IL_01ac
  }
  IL_0197:  ldarg.0
  IL_0198:  ldc.i4.s   -2
  IL_019a:  dup
  IL_019b:  stloc.0
  IL_019c:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_01a1:  ldarg.0
  IL_01a2:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_01a7:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_01ac:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Index_Async_01_ThroughArray()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = { New Item With {.Name = "1"} }
        Call1(item1).Wait()

        Dim item2 = { New Item With {.Name = "2"} }
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T()) As Task
        item(GetArrayIndex()).Position(await GetOffsetAsync(GetOffset(item))) += 1
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T()) As Task
        item(GetArrayIndex()).Position(await GetOffsetAsync(GetOffset(item))) += 1
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T()) As Integer
        value -= 1
        item(0) = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Function GetArrayIndex() As Integer
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-1'
Position get for item '-2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      317 (0x13d)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T() V_3,
                Integer V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse    IL_008e
    IL_000d:  ldarg.0
    IL_000e:  ldarg.0
    IL_000f:  ldarg.0
    IL_0010:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0015:  dup
    IL_0016:  stloc.3
    IL_0017:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T()"
    IL_001c:  ldloc.3
    IL_001d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T()"
    IL_0022:  ldarg.0
    IL_0023:  ldarg.0
    IL_0024:  call       "Function Program.GetArrayIndex() As Integer"
    IL_0029:  dup
    IL_002a:  stloc.s    V_4
    IL_002c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S1 As Integer"
    IL_0031:  ldloc.s    V_4
    IL_0033:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As Integer"
    IL_0038:  ldarg.0
    IL_0039:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T()"
    IL_003e:  ldarg.0
    IL_003f:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As Integer"
    IL_0044:  readonly.
    IL_0046:  ldelema    "SM$T"
    IL_004b:  pop
    IL_004c:  ldarg.0
    IL_004d:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0052:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T()) As Integer"
    IL_0057:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_005c:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0061:  stloc.2
    IL_0062:  ldloca.s   V_2
    IL_0064:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0069:  brtrue.s   IL_00aa
    IL_006b:  ldarg.0
    IL_006c:  ldc.i4.0
    IL_006d:  dup
    IL_006e:  stloc.0
    IL_006f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0074:  ldarg.0
    IL_0075:  ldloc.2
    IL_0076:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007b:  ldarg.0
    IL_007c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0081:  ldloca.s   V_2
    IL_0083:  ldarg.0
    IL_0084:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0089:  leave      IL_013c
    IL_008e:  ldarg.0
    IL_008f:  ldc.i4.m1
    IL_0090:  dup
    IL_0091:  stloc.0
    IL_0092:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0097:  ldarg.0
    IL_0098:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009d:  stloc.2
    IL_009e:  ldarg.0
    IL_009f:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a4:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00aa:  ldarg.0
    IL_00ab:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T()"
    IL_00b0:  ldarg.0
    IL_00b1:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As Integer"
    IL_00b6:  readonly.
    IL_00b8:  ldelema    "SM$T"
    IL_00bd:  ldloca.s   V_2
    IL_00bf:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00c4:  ldloca.s   V_2
    IL_00c6:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00cc:  dup
    IL_00cd:  stloc.1
    IL_00ce:  ldarg.0
    IL_00cf:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T()"
    IL_00d4:  ldarg.0
    IL_00d5:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S1 As Integer"
    IL_00da:  readonly.
    IL_00dc:  ldelema    "SM$T"
    IL_00e1:  ldloc.1
    IL_00e2:  constrained. "SM$T"
    IL_00e8:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_00ed:  ldc.i4.1
    IL_00ee:  add.ovf
    IL_00ef:  constrained. "SM$T"
    IL_00f5:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00fa:  ldarg.0
    IL_00fb:  ldnull
    IL_00fc:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T()"
    IL_0101:  leave.s    IL_0127
  }
  catch System.Exception
  {
    IL_0103:  dup
    IL_0104:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0109:  stloc.s    V_5
    IL_010b:  ldarg.0
    IL_010c:  ldc.i4.s   -2
    IL_010e:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0113:  ldarg.0
    IL_0114:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0119:  ldloc.s    V_5
    IL_011b:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0120:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0125:  leave.s    IL_013c
  }
  IL_0127:  ldarg.0
  IL_0128:  ldc.i4.s   -2
  IL_012a:  dup
  IL_012b:  stloc.0
  IL_012c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0131:  ldarg.0
  IL_0132:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0137:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_013c:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Value_ThroughArray()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Item2
    Inherits Item
End Class

Class Program
    Shared Sub Main()
        Dim item1 = { New Item2 With {.Name = "1"} }
        Call1(DirectCast(item1, Item()))

        Dim item2 = { New Item2 With {.Name = "2"} }
        Call2(DirectCast(item2, Item()))
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(item As T())
        item(GetArrayIndex()).Position(1) += GetOffset(item)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T())
        item(GetArrayIndex()).Position(1) += GetOffset(item)
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T()) As Integer
        value -= 1
        item(0) = DirectCast(DirectCast(New Item2 With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Function GetArrayIndex() As Integer
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       47 (0x2f)
  .maxstack  4
  .locals init (T V_0)
  IL_0000:  ldarg.0
  IL_0001:  call       "Function Program.GetArrayIndex() As Integer"
  IL_0006:  ldelem     "T"
  IL_000b:  dup
  IL_000c:  stloc.0
  IL_000d:  box        "T"
  IL_0012:  ldc.i4.1
  IL_0013:  ldloca.s   V_0
  IL_0015:  ldc.i4.1
  IL_0016:  constrained. "T"
  IL_001c:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0021:  ldarga.s   V_0
  IL_0023:  call       "Function Program.GetOffset(Of T)(ByRef T()) As Integer"
  IL_0028:  add.ovf
  IL_0029:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_002e:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
      // Code size       83 (0x53)
  .maxstack  4
  .locals init (T() V_0,
                Integer V_1,
                T V_2)
  IL_0000:  ldarg.0
  IL_0001:  dup
  IL_0002:  stloc.0
  IL_0003:  call       "Function Program.GetArrayIndex() As Integer"
  IL_0008:  dup
  IL_0009:  stloc.1
  IL_000a:  readonly.
  IL_000c:  ldelema    "T"
  IL_0011:  ldloca.s   V_2
  IL_0013:  initobj    "T"
  IL_0019:  ldloc.2
  IL_001a:  box        "T"
  IL_001f:  brtrue.s   IL_0029
  IL_0021:  ldobj      "T"
  IL_0026:  stloc.2
  IL_0027:  ldloca.s   V_2
  IL_0029:  ldc.i4.1
  IL_002a:  ldloc.0
  IL_002b:  ldloc.1
  IL_002c:  readonly.
  IL_002e:  ldelema    "T"
  IL_0033:  ldc.i4.1
  IL_0034:  constrained. "T"
  IL_003a:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_003f:  ldarga.s   V_0
  IL_0041:  call       "Function Program.GetOffset(Of T)(ByRef T()) As Integer"
  IL_0046:  add.ovf
  IL_0047:  constrained. "T"
  IL_004d:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0052:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Value_ThroughArray()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = { New Item With {.Name = "1"} }
        Call1(item1)

        Dim item2 = { New Item With {.Name = "2"} }
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(item As T())
        item(GetArrayIndex()).Position(1) += GetOffset(item)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T())
        item(GetArrayIndex()).Position(1) += GetOffset(item)
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T()) As Integer
        value -= 1
        item(0) = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Function GetArrayIndex() As Integer
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '-1'
Position get for item '2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       59 (0x3b)
  .maxstack  4
  .locals init (T() V_0,
                Integer V_1)
  IL_0000:  ldarg.0
  IL_0001:  dup
  IL_0002:  stloc.0
  IL_0003:  call       "Function Program.GetArrayIndex() As Integer"
  IL_0008:  dup
  IL_0009:  stloc.1
  IL_000a:  readonly.
  IL_000c:  ldelema    "T"
  IL_0011:  ldc.i4.1
  IL_0012:  ldloc.0
  IL_0013:  ldloc.1
  IL_0014:  readonly.
  IL_0016:  ldelema    "T"
  IL_001b:  ldc.i4.1
  IL_001c:  constrained. "T"
  IL_0022:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0027:  ldarga.s   V_0
  IL_0029:  call       "Function Program.GetOffset(Of T)(ByRef T()) As Integer"
  IL_002e:  add.ovf
  IL_002f:  constrained. "T"
  IL_0035:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_003a:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Value_Async_01_ThroughArray()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item2
    Inherits Item
End Class

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = { New Item2 With {.Name = "1"} }
        Call1(DirectCast(item1, Item())).Wait()

        Dim item2 = { New Item2 With {.Name = "2"} }
        Call2(DirectCast(item2, Item())).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T()) As Task
        item(GetArrayIndex()).Position(1) += await GetOffsetAsync(GetOffset(item))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T()) As Task
        item(GetArrayIndex()).Position(1) += await GetOffsetAsync(GetOffset(item))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T()) As Integer
        value -= 1
        item(0) = DirectCast(DirectCast(New Item2 With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Function GetArrayIndex() As Integer
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      268 (0x10c)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                SM$T V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0083
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldarg.0
    IL_000d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0012:  call       "Function Program.GetArrayIndex() As Integer"
    IL_0017:  ldelem     "SM$T"
    IL_001c:  dup
    IL_001d:  stloc.2
    IL_001e:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0023:  ldloc.2
    IL_0024:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_0029:  ldarg.0
    IL_002a:  ldarg.0
    IL_002b:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0030:  ldc.i4.1
    IL_0031:  constrained. "SM$T"
    IL_0037:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_003c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_0041:  ldarg.0
    IL_0042:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0047:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T()) As Integer"
    IL_004c:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0051:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0056:  stloc.1
    IL_0057:  ldloca.s   V_1
    IL_0059:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_005e:  brtrue.s   IL_009f
    IL_0060:  ldarg.0
    IL_0061:  ldc.i4.0
    IL_0062:  dup
    IL_0063:  stloc.0
    IL_0064:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0069:  ldarg.0
    IL_006a:  ldloc.1
    IL_006b:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0070:  ldarg.0
    IL_0071:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0076:  ldloca.s   V_1
    IL_0078:  ldarg.0
    IL_0079:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_007e:  leave      IL_010b
    IL_0083:  ldarg.0
    IL_0084:  ldc.i4.m1
    IL_0085:  dup
    IL_0086:  stloc.0
    IL_0087:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_008c:  ldarg.0
    IL_008d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0092:  stloc.1
    IL_0093:  ldarg.0
    IL_0094:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0099:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009f:  ldarg.0
    IL_00a0:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_00a5:  box        "SM$T"
    IL_00aa:  ldc.i4.1
    IL_00ab:  ldarg.0
    IL_00ac:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_00b1:  ldloca.s   V_1
    IL_00b3:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00b8:  ldloca.s   V_1
    IL_00ba:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00c0:  add.ovf
    IL_00c1:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00c6:  ldarg.0
    IL_00c7:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_00cc:  initobj    "SM$T"
    IL_00d2:  leave.s    IL_00f6
  }
  catch System.Exception
  {
    IL_00d4:  dup
    IL_00d5:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00da:  stloc.3
    IL_00db:  ldarg.0
    IL_00dc:  ldc.i4.s   -2
    IL_00de:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00e3:  ldarg.0
    IL_00e4:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00e9:  ldloc.3
    IL_00ea:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00ef:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00f4:  leave.s    IL_010b
  }
  IL_00f6:  ldarg.0
  IL_00f7:  ldc.i4.s   -2
  IL_00f9:  dup
  IL_00fa:  stloc.0
  IL_00fb:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0100:  ldarg.0
  IL_0101:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0106:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_010b:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      402 (0x192)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                SM$T() V_2,
                Integer V_3,
                SM$T V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse    IL_00d9
    IL_000d:  ldarg.0
    IL_000e:  ldarg.0
    IL_000f:  ldarg.0
    IL_0010:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T()"
    IL_0015:  dup
    IL_0016:  stloc.2
    IL_0017:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T()"
    IL_001c:  ldloc.2
    IL_001d:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T()"
    IL_0022:  ldarg.0
    IL_0023:  ldarg.0
    IL_0024:  call       "Function Program.GetArrayIndex() As Integer"
    IL_0029:  dup
    IL_002a:  stloc.3
    IL_002b:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S1 As Integer"
    IL_0030:  ldloc.3
    IL_0031:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U3 As Integer"
    IL_0036:  ldarg.0
    IL_0037:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T()"
    IL_003c:  ldarg.0
    IL_003d:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U3 As Integer"
    IL_0042:  readonly.
    IL_0044:  ldelema    "SM$T"
    IL_0049:  pop
    IL_004a:  ldloca.s   V_4
    IL_004c:  initobj    "SM$T"
    IL_0052:  ldloc.s    V_4
    IL_0054:  box        "SM$T"
    IL_0059:  brtrue.s   IL_0072
    IL_005b:  ldarg.0
    IL_005c:  ldarg.0
    IL_005d:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T()"
    IL_0062:  ldarg.0
    IL_0063:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U3 As Integer"
    IL_0068:  ldelem     "SM$T"
    IL_006d:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U4 As SM$T"
    IL_0072:  ldarg.0
    IL_0073:  ldarg.0
    IL_0074:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T()"
    IL_0079:  ldarg.0
    IL_007a:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S1 As Integer"
    IL_007f:  readonly.
    IL_0081:  ldelema    "SM$T"
    IL_0086:  ldc.i4.1
    IL_0087:  constrained. "SM$T"
    IL_008d:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_0092:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As Integer"
    IL_0097:  ldarg.0
    IL_0098:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T()"
    IL_009d:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T()) As Integer"
    IL_00a2:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_00a7:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00ac:  stloc.1
    IL_00ad:  ldloca.s   V_1
    IL_00af:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_00b4:  brtrue.s   IL_00f5
    IL_00b6:  ldarg.0
    IL_00b7:  ldc.i4.0
    IL_00b8:  dup
    IL_00b9:  stloc.0
    IL_00ba:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00bf:  ldarg.0
    IL_00c0:  ldloc.1
    IL_00c1:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00c6:  ldarg.0
    IL_00c7:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00cc:  ldloca.s   V_1
    IL_00ce:  ldarg.0
    IL_00cf:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_00d4:  leave      IL_0191
    IL_00d9:  ldarg.0
    IL_00da:  ldc.i4.m1
    IL_00db:  dup
    IL_00dc:  stloc.0
    IL_00dd:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00e2:  ldarg.0
    IL_00e3:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00e8:  stloc.1
    IL_00e9:  ldarg.0
    IL_00ea:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00ef:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00f5:  ldloca.s   V_4
    IL_00f7:  initobj    "SM$T"
    IL_00fd:  ldloc.s    V_4
    IL_00ff:  box        "SM$T"
    IL_0104:  brtrue.s   IL_010e
    IL_0106:  ldarg.0
    IL_0107:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U4 As SM$T"
    IL_010c:  br.s       IL_0121
    IL_010e:  ldarg.0
    IL_010f:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T()"
    IL_0114:  ldarg.0
    IL_0115:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U3 As Integer"
    IL_011a:  readonly.
    IL_011c:  ldelema    "SM$T"
    IL_0121:  ldc.i4.1
    IL_0122:  ldarg.0
    IL_0123:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As Integer"
    IL_0128:  ldloca.s   V_1
    IL_012a:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_012f:  ldloca.s   V_1
    IL_0131:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0137:  add.ovf
    IL_0138:  constrained. "SM$T"
    IL_013e:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_0143:  ldarg.0
    IL_0144:  ldnull
    IL_0145:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T()"
    IL_014a:  ldarg.0
    IL_014b:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U4 As SM$T"
    IL_0150:  initobj    "SM$T"
    IL_0156:  leave.s    IL_017c
  }
  catch System.Exception
  {
    IL_0158:  dup
    IL_0159:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_015e:  stloc.s    V_5
    IL_0160:  ldarg.0
    IL_0161:  ldc.i4.s   -2
    IL_0163:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0168:  ldarg.0
    IL_0169:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_016e:  ldloc.s    V_5
    IL_0170:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0175:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_017a:  leave.s    IL_0191
  }
  IL_017c:  ldarg.0
  IL_017d:  ldc.i4.s   -2
  IL_017f:  dup
  IL_0180:  stloc.0
  IL_0181:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0186:  ldarg.0
  IL_0187:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_018c:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0191:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Value_Async_01_ThroughArray()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = { New Item With {.Name = "1"} }
        Call1(item1).Wait()

        Dim item2 = { New Item With {.Name = "2"} }
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T()) As Task
        item(GetArrayIndex()).Position(1) += await GetOffsetAsync(GetOffset(item))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T()) As Task
        item(GetArrayIndex()).Position(1) += await GetOffsetAsync(GetOffset(item))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T()) As Integer
        value -= 1
        item(0) = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Function GetArrayIndex() As Integer
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '-1'
Position get for item '2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      325 (0x145)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                SM$T() V_2,
                Integer V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse    IL_00b1
    IL_000d:  ldarg.0
    IL_000e:  ldarg.0
    IL_000f:  ldarg.0
    IL_0010:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0015:  dup
    IL_0016:  stloc.2
    IL_0017:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T()"
    IL_001c:  ldloc.2
    IL_001d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T()"
    IL_0022:  ldarg.0
    IL_0023:  ldarg.0
    IL_0024:  call       "Function Program.GetArrayIndex() As Integer"
    IL_0029:  dup
    IL_002a:  stloc.3
    IL_002b:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S1 As Integer"
    IL_0030:  ldloc.3
    IL_0031:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U3 As Integer"
    IL_0036:  ldarg.0
    IL_0037:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T()"
    IL_003c:  ldarg.0
    IL_003d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U3 As Integer"
    IL_0042:  readonly.
    IL_0044:  ldelema    "SM$T"
    IL_0049:  pop
    IL_004a:  ldarg.0
    IL_004b:  ldarg.0
    IL_004c:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T()"
    IL_0051:  ldarg.0
    IL_0052:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S1 As Integer"
    IL_0057:  readonly.
    IL_0059:  ldelema    "SM$T"
    IL_005e:  ldc.i4.1
    IL_005f:  constrained. "SM$T"
    IL_0065:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_006a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_006f:  ldarg.0
    IL_0070:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0075:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T()) As Integer"
    IL_007a:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_007f:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0084:  stloc.1
    IL_0085:  ldloca.s   V_1
    IL_0087:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_008c:  brtrue.s   IL_00cd
    IL_008e:  ldarg.0
    IL_008f:  ldc.i4.0
    IL_0090:  dup
    IL_0091:  stloc.0
    IL_0092:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0097:  ldarg.0
    IL_0098:  ldloc.1
    IL_0099:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009e:  ldarg.0
    IL_009f:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00a4:  ldloca.s   V_1
    IL_00a6:  ldarg.0
    IL_00a7:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_00ac:  leave      IL_0144
    IL_00b1:  ldarg.0
    IL_00b2:  ldc.i4.m1
    IL_00b3:  dup
    IL_00b4:  stloc.0
    IL_00b5:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00ba:  ldarg.0
    IL_00bb:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00c0:  stloc.1
    IL_00c1:  ldarg.0
    IL_00c2:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00c7:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00cd:  ldarg.0
    IL_00ce:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T()"
    IL_00d3:  ldarg.0
    IL_00d4:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U3 As Integer"
    IL_00d9:  readonly.
    IL_00db:  ldelema    "SM$T"
    IL_00e0:  ldc.i4.1
    IL_00e1:  ldarg.0
    IL_00e2:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_00e7:  ldloca.s   V_1
    IL_00e9:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00ee:  ldloca.s   V_1
    IL_00f0:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00f6:  add.ovf
    IL_00f7:  constrained. "SM$T"
    IL_00fd:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_0102:  ldarg.0
    IL_0103:  ldnull
    IL_0104:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T()"
    IL_0109:  leave.s    IL_012f
  }
  catch System.Exception
  {
    IL_010b:  dup
    IL_010c:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0111:  stloc.s    V_4
    IL_0113:  ldarg.0
    IL_0114:  ldc.i4.s   -2
    IL_0116:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_011b:  ldarg.0
    IL_011c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0121:  ldloc.s    V_4
    IL_0123:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0128:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_012d:  leave.s    IL_0144
  }
  IL_012f:  ldarg.0
  IL_0130:  ldc.i4.s   -2
  IL_0132:  dup
  IL_0133:  stloc.0
  IL_0134:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0139:  ldarg.0
  IL_013a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_013f:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0144:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Index_ThroughArray_InWith()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Item2
    Inherits Item
End Class

Class Program
    Shared Sub Main()
        Dim item1 = { New Item2 With {.Name = "1"} }
        Call1(DirectCast(item1, Item()))

        Dim item2 = { New Item2 With {.Name = "2"} }
        Call2(DirectCast(item2, Item()))
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(item As T())
        With item(GetArrayIndex())
            .Position(GetOffset(item)) += 1
        End With
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T())
        With item(GetArrayIndex())
            .Position(GetOffset(item)) += 1
        End With
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(item As T()) As Integer
        value -= 1
        item(0) = DirectCast(DirectCast(New Item2 With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Function GetArrayIndex() As Integer
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       48 (0x30)
  .maxstack  4
  .locals init (T V_0,
                Integer V_1)
  IL_0000:  ldarg.0
  IL_0001:  call       "Function Program.GetArrayIndex() As Integer"
  IL_0006:  ldelem     "T"
  IL_000b:  dup
  IL_000c:  stloc.0
  IL_000d:  box        "T"
  IL_0012:  ldarg.0
  IL_0013:  call       "Function Program.GetOffset(Of T)(T()) As Integer"
  IL_0018:  dup
  IL_0019:  stloc.1
  IL_001a:  ldloca.s   V_0
  IL_001c:  ldloc.1
  IL_001d:  constrained. "T"
  IL_0023:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0028:  ldc.i4.1
  IL_0029:  add.ovf
  IL_002a:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_002f:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
      // Code size      122 (0x7a)
  .maxstack  4
  .locals init (T() V_0, //$W0
                Integer V_1, //$W1
                T V_2,
                Integer V_3,
                T V_4)
  IL_0000:  ldarg.0
  IL_0001:  stloc.0
  IL_0002:  call       "Function Program.GetArrayIndex() As Integer"
  IL_0007:  stloc.1
  IL_0008:  ldloc.0
  IL_0009:  ldloc.1
  IL_000a:  readonly.
  IL_000c:  ldelema    "T"
  IL_0011:  pop
  IL_0012:  ldloca.s   V_4
  IL_0014:  initobj    "T"
  IL_001a:  ldloc.s    V_4
  IL_001c:  box        "T"
  IL_0021:  brtrue.s   IL_002f
  IL_0023:  ldloc.0
  IL_0024:  ldloc.1
  IL_0025:  ldelem     "T"
  IL_002a:  stloc.2
  IL_002b:  ldloca.s   V_2
  IL_002d:  br.s       IL_0038
  IL_002f:  ldloc.0
  IL_0030:  ldloc.1
  IL_0031:  readonly.
  IL_0033:  ldelema    "T"
  IL_0038:  ldarg.0
  IL_0039:  call       "Function Program.GetOffset(Of T)(T()) As Integer"
  IL_003e:  dup
  IL_003f:  stloc.3
  IL_0040:  ldloca.s   V_4
  IL_0042:  initobj    "T"
  IL_0048:  ldloc.s    V_4
  IL_004a:  box        "T"
  IL_004f:  brtrue.s   IL_0055
  IL_0051:  ldloca.s   V_2
  IL_0053:  br.s       IL_005e
  IL_0055:  ldloc.0
  IL_0056:  ldloc.1
  IL_0057:  readonly.
  IL_0059:  ldelema    "T"
  IL_005e:  ldloc.3
  IL_005f:  constrained. "T"
  IL_0065:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_006a:  ldc.i4.1
  IL_006b:  add.ovf
  IL_006c:  constrained. "T"
  IL_0072:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0077:  ldnull
  IL_0078:  stloc.0
  IL_0079:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Index_ThroughArray_InWith()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = { New Item With {.Name = "1"} }
        Call1(item1)

        Dim item2 = { New Item With {.Name = "2"} }
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(item As T())
        With item(GetArrayIndex())
            .Position(GetOffset(item)) += 1
        End With
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T())
        With item(GetArrayIndex())
            .Position(GetOffset(item)) += 1
        End With
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(item As T()) As Integer
        value -= 1
        item(0) = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Function GetArrayIndex() As Integer
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-1'
Position get for item '-2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       62 (0x3e)
  .maxstack  4
  .locals init (T() V_0, //$W0
                Integer V_1, //$W1
                Integer V_2)
  IL_0000:  ldarg.0
  IL_0001:  stloc.0
  IL_0002:  call       "Function Program.GetArrayIndex() As Integer"
  IL_0007:  stloc.1
  IL_0008:  ldloc.0
  IL_0009:  ldloc.1
  IL_000a:  readonly.
  IL_000c:  ldelema    "T"
  IL_0011:  ldarg.0
  IL_0012:  call       "Function Program.GetOffset(Of T)(T()) As Integer"
  IL_0017:  dup
  IL_0018:  stloc.2
  IL_0019:  ldloc.0
  IL_001a:  ldloc.1
  IL_001b:  readonly.
  IL_001d:  ldelema    "T"
  IL_0022:  ldloc.2
  IL_0023:  constrained. "T"
  IL_0029:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_002e:  ldc.i4.1
  IL_002f:  add.ovf
  IL_0030:  constrained. "T"
  IL_0036:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_003b:  ldnull
  IL_003c:  stloc.0
  IL_003d:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Index_Async_01_ThroughArray_InWith()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Item2
    Inherits Item
End Class

Class Program
    Shared Sub Main()
        Dim item1 = { New Item2 With {.Name = "1"} }
        Call1(DirectCast(item1, Item())).Wait()

        Dim item2 = { New Item2 With {.Name = "2"} }
        Call2(DirectCast(item2, Item())).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T()) As Task
        With item(GetArrayIndex())
            .Position(await GetOffsetAsync(GetOffset(item))) += 1
        End With
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T()) As Task
        With item(GetArrayIndex())
            .Position(await GetOffsetAsync(GetOffset(item))) += 1
        End With
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T()) As Integer
        value -= 1
        item(0) = DirectCast(DirectCast(New Item2 With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Function GetArrayIndex() As Integer
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      272 (0x110)
  .maxstack  4
  .locals init (Integer V_0,
                SM$T V_1, //$W0
                Integer V_2,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_3,
                SM$T V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_006f
    IL_000a:  ldarg.0
    IL_000b:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0010:  call       "Function Program.GetArrayIndex() As Integer"
    IL_0015:  ldelem     "SM$T"
    IL_001a:  stloc.1
    IL_001b:  ldarg.0
    IL_001c:  ldarg.0
    IL_001d:  ldloc.1
    IL_001e:  dup
    IL_001f:  stloc.s    V_4
    IL_0021:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0026:  ldloc.s    V_4
    IL_0028:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_002d:  ldarg.0
    IL_002e:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0033:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T()) As Integer"
    IL_0038:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_003d:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0042:  stloc.3
    IL_0043:  ldloca.s   V_3
    IL_0045:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_004a:  brtrue.s   IL_008b
    IL_004c:  ldarg.0
    IL_004d:  ldc.i4.0
    IL_004e:  dup
    IL_004f:  stloc.0
    IL_0050:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0055:  ldarg.0
    IL_0056:  ldloc.3
    IL_0057:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_005c:  ldarg.0
    IL_005d:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0062:  ldloca.s   V_3
    IL_0064:  ldarg.0
    IL_0065:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_006a:  leave      IL_010f
    IL_006f:  ldarg.0
    IL_0070:  ldc.i4.m1
    IL_0071:  dup
    IL_0072:  stloc.0
    IL_0073:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0078:  ldarg.0
    IL_0079:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007e:  stloc.3
    IL_007f:  ldarg.0
    IL_0080:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0085:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008b:  ldarg.0
    IL_008c:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0091:  box        "SM$T"
    IL_0096:  ldloca.s   V_3
    IL_0098:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_009d:  ldloca.s   V_3
    IL_009f:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a5:  dup
    IL_00a6:  stloc.2
    IL_00a7:  ldarg.0
    IL_00a8:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_00ad:  ldloc.2
    IL_00ae:  constrained. "SM$T"
    IL_00b4:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_00b9:  ldc.i4.1
    IL_00ba:  add.ovf
    IL_00bb:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00c0:  ldarg.0
    IL_00c1:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_00c6:  initobj    "SM$T"
    IL_00cc:  ldloca.s   V_1
    IL_00ce:  initobj    "SM$T"
    IL_00d4:  leave.s    IL_00fa
  }
  catch System.Exception
  {
    IL_00d6:  dup
    IL_00d7:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00dc:  stloc.s    V_5
    IL_00de:  ldarg.0
    IL_00df:  ldc.i4.s   -2
    IL_00e1:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00e6:  ldarg.0
    IL_00e7:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00ec:  ldloc.s    V_5
    IL_00ee:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00f3:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00f8:  leave.s    IL_010f
  }
  IL_00fa:  ldarg.0
  IL_00fb:  ldc.i4.s   -2
  IL_00fd:  dup
  IL_00fe:  stloc.0
  IL_00ff:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0104:  ldarg.0
  IL_0105:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_010a:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_010f:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      437 (0x1b5)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse    IL_00cf
    IL_000d:  ldarg.0
    IL_000e:  ldarg.0
    IL_000f:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T()"
    IL_0014:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$W0 As SM$T()"
    IL_0019:  ldarg.0
    IL_001a:  call       "Function Program.GetArrayIndex() As Integer"
    IL_001f:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$W1 As Integer"
    IL_0024:  ldarg.0
    IL_0025:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$W0 As SM$T()"
    IL_002a:  ldarg.0
    IL_002b:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$W1 As Integer"
    IL_0030:  readonly.
    IL_0032:  ldelema    "SM$T"
    IL_0037:  pop
    IL_0038:  ldloca.s   V_3
    IL_003a:  initobj    "SM$T"
    IL_0040:  ldloc.3
    IL_0041:  box        "SM$T"
    IL_0046:  brfalse.s  IL_0076
    IL_0048:  ldarg.0
    IL_0049:  ldarg.0
    IL_004a:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$W0 As SM$T()"
    IL_004f:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T()"
    IL_0054:  ldarg.0
    IL_0055:  ldarg.0
    IL_0056:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$W1 As Integer"
    IL_005b:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As Integer"
    IL_0060:  ldarg.0
    IL_0061:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T()"
    IL_0066:  ldarg.0
    IL_0067:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As Integer"
    IL_006c:  readonly.
    IL_006e:  ldelema    "SM$T"
    IL_0073:  pop
    IL_0074:  br.s       IL_008d
    IL_0076:  ldarg.0
    IL_0077:  ldarg.0
    IL_0078:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$W0 As SM$T()"
    IL_007d:  ldarg.0
    IL_007e:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$W1 As Integer"
    IL_0083:  ldelem     "SM$T"
    IL_0088:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S2 As SM$T"
    IL_008d:  ldarg.0
    IL_008e:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T()"
    IL_0093:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T()) As Integer"
    IL_0098:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_009d:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a2:  stloc.2
    IL_00a3:  ldloca.s   V_2
    IL_00a5:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_00aa:  brtrue.s   IL_00eb
    IL_00ac:  ldarg.0
    IL_00ad:  ldc.i4.0
    IL_00ae:  dup
    IL_00af:  stloc.0
    IL_00b0:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00b5:  ldarg.0
    IL_00b6:  ldloc.2
    IL_00b7:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00bc:  ldarg.0
    IL_00bd:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00c2:  ldloca.s   V_2
    IL_00c4:  ldarg.0
    IL_00c5:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_00ca:  leave      IL_01b4
    IL_00cf:  ldarg.0
    IL_00d0:  ldc.i4.m1
    IL_00d1:  dup
    IL_00d2:  stloc.0
    IL_00d3:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00d8:  ldarg.0
    IL_00d9:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00de:  stloc.2
    IL_00df:  ldarg.0
    IL_00e0:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00e5:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00eb:  ldloca.s   V_3
    IL_00ed:  initobj    "SM$T"
    IL_00f3:  ldloc.3
    IL_00f4:  box        "SM$T"
    IL_00f9:  brtrue.s   IL_0103
    IL_00fb:  ldarg.0
    IL_00fc:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S2 As SM$T"
    IL_0101:  br.s       IL_0116
    IL_0103:  ldarg.0
    IL_0104:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T()"
    IL_0109:  ldarg.0
    IL_010a:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As Integer"
    IL_010f:  readonly.
    IL_0111:  ldelema    "SM$T"
    IL_0116:  ldloca.s   V_2
    IL_0118:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_011d:  ldloca.s   V_2
    IL_011f:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0125:  dup
    IL_0126:  stloc.1
    IL_0127:  ldloca.s   V_3
    IL_0129:  initobj    "SM$T"
    IL_012f:  ldloc.3
    IL_0130:  box        "SM$T"
    IL_0135:  brtrue.s   IL_013f
    IL_0137:  ldarg.0
    IL_0138:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S2 As SM$T"
    IL_013d:  br.s       IL_0152
    IL_013f:  ldarg.0
    IL_0140:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$W0 As SM$T()"
    IL_0145:  ldarg.0
    IL_0146:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$W1 As Integer"
    IL_014b:  readonly.
    IL_014d:  ldelema    "SM$T"
    IL_0152:  ldloc.1
    IL_0153:  constrained. "SM$T"
    IL_0159:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_015e:  ldc.i4.1
    IL_015f:  add.ovf
    IL_0160:  constrained. "SM$T"
    IL_0166:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_016b:  ldarg.0
    IL_016c:  ldnull
    IL_016d:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As SM$T()"
    IL_0172:  ldarg.0
    IL_0173:  ldnull
    IL_0174:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$W0 As SM$T()"
    IL_0179:  leave.s    IL_019f
  }
  catch System.Exception
  {
    IL_017b:  dup
    IL_017c:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0181:  stloc.s    V_4
    IL_0183:  ldarg.0
    IL_0184:  ldc.i4.s   -2
    IL_0186:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_018b:  ldarg.0
    IL_018c:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0191:  ldloc.s    V_4
    IL_0193:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0198:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_019d:  leave.s    IL_01b4
  }
  IL_019f:  ldarg.0
  IL_01a0:  ldc.i4.s   -2
  IL_01a2:  dup
  IL_01a3:  stloc.0
  IL_01a4:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_01a9:  ldarg.0
  IL_01aa:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_01af:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_01b4:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Index_Async_01_ThroughArray_InWith()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = { New Item With {.Name = "1"} }
        Call1(item1).Wait()

        Dim item2 = { New Item With {.Name = "2"} }
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T()) As Task
        With item(GetArrayIndex())
            .Position(await GetOffsetAsync(GetOffset(item))) += 1
        End With
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T()) As Task
        With item(GetArrayIndex())
            .Position(await GetOffsetAsync(GetOffset(item))) += 1
        End With
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T()) As Integer
        value -= 1
        item(0) = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Function GetArrayIndex() As Integer
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-1'
Position get for item '-2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      326 (0x146)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse    IL_0092
    IL_000d:  ldarg.0
    IL_000e:  ldarg.0
    IL_000f:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0014:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$W0 As SM$T()"
    IL_0019:  ldarg.0
    IL_001a:  call       "Function Program.GetArrayIndex() As Integer"
    IL_001f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$W1 As Integer"
    IL_0024:  ldarg.0
    IL_0025:  ldarg.0
    IL_0026:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$W0 As SM$T()"
    IL_002b:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T()"
    IL_0030:  ldarg.0
    IL_0031:  ldarg.0
    IL_0032:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$W1 As Integer"
    IL_0037:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As Integer"
    IL_003c:  ldarg.0
    IL_003d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T()"
    IL_0042:  ldarg.0
    IL_0043:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As Integer"
    IL_0048:  readonly.
    IL_004a:  ldelema    "SM$T"
    IL_004f:  pop
    IL_0050:  ldarg.0
    IL_0051:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0056:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T()) As Integer"
    IL_005b:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0060:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0065:  stloc.2
    IL_0066:  ldloca.s   V_2
    IL_0068:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_006d:  brtrue.s   IL_00ae
    IL_006f:  ldarg.0
    IL_0070:  ldc.i4.0
    IL_0071:  dup
    IL_0072:  stloc.0
    IL_0073:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0078:  ldarg.0
    IL_0079:  ldloc.2
    IL_007a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007f:  ldarg.0
    IL_0080:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0085:  ldloca.s   V_2
    IL_0087:  ldarg.0
    IL_0088:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_008d:  leave      IL_0145
    IL_0092:  ldarg.0
    IL_0093:  ldc.i4.m1
    IL_0094:  dup
    IL_0095:  stloc.0
    IL_0096:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_009b:  ldarg.0
    IL_009c:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a1:  stloc.2
    IL_00a2:  ldarg.0
    IL_00a3:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a8:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00ae:  ldarg.0
    IL_00af:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T()"
    IL_00b4:  ldarg.0
    IL_00b5:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As Integer"
    IL_00ba:  readonly.
    IL_00bc:  ldelema    "SM$T"
    IL_00c1:  ldloca.s   V_2
    IL_00c3:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00c8:  ldloca.s   V_2
    IL_00ca:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d0:  dup
    IL_00d1:  stloc.1
    IL_00d2:  ldarg.0
    IL_00d3:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$W0 As SM$T()"
    IL_00d8:  ldarg.0
    IL_00d9:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$W1 As Integer"
    IL_00de:  readonly.
    IL_00e0:  ldelema    "SM$T"
    IL_00e5:  ldloc.1
    IL_00e6:  constrained. "SM$T"
    IL_00ec:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_00f1:  ldc.i4.1
    IL_00f2:  add.ovf
    IL_00f3:  constrained. "SM$T"
    IL_00f9:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00fe:  ldarg.0
    IL_00ff:  ldnull
    IL_0100:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T()"
    IL_0105:  ldarg.0
    IL_0106:  ldnull
    IL_0107:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$W0 As SM$T()"
    IL_010c:  leave.s    IL_0130
  }
  catch System.Exception
  {
    IL_010e:  dup
    IL_010f:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0114:  stloc.3
    IL_0115:  ldarg.0
    IL_0116:  ldc.i4.s   -2
    IL_0118:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_011d:  ldarg.0
    IL_011e:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0123:  ldloc.3
    IL_0124:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0129:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_012e:  leave.s    IL_0145
  }
  IL_0130:  ldarg.0
  IL_0131:  ldc.i4.s   -2
  IL_0133:  dup
  IL_0134:  stloc.0
  IL_0135:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_013a:  ldarg.0
  IL_013b:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0140:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0145:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Value_ThroughArray_InWith()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Item2
    Inherits Item
End Class

Class Program
    Shared Sub Main()
        Dim item1 = { New Item2 With {.Name = "1"} }
        Call1(DirectCast(item1, Item()))

        Dim item2 = { New Item2 With {.Name = "2"} }
        Call2(DirectCast(item2, Item()))
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(item As T())
        With item(GetArrayIndex())
            .Position(1) += GetOffset(item)
        End With
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T())
        With item(GetArrayIndex())
            .Position(1) += GetOffset(item)
        End With
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T()) As Integer
        value -= 1
        item(0) = DirectCast(DirectCast(New Item2 With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Function GetArrayIndex() As Integer
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       47 (0x2f)
  .maxstack  4
  .locals init (T V_0)
  IL_0000:  ldarg.0
  IL_0001:  call       "Function Program.GetArrayIndex() As Integer"
  IL_0006:  ldelem     "T"
  IL_000b:  dup
  IL_000c:  stloc.0
  IL_000d:  box        "T"
  IL_0012:  ldc.i4.1
  IL_0013:  ldloca.s   V_0
  IL_0015:  ldc.i4.1
  IL_0016:  constrained. "T"
  IL_001c:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0021:  ldarga.s   V_0
  IL_0023:  call       "Function Program.GetOffset(Of T)(ByRef T()) As Integer"
  IL_0028:  add.ovf
  IL_0029:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_002e:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
      // Code size       85 (0x55)
  .maxstack  4
  .locals init (T() V_0, //$W0
                Integer V_1, //$W1
                T V_2)
  IL_0000:  ldarg.0
  IL_0001:  stloc.0
  IL_0002:  call       "Function Program.GetArrayIndex() As Integer"
  IL_0007:  stloc.1
  IL_0008:  ldloc.0
  IL_0009:  ldloc.1
  IL_000a:  readonly.
  IL_000c:  ldelema    "T"
  IL_0011:  ldloca.s   V_2
  IL_0013:  initobj    "T"
  IL_0019:  ldloc.2
  IL_001a:  box        "T"
  IL_001f:  brtrue.s   IL_0029
  IL_0021:  ldobj      "T"
  IL_0026:  stloc.2
  IL_0027:  ldloca.s   V_2
  IL_0029:  ldc.i4.1
  IL_002a:  ldloc.0
  IL_002b:  ldloc.1
  IL_002c:  readonly.
  IL_002e:  ldelema    "T"
  IL_0033:  ldc.i4.1
  IL_0034:  constrained. "T"
  IL_003a:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_003f:  ldarga.s   V_0
  IL_0041:  call       "Function Program.GetOffset(Of T)(ByRef T()) As Integer"
  IL_0046:  add.ovf
  IL_0047:  constrained. "T"
  IL_004d:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0052:  ldnull
  IL_0053:  stloc.0
  IL_0054:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Value_ThroughArray_InWith()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = { New Item With {.Name = "1"} }
        Call1(item1)

        Dim item2 = { New Item With {.Name = "2"} }
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(item As T())
        With item(GetArrayIndex())
            .Position(1) += GetOffset(item)
        End With
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T())
        With item(GetArrayIndex())
            .Position(1) += GetOffset(item)
        End With
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T()) As Integer
        value -= 1
        item(0) = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Function GetArrayIndex() As Integer
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '-1'
Position get for item '2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       61 (0x3d)
  .maxstack  4
  .locals init (T() V_0, //$W0
                Integer V_1) //$W1
  IL_0000:  ldarg.0
  IL_0001:  stloc.0
  IL_0002:  call       "Function Program.GetArrayIndex() As Integer"
  IL_0007:  stloc.1
  IL_0008:  ldloc.0
  IL_0009:  ldloc.1
  IL_000a:  readonly.
  IL_000c:  ldelema    "T"
  IL_0011:  ldc.i4.1
  IL_0012:  ldloc.0
  IL_0013:  ldloc.1
  IL_0014:  readonly.
  IL_0016:  ldelema    "T"
  IL_001b:  ldc.i4.1
  IL_001c:  constrained. "T"
  IL_0022:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0027:  ldarga.s   V_0
  IL_0029:  call       "Function Program.GetOffset(Of T)(ByRef T()) As Integer"
  IL_002e:  add.ovf
  IL_002f:  constrained. "T"
  IL_0035:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_003a:  ldnull
  IL_003b:  stloc.0
  IL_003c:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Value_Async_01_ThroughArray_InWith()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item2
    Inherits Item
End Class

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = { New Item2 With {.Name = "1"} }
        Call1(DirectCast(item1, Item())).Wait()

        Dim item2 = { New Item2 With {.Name = "2"} }
        Call2(DirectCast(item2, Item())).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T()) As Task
        With item(GetArrayIndex())
            .Position(1) += await GetOffsetAsync(GetOffset(item))
        End With
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T()) As Task
        With item(GetArrayIndex())
            .Position(1) += await GetOffsetAsync(GetOffset(item))
        End With
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T()) As Integer
        value -= 1
        item(0) = DirectCast(DirectCast(New Item2 With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Function GetArrayIndex() As Integer
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      280 (0x118)
  .maxstack  5
  .locals init (Integer V_0,
                SM$T V_1, //$W0
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0085
    IL_000a:  ldarg.0
    IL_000b:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0010:  call       "Function Program.GetArrayIndex() As Integer"
    IL_0015:  ldelem     "SM$T"
    IL_001a:  stloc.1
    IL_001b:  ldarg.0
    IL_001c:  ldarg.0
    IL_001d:  ldloc.1
    IL_001e:  dup
    IL_001f:  stloc.3
    IL_0020:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0025:  ldloc.3
    IL_0026:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_002b:  ldarg.0
    IL_002c:  ldarg.0
    IL_002d:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0032:  ldc.i4.1
    IL_0033:  constrained. "SM$T"
    IL_0039:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_003e:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_0043:  ldarg.0
    IL_0044:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0049:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T()) As Integer"
    IL_004e:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0053:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0058:  stloc.2
    IL_0059:  ldloca.s   V_2
    IL_005b:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0060:  brtrue.s   IL_00a1
    IL_0062:  ldarg.0
    IL_0063:  ldc.i4.0
    IL_0064:  dup
    IL_0065:  stloc.0
    IL_0066:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_006b:  ldarg.0
    IL_006c:  ldloc.2
    IL_006d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0072:  ldarg.0
    IL_0073:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0078:  ldloca.s   V_2
    IL_007a:  ldarg.0
    IL_007b:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0080:  leave      IL_0117
    IL_0085:  ldarg.0
    IL_0086:  ldc.i4.m1
    IL_0087:  dup
    IL_0088:  stloc.0
    IL_0089:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_008e:  ldarg.0
    IL_008f:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0094:  stloc.2
    IL_0095:  ldarg.0
    IL_0096:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009b:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a1:  ldarg.0
    IL_00a2:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_00a7:  box        "SM$T"
    IL_00ac:  ldc.i4.1
    IL_00ad:  ldarg.0
    IL_00ae:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_00b3:  ldloca.s   V_2
    IL_00b5:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00ba:  ldloca.s   V_2
    IL_00bc:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00c2:  add.ovf
    IL_00c3:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00c8:  ldarg.0
    IL_00c9:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_00ce:  initobj    "SM$T"
    IL_00d4:  ldloca.s   V_1
    IL_00d6:  initobj    "SM$T"
    IL_00dc:  leave.s    IL_0102
  }
  catch System.Exception
  {
    IL_00de:  dup
    IL_00df:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00e4:  stloc.s    V_4
    IL_00e6:  ldarg.0
    IL_00e7:  ldc.i4.s   -2
    IL_00e9:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00ee:  ldarg.0
    IL_00ef:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00f4:  ldloc.s    V_4
    IL_00f6:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00fb:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0100:  leave.s    IL_0117
  }
  IL_0102:  ldarg.0
  IL_0103:  ldc.i4.s   -2
  IL_0105:  dup
  IL_0106:  stloc.0
  IL_0107:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_010c:  ldarg.0
  IL_010d:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0112:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0117:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      380 (0x17c)
  .maxstack  5
  .locals init (Integer V_0,
                SM$T() V_1, //$W0
                Integer V_2, //$W1
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_3,
                SM$T V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse    IL_00c1
    IL_000d:  ldarg.0
    IL_000e:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T()"
    IL_0013:  stloc.1
    IL_0014:  call       "Function Program.GetArrayIndex() As Integer"
    IL_0019:  stloc.2
    IL_001a:  ldarg.0
    IL_001b:  ldloc.1
    IL_001c:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T()"
    IL_0021:  ldarg.0
    IL_0022:  ldloc.2
    IL_0023:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U3 As Integer"
    IL_0028:  ldarg.0
    IL_0029:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T()"
    IL_002e:  ldarg.0
    IL_002f:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U3 As Integer"
    IL_0034:  readonly.
    IL_0036:  ldelema    "SM$T"
    IL_003b:  pop
    IL_003c:  ldloca.s   V_4
    IL_003e:  initobj    "SM$T"
    IL_0044:  ldloc.s    V_4
    IL_0046:  box        "SM$T"
    IL_004b:  brtrue.s   IL_0064
    IL_004d:  ldarg.0
    IL_004e:  ldarg.0
    IL_004f:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T()"
    IL_0054:  ldarg.0
    IL_0055:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U3 As Integer"
    IL_005a:  ldelem     "SM$T"
    IL_005f:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U4 As SM$T"
    IL_0064:  ldarg.0
    IL_0065:  ldloc.1
    IL_0066:  ldloc.2
    IL_0067:  readonly.
    IL_0069:  ldelema    "SM$T"
    IL_006e:  ldc.i4.1
    IL_006f:  constrained. "SM$T"
    IL_0075:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_007a:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As Integer"
    IL_007f:  ldarg.0
    IL_0080:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T()"
    IL_0085:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T()) As Integer"
    IL_008a:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_008f:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0094:  stloc.3
    IL_0095:  ldloca.s   V_3
    IL_0097:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_009c:  brtrue.s   IL_00dd
    IL_009e:  ldarg.0
    IL_009f:  ldc.i4.0
    IL_00a0:  dup
    IL_00a1:  stloc.0
    IL_00a2:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00a7:  ldarg.0
    IL_00a8:  ldloc.3
    IL_00a9:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00ae:  ldarg.0
    IL_00af:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00b4:  ldloca.s   V_3
    IL_00b6:  ldarg.0
    IL_00b7:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_00bc:  leave      IL_017b
    IL_00c1:  ldarg.0
    IL_00c2:  ldc.i4.m1
    IL_00c3:  dup
    IL_00c4:  stloc.0
    IL_00c5:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00ca:  ldarg.0
    IL_00cb:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d0:  stloc.3
    IL_00d1:  ldarg.0
    IL_00d2:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d7:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00dd:  ldloca.s   V_4
    IL_00df:  initobj    "SM$T"
    IL_00e5:  ldloc.s    V_4
    IL_00e7:  box        "SM$T"
    IL_00ec:  brtrue.s   IL_00f6
    IL_00ee:  ldarg.0
    IL_00ef:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U4 As SM$T"
    IL_00f4:  br.s       IL_0109
    IL_00f6:  ldarg.0
    IL_00f7:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T()"
    IL_00fc:  ldarg.0
    IL_00fd:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U3 As Integer"
    IL_0102:  readonly.
    IL_0104:  ldelema    "SM$T"
    IL_0109:  ldc.i4.1
    IL_010a:  ldarg.0
    IL_010b:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As Integer"
    IL_0110:  ldloca.s   V_3
    IL_0112:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0117:  ldloca.s   V_3
    IL_0119:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_011f:  add.ovf
    IL_0120:  constrained. "SM$T"
    IL_0126:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_012b:  ldarg.0
    IL_012c:  ldnull
    IL_012d:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T()"
    IL_0132:  ldarg.0
    IL_0133:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U4 As SM$T"
    IL_0138:  initobj    "SM$T"
    IL_013e:  ldnull
    IL_013f:  stloc.1
    IL_0140:  leave.s    IL_0166
  }
  catch System.Exception
  {
    IL_0142:  dup
    IL_0143:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0148:  stloc.s    V_5
    IL_014a:  ldarg.0
    IL_014b:  ldc.i4.s   -2
    IL_014d:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0152:  ldarg.0
    IL_0153:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0158:  ldloc.s    V_5
    IL_015a:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_015f:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0164:  leave.s    IL_017b
  }
  IL_0166:  ldarg.0
  IL_0167:  ldc.i4.s   -2
  IL_0169:  dup
  IL_016a:  stloc.0
  IL_016b:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0170:  ldarg.0
  IL_0171:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0176:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_017b:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Value_Async_01_ThroughArray_InWith()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = { New Item With {.Name = "1"} }
        Call1(item1).Wait()

        Dim item2 = { New Item With {.Name = "2"} }
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T()) As Task
        With item(GetArrayIndex())
            .Position(1) += await GetOffsetAsync(GetOffset(item))
        End With
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T()) As Task
        With item(GetArrayIndex())
            .Position(1) += await GetOffsetAsync(GetOffset(item))
        End With
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T()) As Integer
        value -= 1
        item(0) = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Function GetArrayIndex() As Integer
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '-1'
Position get for item '2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      303 (0x12f)
  .maxstack  5
  .locals init (Integer V_0,
                SM$T() V_1, //$W0
                Integer V_2, //$W1
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse    IL_0099
    IL_000d:  ldarg.0
    IL_000e:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_0013:  stloc.1
    IL_0014:  call       "Function Program.GetArrayIndex() As Integer"
    IL_0019:  stloc.2
    IL_001a:  ldarg.0
    IL_001b:  ldloc.1
    IL_001c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T()"
    IL_0021:  ldarg.0
    IL_0022:  ldloc.2
    IL_0023:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U3 As Integer"
    IL_0028:  ldarg.0
    IL_0029:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T()"
    IL_002e:  ldarg.0
    IL_002f:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U3 As Integer"
    IL_0034:  readonly.
    IL_0036:  ldelema    "SM$T"
    IL_003b:  pop
    IL_003c:  ldarg.0
    IL_003d:  ldloc.1
    IL_003e:  ldloc.2
    IL_003f:  readonly.
    IL_0041:  ldelema    "SM$T"
    IL_0046:  ldc.i4.1
    IL_0047:  constrained. "SM$T"
    IL_004d:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_0052:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_0057:  ldarg.0
    IL_0058:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T()"
    IL_005d:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T()) As Integer"
    IL_0062:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0067:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_006c:  stloc.3
    IL_006d:  ldloca.s   V_3
    IL_006f:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0074:  brtrue.s   IL_00b5
    IL_0076:  ldarg.0
    IL_0077:  ldc.i4.0
    IL_0078:  dup
    IL_0079:  stloc.0
    IL_007a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_007f:  ldarg.0
    IL_0080:  ldloc.3
    IL_0081:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0086:  ldarg.0
    IL_0087:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_008c:  ldloca.s   V_3
    IL_008e:  ldarg.0
    IL_008f:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0094:  leave      IL_012e
    IL_0099:  ldarg.0
    IL_009a:  ldc.i4.m1
    IL_009b:  dup
    IL_009c:  stloc.0
    IL_009d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00a2:  ldarg.0
    IL_00a3:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a8:  stloc.3
    IL_00a9:  ldarg.0
    IL_00aa:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00af:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b5:  ldarg.0
    IL_00b6:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T()"
    IL_00bb:  ldarg.0
    IL_00bc:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U3 As Integer"
    IL_00c1:  readonly.
    IL_00c3:  ldelema    "SM$T"
    IL_00c8:  ldc.i4.1
    IL_00c9:  ldarg.0
    IL_00ca:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Integer"
    IL_00cf:  ldloca.s   V_3
    IL_00d1:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00d6:  ldloca.s   V_3
    IL_00d8:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00de:  add.ovf
    IL_00df:  constrained. "SM$T"
    IL_00e5:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00ea:  ldarg.0
    IL_00eb:  ldnull
    IL_00ec:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T()"
    IL_00f1:  ldnull
    IL_00f2:  stloc.1
    IL_00f3:  leave.s    IL_0119
  }
  catch System.Exception
  {
    IL_00f5:  dup
    IL_00f6:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00fb:  stloc.s    V_4
    IL_00fd:  ldarg.0
    IL_00fe:  ldc.i4.s   -2
    IL_0100:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0105:  ldarg.0
    IL_0106:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_010b:  ldloc.s    V_4
    IL_010d:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0112:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0117:  leave.s    IL_012e
  }
  IL_0119:  ldarg.0
  IL_011a:  ldc.i4.s   -2
  IL_011c:  dup
  IL_011d:  stloc.0
  IL_011e:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0123:  ldarg.0
  IL_0124:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0129:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_012e:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Index_ThroughField()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Item2(Of T)
    Public Item As T
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item2(Of Item)() With { .Item = New Item With {.Name = "1"} }
        Call1(item1)

        Dim item2 = New Item2(Of Item)() With { .Item = New Item With {.Name = "2"} }
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(item As Item2(Of T))
        item.Item.Position(GetOffset(item)) += 1
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As Item2(Of T))
        item.Item.Position(GetOffset(item)) += 1
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(item As Item2(Of T)) As Integer
        value -= 1
        item.Item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       43 (0x2b)
  .maxstack  4
  .locals init (T V_0,
                Integer V_1)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Item2(Of T).Item As T"
  IL_0006:  dup
  IL_0007:  stloc.0
  IL_0008:  box        "T"
  IL_000d:  ldarg.0
  IL_000e:  call       "Function Program.GetOffset(Of T)(Item2(Of T)) As Integer"
  IL_0013:  dup
  IL_0014:  stloc.1
  IL_0015:  ldloca.s   V_0
  IL_0017:  ldloc.1
  IL_0018:  constrained. "T"
  IL_001e:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0023:  ldc.i4.1
  IL_0024:  add.ovf
  IL_0025:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_002a:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
  // Code size       90 (0x5a)
  .maxstack  4
  .locals init (T& V_0,
                T V_1,
                Integer V_2,
                T V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldflda     "Item2(Of T).Item As T"
  IL_0006:  stloc.0
  IL_0007:  ldloca.s   V_3
  IL_0009:  initobj    "T"
  IL_000f:  ldloc.3
  IL_0010:  box        "T"
  IL_0015:  brtrue.s   IL_0022
  IL_0017:  ldloc.0
  IL_0018:  ldobj      "T"
  IL_001d:  stloc.1
  IL_001e:  ldloca.s   V_1
  IL_0020:  br.s       IL_0023
  IL_0022:  ldloc.0
  IL_0023:  ldarg.0
  IL_0024:  call       "Function Program.GetOffset(Of T)(Item2(Of T)) As Integer"
  IL_0029:  dup
  IL_002a:  stloc.2
  IL_002b:  ldloca.s   V_3
  IL_002d:  initobj    "T"
  IL_0033:  ldloc.3
  IL_0034:  box        "T"
  IL_0039:  brtrue.s   IL_003f
  IL_003b:  ldloca.s   V_1
  IL_003d:  br.s       IL_0040
  IL_003f:  ldloc.0
  IL_0040:  ldloc.2
  IL_0041:  constrained. "T"
  IL_0047:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_004c:  ldc.i4.1
  IL_004d:  add.ovf
  IL_004e:  constrained. "T"
  IL_0054:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0059:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Index_ThroughField()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Item2(Of T)
    Public Item As T
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item2(Of Item)() With { .Item = New Item With {.Name = "1"} }
        Call1(item1)

        Dim item2 = New Item2(Of Item)() With { .Item = New Item With {.Name = "2"} }
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(item As Item2(Of T))
        item.Item.Position(GetOffset(item)) += 1
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As Item2(Of T))
        item.Item.Position(GetOffset(item)) += 1
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(item As Item2(Of T)) As Integer
        value -= 1
        item.Item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-1'
Position get for item '-2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       43 (0x2b)
  .maxstack  4
  .locals init (T& V_0,
                Integer V_1)
  IL_0000:  ldarg.0
  IL_0001:  ldflda     "Item2(Of T).Item As T"
  IL_0006:  dup
  IL_0007:  stloc.0
  IL_0008:  ldarg.0
  IL_0009:  call       "Function Program.GetOffset(Of T)(Item2(Of T)) As Integer"
  IL_000e:  dup
  IL_000f:  stloc.1
  IL_0010:  ldloc.0
  IL_0011:  ldloc.1
  IL_0012:  constrained. "T"
  IL_0018:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_001d:  ldc.i4.1
  IL_001e:  add.ovf
  IL_001f:  constrained. "T"
  IL_0025:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_002a:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Class_Index_Async_01_ThroughField()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Item2(Of T)
    Public Item As T
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item2(Of Item)() With { .Item = New Item With {.Name = "1"} }
        Call1(item1).Wait()

        Dim item2 = New Item2(Of Item)() With { .Item = New Item With {.Name = "2"} }
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As Item2(Of T)) As Task
        item.Item.Position(await GetOffsetAsync(GetOffset(item))) += 1
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As Item2(Of T)) As Task
        item.Item.Position(await GetOffsetAsync(GetOffset(item))) += 1
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As Item2(Of T)) As Integer
        value -= 1
        item.Item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      255 (0xff)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0066
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldarg.0
    IL_000d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As Item2(Of SM$T)"
    IL_0012:  ldfld      "Item2(Of SM$T).Item As SM$T"
    IL_0017:  dup
    IL_0018:  stloc.3
    IL_0019:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_001e:  ldloc.3
    IL_001f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0024:  ldarg.0
    IL_0025:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As Item2(Of SM$T)"
    IL_002a:  call       "Function Program.GetOffset(Of SM$T)(ByRef Item2(Of SM$T)) As Integer"
    IL_002f:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0034:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0039:  stloc.2
    IL_003a:  ldloca.s   V_2
    IL_003c:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0041:  brtrue.s   IL_0082
    IL_0043:  ldarg.0
    IL_0044:  ldc.i4.0
    IL_0045:  dup
    IL_0046:  stloc.0
    IL_0047:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_004c:  ldarg.0
    IL_004d:  ldloc.2
    IL_004e:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0053:  ldarg.0
    IL_0054:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0059:  ldloca.s   V_2
    IL_005b:  ldarg.0
    IL_005c:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0061:  leave      IL_00fe
    IL_0066:  ldarg.0
    IL_0067:  ldc.i4.m1
    IL_0068:  dup
    IL_0069:  stloc.0
    IL_006a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_006f:  ldarg.0
    IL_0070:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0075:  stloc.2
    IL_0076:  ldarg.0
    IL_0077:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007c:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0082:  ldarg.0
    IL_0083:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0088:  box        "SM$T"
    IL_008d:  ldloca.s   V_2
    IL_008f:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0094:  ldloca.s   V_2
    IL_0096:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009c:  dup
    IL_009d:  stloc.1
    IL_009e:  ldarg.0
    IL_009f:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_00a4:  ldloc.1
    IL_00a5:  constrained. "SM$T"
    IL_00ab:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_00b0:  ldc.i4.1
    IL_00b1:  add.ovf
    IL_00b2:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00b7:  ldarg.0
    IL_00b8:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_00bd:  initobj    "SM$T"
    IL_00c3:  leave.s    IL_00e9
  }
  catch System.Exception
  {
    IL_00c5:  dup
    IL_00c6:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00cb:  stloc.s    V_4
    IL_00cd:  ldarg.0
    IL_00ce:  ldc.i4.s   -2
    IL_00d0:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00d5:  ldarg.0
    IL_00d6:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00db:  ldloc.s    V_4
    IL_00dd:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00e2:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00e7:  leave.s    IL_00fe
  }
  IL_00e9:  ldarg.0
  IL_00ea:  ldc.i4.s   -2
  IL_00ec:  dup
  IL_00ed:  stloc.0
  IL_00ee:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00f3:  ldarg.0
  IL_00f4:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00f9:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_00fe:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      357 (0x165)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse    IL_0096
    IL_000d:  ldarg.0
    IL_000e:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As Item2(Of SM$T)"
    IL_0013:  ldfld      "Item2(Of SM$T).Item As SM$T"
    IL_0018:  pop
    IL_0019:  ldloca.s   V_3
    IL_001b:  initobj    "SM$T"
    IL_0021:  ldloc.3
    IL_0022:  box        "SM$T"
    IL_0027:  brfalse.s  IL_0043
    IL_0029:  ldarg.0
    IL_002a:  ldarg.0
    IL_002b:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As Item2(Of SM$T)"
    IL_0030:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As Item2(Of SM$T)"
    IL_0035:  ldarg.0
    IL_0036:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As Item2(Of SM$T)"
    IL_003b:  ldfld      "Item2(Of SM$T).Item As SM$T"
    IL_0040:  pop
    IL_0041:  br.s       IL_0054
    IL_0043:  ldarg.0
    IL_0044:  ldarg.0
    IL_0045:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As Item2(Of SM$T)"
    IL_004a:  ldfld      "Item2(Of SM$T).Item As SM$T"
    IL_004f:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_0054:  ldarg.0
    IL_0055:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As Item2(Of SM$T)"
    IL_005a:  call       "Function Program.GetOffset(Of SM$T)(ByRef Item2(Of SM$T)) As Integer"
    IL_005f:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0064:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0069:  stloc.2
    IL_006a:  ldloca.s   V_2
    IL_006c:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0071:  brtrue.s   IL_00b2
    IL_0073:  ldarg.0
    IL_0074:  ldc.i4.0
    IL_0075:  dup
    IL_0076:  stloc.0
    IL_0077:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_007c:  ldarg.0
    IL_007d:  ldloc.2
    IL_007e:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0083:  ldarg.0
    IL_0084:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0089:  ldloca.s   V_2
    IL_008b:  ldarg.0
    IL_008c:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_0091:  leave      IL_0164
    IL_0096:  ldarg.0
    IL_0097:  ldc.i4.m1
    IL_0098:  dup
    IL_0099:  stloc.0
    IL_009a:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_009f:  ldarg.0
    IL_00a0:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a5:  stloc.2
    IL_00a6:  ldarg.0
    IL_00a7:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00ac:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b2:  ldloca.s   V_3
    IL_00b4:  initobj    "SM$T"
    IL_00ba:  ldloc.3
    IL_00bb:  box        "SM$T"
    IL_00c0:  brtrue.s   IL_00ca
    IL_00c2:  ldarg.0
    IL_00c3:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_00c8:  br.s       IL_00d5
    IL_00ca:  ldarg.0
    IL_00cb:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As Item2(Of SM$T)"
    IL_00d0:  ldflda     "Item2(Of SM$T).Item As SM$T"
    IL_00d5:  ldloca.s   V_2
    IL_00d7:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00dc:  ldloca.s   V_2
    IL_00de:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00e4:  dup
    IL_00e5:  stloc.1
    IL_00e6:  ldloca.s   V_3
    IL_00e8:  initobj    "SM$T"
    IL_00ee:  ldloc.3
    IL_00ef:  box        "SM$T"
    IL_00f4:  brtrue.s   IL_00fe
    IL_00f6:  ldarg.0
    IL_00f7:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_00fc:  br.s       IL_0109
    IL_00fe:  ldarg.0
    IL_00ff:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As Item2(Of SM$T)"
    IL_0104:  ldflda     "Item2(Of SM$T).Item As SM$T"
    IL_0109:  ldloc.1
    IL_010a:  constrained. "SM$T"
    IL_0110:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_0115:  ldc.i4.1
    IL_0116:  add.ovf
    IL_0117:  constrained. "SM$T"
    IL_011d:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_0122:  ldarg.0
    IL_0123:  ldnull
    IL_0124:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As Item2(Of SM$T)"
    IL_0129:  leave.s    IL_014f
  }
  catch System.Exception
  {
    IL_012b:  dup
    IL_012c:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0131:  stloc.s    V_4
    IL_0133:  ldarg.0
    IL_0134:  ldc.i4.s   -2
    IL_0136:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_013b:  ldarg.0
    IL_013c:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0141:  ldloc.s    V_4
    IL_0143:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0148:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_014d:  leave.s    IL_0164
  }
  IL_014f:  ldarg.0
  IL_0150:  ldc.i4.s   -2
  IL_0152:  dup
  IL_0153:  stloc.0
  IL_0154:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0159:  ldarg.0
  IL_015a:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_015f:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0164:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Assignment_Compound_Indexer_Struct_Index_Async_01_ThroughField()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Item2(Of T)
    Public Item As T
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item2(Of Item)() With { .Item = New Item With {.Name = "1"} }
        Call1(item1).Wait()

        Dim item2 = New Item2(Of Item)() With { .Item = New Item With {.Name = "2"} }
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As Item2(Of T)) As Task
        item.Item.Position(await GetOffsetAsync(GetOffset(item))) += 1
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As Item2(Of T)) As Task
        item.Item.Position(await GetOffsetAsync(GetOffset(item))) += 1
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As Item2(Of T)) As Integer
        value -= 1
        item.Item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-1'
Position get for item '-2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      257 (0x101)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0064
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As Item2(Of SM$T)"
    IL_0011:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Item2(Of SM$T)"
    IL_0016:  ldarg.0
    IL_0017:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Item2(Of SM$T)"
    IL_001c:  ldfld      "Item2(Of SM$T).Item As SM$T"
    IL_0021:  pop
    IL_0022:  ldarg.0
    IL_0023:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As Item2(Of SM$T)"
    IL_0028:  call       "Function Program.GetOffset(Of SM$T)(ByRef Item2(Of SM$T)) As Integer"
    IL_002d:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0032:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0037:  stloc.2
    IL_0038:  ldloca.s   V_2
    IL_003a:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_003f:  brtrue.s   IL_0080
    IL_0041:  ldarg.0
    IL_0042:  ldc.i4.0
    IL_0043:  dup
    IL_0044:  stloc.0
    IL_0045:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_004a:  ldarg.0
    IL_004b:  ldloc.2
    IL_004c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0051:  ldarg.0
    IL_0052:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0057:  ldloca.s   V_2
    IL_0059:  ldarg.0
    IL_005a:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_005f:  leave      IL_0100
    IL_0064:  ldarg.0
    IL_0065:  ldc.i4.m1
    IL_0066:  dup
    IL_0067:  stloc.0
    IL_0068:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_006d:  ldarg.0
    IL_006e:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0073:  stloc.2
    IL_0074:  ldarg.0
    IL_0075:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007a:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0080:  ldarg.0
    IL_0081:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Item2(Of SM$T)"
    IL_0086:  ldflda     "Item2(Of SM$T).Item As SM$T"
    IL_008b:  ldloca.s   V_2
    IL_008d:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0092:  ldloca.s   V_2
    IL_0094:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009a:  dup
    IL_009b:  stloc.1
    IL_009c:  ldarg.0
    IL_009d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As Item2(Of SM$T)"
    IL_00a2:  ldflda     "Item2(Of SM$T).Item As SM$T"
    IL_00a7:  ldloc.1
    IL_00a8:  constrained. "SM$T"
    IL_00ae:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_00b3:  ldc.i4.1
    IL_00b4:  add.ovf
    IL_00b5:  constrained. "SM$T"
    IL_00bb:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00c0:  ldarg.0
    IL_00c1:  ldnull
    IL_00c2:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As Item2(Of SM$T)"
    IL_00c7:  leave.s    IL_00eb
  }
  catch System.Exception
  {
    IL_00c9:  dup
    IL_00ca:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00cf:  stloc.3
    IL_00d0:  ldarg.0
    IL_00d1:  ldc.i4.s   -2
    IL_00d3:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00d8:  ldarg.0
    IL_00d9:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00de:  ldloc.3
    IL_00df:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00e4:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00e9:  leave.s    IL_0100
  }
  IL_00eb:  ldarg.0
  IL_00ec:  ldc.i4.s   -2
  IL_00ee:  dup
  IL_00ef:  stloc.0
  IL_00f0:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00f5:  ldarg.0
  IL_00f6:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00fb:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0100:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Class_Index()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(item As T)
        Redim Preserve item.Position(GetOffset(item))(1)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        Redim Preserve item.Position(GetOffset(item))(1)
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       53 (0x35)
  .maxstack  4
  .locals init (T V_0,
                Integer V_1)
  IL_0000:  ldarg.0
  IL_0001:  dup
  IL_0002:  stloc.0
  IL_0003:  box        "T"
  IL_0008:  ldarga.s   V_0
  IL_000a:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_000f:  dup
  IL_0010:  stloc.1
  IL_0011:  ldloca.s   V_0
  IL_0013:  ldloc.1
  IL_0014:  constrained. "T"
  IL_001a:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
  IL_001f:  ldc.i4.2
  IL_0020:  newarr     "Integer"
  IL_0025:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
  IL_002a:  castclass  "Integer()"
  IL_002f:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
  IL_0034:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
      // Code size       95 (0x5f)
  .maxstack  4
  .locals init (T V_0,
                Integer V_1,
                T V_2)
  IL_0000:  ldloca.s   V_2
  IL_0002:  initobj    "T"
  IL_0008:  ldloc.2
  IL_0009:  box        "T"
  IL_000e:  brtrue.s   IL_0016
  IL_0010:  ldarg.0
  IL_0011:  stloc.0
  IL_0012:  ldloca.s   V_0
  IL_0014:  br.s       IL_0018
  IL_0016:  ldarga.s   V_0
  IL_0018:  ldarga.s   V_0
  IL_001a:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_001f:  dup
  IL_0020:  stloc.1
  IL_0021:  ldloca.s   V_2
  IL_0023:  initobj    "T"
  IL_0029:  ldloc.2
  IL_002a:  box        "T"
  IL_002f:  brtrue.s   IL_0035
  IL_0031:  ldloca.s   V_0
  IL_0033:  br.s       IL_0037
  IL_0035:  ldarga.s   V_0
  IL_0037:  ldloc.1
  IL_0038:  constrained. "T"
  IL_003e:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
  IL_0043:  ldc.i4.2
  IL_0044:  newarr     "Integer"
  IL_0049:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
  IL_004e:  castclass  "Integer()"
  IL_0053:  constrained. "T"
  IL_0059:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
  IL_005e:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Struct_Index()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(item As T)
        Redim Preserve item.Position(GetOffset(item))(1)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        Redim Preserve item.Position(GetOffset(item))(1)
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-1'
Position get for item '-2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       53 (0x35)
  .maxstack  4
  .locals init (Integer V_0)
  IL_0000:  ldarga.s   V_0
  IL_0002:  ldarga.s   V_0
  IL_0004:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0009:  dup
  IL_000a:  stloc.0
  IL_000b:  ldarga.s   V_0
  IL_000d:  ldloc.0
  IL_000e:  constrained. "T"
  IL_0014:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
  IL_0019:  ldc.i4.2
  IL_001a:  newarr     "Integer"
  IL_001f:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
  IL_0024:  castclass  "Integer()"
  IL_0029:  constrained. "T"
  IL_002f:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
  IL_0034:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Class_Index_Ref()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(ByRef item As T)
        Redim Preserve item.Position(GetOffset(item))(1)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        Redim Preserve item.Position(GetOffset(item))(1)
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       57 (0x39)
  .maxstack  4
  .locals init (T V_0,
                Integer V_1)
  IL_0000:  ldarg.0
  IL_0001:  ldobj      "T"
  IL_0006:  dup
  IL_0007:  stloc.0
  IL_0008:  box        "T"
  IL_000d:  ldarg.0
  IL_000e:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0013:  dup
  IL_0014:  stloc.1
  IL_0015:  ldloca.s   V_0
  IL_0017:  ldloc.1
  IL_0018:  constrained. "T"
  IL_001e:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
  IL_0023:  ldc.i4.2
  IL_0024:  newarr     "Integer"
  IL_0029:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
  IL_002e:  castclass  "Integer()"
  IL_0033:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
  IL_0038:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
      // Code size       97 (0x61)
  .maxstack  4
  .locals init (T V_0,
                Integer V_1,
                T V_2)
  IL_0000:  ldloca.s   V_2
  IL_0002:  initobj    "T"
  IL_0008:  ldloc.2
  IL_0009:  box        "T"
  IL_000e:  brtrue.s   IL_001b
  IL_0010:  ldarg.0
  IL_0011:  ldobj      "T"
  IL_0016:  stloc.0
  IL_0017:  ldloca.s   V_0
  IL_0019:  br.s       IL_001c
  IL_001b:  ldarg.0
  IL_001c:  ldarg.0
  IL_001d:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0022:  dup
  IL_0023:  stloc.1
  IL_0024:  ldloca.s   V_2
  IL_0026:  initobj    "T"
  IL_002c:  ldloc.2
  IL_002d:  box        "T"
  IL_0032:  brtrue.s   IL_0038
  IL_0034:  ldloca.s   V_0
  IL_0036:  br.s       IL_0039
  IL_0038:  ldarg.0
  IL_0039:  ldloc.1
  IL_003a:  constrained. "T"
  IL_0040:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
  IL_0045:  ldc.i4.2
  IL_0046:  newarr     "Integer"
  IL_004b:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
  IL_0050:  castclass  "Integer()"
  IL_0055:  constrained. "T"
  IL_005b:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
  IL_0060:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Struct_Index_Ref()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(ByRef item As T)
        Redim Preserve item.Position(GetOffset(item))(1)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        Redim Preserve item.Position(GetOffset(item))(1)
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-1'
Position get for item '-2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       50 (0x32)
  .maxstack  4
  .locals init (Integer V_0)
  IL_0000:  ldarg.0
  IL_0001:  ldarg.0
  IL_0002:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0007:  dup
  IL_0008:  stloc.0
  IL_0009:  ldarg.0
  IL_000a:  ldloc.0
  IL_000b:  constrained. "T"
  IL_0011:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
  IL_0016:  ldc.i4.2
  IL_0017:  newarr     "Integer"
  IL_001c:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
  IL_0021:  castclass  "Integer()"
  IL_0026:  constrained. "T"
  IL_002c:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
  IL_0031:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Class_Index_Async_01()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        Redim Preserve item.Position(await GetOffsetAsync(GetOffset(item)))(1)
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        Redim Preserve item.Position(await GetOffsetAsync(GetOffset(item)))(1)
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      264 (0x108)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0061
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldarg.0
    IL_000d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0012:  dup
    IL_0013:  stloc.3
    IL_0014:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0019:  ldloc.3
    IL_001a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_001f:  ldarg.0
    IL_0020:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0025:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_002a:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_002f:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0034:  stloc.2
    IL_0035:  ldloca.s   V_2
    IL_0037:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_003c:  brtrue.s   IL_007d
    IL_003e:  ldarg.0
    IL_003f:  ldc.i4.0
    IL_0040:  dup
    IL_0041:  stloc.0
    IL_0042:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0047:  ldarg.0
    IL_0048:  ldloc.2
    IL_0049:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_004e:  ldarg.0
    IL_004f:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0054:  ldloca.s   V_2
    IL_0056:  ldarg.0
    IL_0057:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_005c:  leave      IL_0107
    IL_0061:  ldarg.0
    IL_0062:  ldc.i4.m1
    IL_0063:  dup
    IL_0064:  stloc.0
    IL_0065:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_006a:  ldarg.0
    IL_006b:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0070:  stloc.2
    IL_0071:  ldarg.0
    IL_0072:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0077:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007d:  ldarg.0
    IL_007e:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0083:  box        "SM$T"
    IL_0088:  ldloca.s   V_2
    IL_008a:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_008f:  ldloca.s   V_2
    IL_0091:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0097:  dup
    IL_0098:  stloc.1
    IL_0099:  ldarg.0
    IL_009a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_009f:  ldloc.1
    IL_00a0:  constrained. "SM$T"
    IL_00a6:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
    IL_00ab:  ldc.i4.2
    IL_00ac:  newarr     "Integer"
    IL_00b1:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
    IL_00b6:  castclass  "Integer()"
    IL_00bb:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
    IL_00c0:  ldarg.0
    IL_00c1:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_00c6:  initobj    "SM$T"
    IL_00cc:  leave.s    IL_00f2
  }
  catch System.Exception
  {
    IL_00ce:  dup
    IL_00cf:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00d4:  stloc.s    V_4
    IL_00d6:  ldarg.0
    IL_00d7:  ldc.i4.s   -2
    IL_00d9:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00de:  ldarg.0
    IL_00df:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00e4:  ldloc.s    V_4
    IL_00e6:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00eb:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00f0:  leave.s    IL_0107
  }
  IL_00f2:  ldarg.0
  IL_00f3:  ldc.i4.s   -2
  IL_00f5:  dup
  IL_00f6:  stloc.0
  IL_00f7:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00fc:  ldarg.0
  IL_00fd:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0102:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0107:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      308 (0x134)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0068
    IL_000a:  ldloca.s   V_3
    IL_000c:  initobj    "SM$T"
    IL_0012:  ldloc.3
    IL_0013:  box        "SM$T"
    IL_0018:  brtrue.s   IL_0026
    IL_001a:  ldarg.0
    IL_001b:  ldarg.0
    IL_001c:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0021:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_0026:  ldarg.0
    IL_0027:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_002c:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0031:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0036:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_003b:  stloc.2
    IL_003c:  ldloca.s   V_2
    IL_003e:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0043:  brtrue.s   IL_0084
    IL_0045:  ldarg.0
    IL_0046:  ldc.i4.0
    IL_0047:  dup
    IL_0048:  stloc.0
    IL_0049:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_004e:  ldarg.0
    IL_004f:  ldloc.2
    IL_0050:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0055:  ldarg.0
    IL_0056:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_005b:  ldloca.s   V_2
    IL_005d:  ldarg.0
    IL_005e:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_0063:  leave      IL_0133
    IL_0068:  ldarg.0
    IL_0069:  ldc.i4.m1
    IL_006a:  dup
    IL_006b:  stloc.0
    IL_006c:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0071:  ldarg.0
    IL_0072:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0077:  stloc.2
    IL_0078:  ldarg.0
    IL_0079:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007e:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0084:  ldloca.s   V_3
    IL_0086:  initobj    "SM$T"
    IL_008c:  ldloc.3
    IL_008d:  box        "SM$T"
    IL_0092:  brtrue.s   IL_009c
    IL_0094:  ldarg.0
    IL_0095:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_009a:  br.s       IL_00a2
    IL_009c:  ldarg.0
    IL_009d:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00a2:  ldloca.s   V_2
    IL_00a4:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00a9:  ldloca.s   V_2
    IL_00ab:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b1:  dup
    IL_00b2:  stloc.1
    IL_00b3:  ldloca.s   V_3
    IL_00b5:  initobj    "SM$T"
    IL_00bb:  ldloc.3
    IL_00bc:  box        "SM$T"
    IL_00c1:  brtrue.s   IL_00cb
    IL_00c3:  ldarg.0
    IL_00c4:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_00c9:  br.s       IL_00d1
    IL_00cb:  ldarg.0
    IL_00cc:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00d1:  ldloc.1
    IL_00d2:  constrained. "SM$T"
    IL_00d8:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
    IL_00dd:  ldc.i4.2
    IL_00de:  newarr     "Integer"
    IL_00e3:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
    IL_00e8:  castclass  "Integer()"
    IL_00ed:  constrained. "SM$T"
    IL_00f3:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
    IL_00f8:  leave.s    IL_011e
  }
  catch System.Exception
  {
    IL_00fa:  dup
    IL_00fb:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0100:  stloc.s    V_4
    IL_0102:  ldarg.0
    IL_0103:  ldc.i4.s   -2
    IL_0105:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_010a:  ldarg.0
    IL_010b:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0110:  ldloc.s    V_4
    IL_0112:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0117:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_011c:  leave.s    IL_0133
  }
  IL_011e:  ldarg.0
  IL_011f:  ldc.i4.s   -2
  IL_0121:  dup
  IL_0122:  stloc.0
  IL_0123:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0128:  ldarg.0
  IL_0129:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_012e:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0133:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Struct_Index_Async_01()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        Redim Preserve item.Position(await GetOffsetAsync(GetOffset(item)))(1)
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        Redim Preserve item.Position(await GetOffsetAsync(GetOffset(item)))(1)
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-1'
Position get for item '-2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      230 (0xe6)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004c
    IL_000a:  ldarg.0
    IL_000b:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0010:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0015:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_001a:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_001f:  stloc.2
    IL_0020:  ldloca.s   V_2
    IL_0022:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0027:  brtrue.s   IL_0068
    IL_0029:  ldarg.0
    IL_002a:  ldc.i4.0
    IL_002b:  dup
    IL_002c:  stloc.0
    IL_002d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0032:  ldarg.0
    IL_0033:  ldloc.2
    IL_0034:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0039:  ldarg.0
    IL_003a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003f:  ldloca.s   V_2
    IL_0041:  ldarg.0
    IL_0042:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0047:  leave      IL_00e5
    IL_004c:  ldarg.0
    IL_004d:  ldc.i4.m1
    IL_004e:  dup
    IL_004f:  stloc.0
    IL_0050:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0055:  ldarg.0
    IL_0056:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_005b:  stloc.2
    IL_005c:  ldarg.0
    IL_005d:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0062:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0068:  ldarg.0
    IL_0069:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_006e:  ldloca.s   V_2
    IL_0070:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0075:  ldloca.s   V_2
    IL_0077:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007d:  dup
    IL_007e:  stloc.1
    IL_007f:  ldarg.0
    IL_0080:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0085:  ldloc.1
    IL_0086:  constrained. "SM$T"
    IL_008c:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
    IL_0091:  ldc.i4.2
    IL_0092:  newarr     "Integer"
    IL_0097:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
    IL_009c:  castclass  "Integer()"
    IL_00a1:  constrained. "SM$T"
    IL_00a7:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
    IL_00ac:  leave.s    IL_00d0
  }
  catch System.Exception
  {
    IL_00ae:  dup
    IL_00af:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00b4:  stloc.3
    IL_00b5:  ldarg.0
    IL_00b6:  ldc.i4.s   -2
    IL_00b8:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00bd:  ldarg.0
    IL_00be:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00c3:  ldloc.3
    IL_00c4:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00c9:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00ce:  leave.s    IL_00e5
  }
  IL_00d0:  ldarg.0
  IL_00d1:  ldc.i4.s   -2
  IL_00d3:  dup
  IL_00d4:  stloc.0
  IL_00d5:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00da:  ldarg.0
  IL_00db:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00e0:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_00e5:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Class_Index_Async_02()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        await Task.Yield()
        Redim Preserve item.Position(await GetOffsetAsync(GetOffset(item)))(1)
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        await Task.Yield()
        Redim Preserve item.Position(await GetOffsetAsync(GetOffset(item)))(1)
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      377 (0x179)
  .maxstack  4
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter V_1,
                System.Runtime.CompilerServices.YieldAwaitable V_2,
                Integer V_3,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_4,
                SM$T V_5,
                System.Exception V_6)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004b
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_00d1
    IL_0011:  call       "Function System.Threading.Tasks.Task.Yield() As System.Runtime.CompilerServices.YieldAwaitable"
    IL_0016:  stloc.2
    IL_0017:  ldloca.s   V_2
    IL_0019:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.GetAwaiter() As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_001e:  stloc.1
    IL_001f:  ldloca.s   V_1
    IL_0021:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.get_IsCompleted() As Boolean"
    IL_0026:  brtrue.s   IL_0067
    IL_0028:  ldarg.0
    IL_0029:  ldc.i4.0
    IL_002a:  dup
    IL_002b:  stloc.0
    IL_002c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0031:  ldarg.0
    IL_0032:  ldloc.1
    IL_0033:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0038:  ldarg.0
    IL_0039:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003e:  ldloca.s   V_1
    IL_0040:  ldarg.0
    IL_0041:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0046:  leave      IL_0178
    IL_004b:  ldarg.0
    IL_004c:  ldc.i4.m1
    IL_004d:  dup
    IL_004e:  stloc.0
    IL_004f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0054:  ldarg.0
    IL_0055:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_005a:  stloc.1
    IL_005b:  ldarg.0
    IL_005c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0061:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0067:  ldloca.s   V_1
    IL_0069:  call       "Sub System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.GetResult()"
    IL_006e:  ldloca.s   V_1
    IL_0070:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0076:  ldarg.0
    IL_0077:  ldarg.0
    IL_0078:  ldarg.0
    IL_0079:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_007e:  dup
    IL_007f:  stloc.s    V_5
    IL_0081:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0086:  ldloc.s    V_5
    IL_0088:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_008d:  ldarg.0
    IL_008e:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0093:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0098:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_009d:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a2:  stloc.s    V_4
    IL_00a4:  ldloca.s   V_4
    IL_00a6:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_00ab:  brtrue.s   IL_00ee
    IL_00ad:  ldarg.0
    IL_00ae:  ldc.i4.1
    IL_00af:  dup
    IL_00b0:  stloc.0
    IL_00b1:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00b6:  ldarg.0
    IL_00b7:  ldloc.s    V_4
    IL_00b9:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00be:  ldarg.0
    IL_00bf:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00c4:  ldloca.s   V_4
    IL_00c6:  ldarg.0
    IL_00c7:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_00cc:  leave      IL_0178
    IL_00d1:  ldarg.0
    IL_00d2:  ldc.i4.m1
    IL_00d3:  dup
    IL_00d4:  stloc.0
    IL_00d5:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00da:  ldarg.0
    IL_00db:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00e0:  stloc.s    V_4
    IL_00e2:  ldarg.0
    IL_00e3:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00e8:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00ee:  ldarg.0
    IL_00ef:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_00f4:  box        "SM$T"
    IL_00f9:  ldloca.s   V_4
    IL_00fb:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0100:  ldloca.s   V_4
    IL_0102:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0108:  dup
    IL_0109:  stloc.3
    IL_010a:  ldarg.0
    IL_010b:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0110:  ldloc.3
    IL_0111:  constrained. "SM$T"
    IL_0117:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
    IL_011c:  ldc.i4.2
    IL_011d:  newarr     "Integer"
    IL_0122:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
    IL_0127:  castclass  "Integer()"
    IL_012c:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
    IL_0131:  ldarg.0
    IL_0132:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0137:  initobj    "SM$T"
    IL_013d:  leave.s    IL_0163
  }
  catch System.Exception
  {
    IL_013f:  dup
    IL_0140:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0145:  stloc.s    V_6
    IL_0147:  ldarg.0
    IL_0148:  ldc.i4.s   -2
    IL_014a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_014f:  ldarg.0
    IL_0150:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0155:  ldloc.s    V_6
    IL_0157:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_015c:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0161:  leave.s    IL_0178
  }
  IL_0163:  ldarg.0
  IL_0164:  ldc.i4.s   -2
  IL_0166:  dup
  IL_0167:  stloc.0
  IL_0168:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_016d:  ldarg.0
  IL_016e:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0173:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0178:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      422 (0x1a6)
  .maxstack  4
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter V_1,
                System.Runtime.CompilerServices.YieldAwaitable V_2,
                Integer V_3,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_4,
                SM$T V_5,
                System.Exception V_6)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004b
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_00d7
    IL_0011:  call       "Function System.Threading.Tasks.Task.Yield() As System.Runtime.CompilerServices.YieldAwaitable"
    IL_0016:  stloc.2
    IL_0017:  ldloca.s   V_2
    IL_0019:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.GetAwaiter() As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_001e:  stloc.1
    IL_001f:  ldloca.s   V_1
    IL_0021:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.get_IsCompleted() As Boolean"
    IL_0026:  brtrue.s   IL_0067
    IL_0028:  ldarg.0
    IL_0029:  ldc.i4.0
    IL_002a:  dup
    IL_002b:  stloc.0
    IL_002c:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0031:  ldarg.0
    IL_0032:  ldloc.1
    IL_0033:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0038:  ldarg.0
    IL_0039:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003e:  ldloca.s   V_1
    IL_0040:  ldarg.0
    IL_0041:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_0046:  leave      IL_01a5
    IL_004b:  ldarg.0
    IL_004c:  ldc.i4.m1
    IL_004d:  dup
    IL_004e:  stloc.0
    IL_004f:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0054:  ldarg.0
    IL_0055:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_005a:  stloc.1
    IL_005b:  ldarg.0
    IL_005c:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0061:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0067:  ldloca.s   V_1
    IL_0069:  call       "Sub System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.GetResult()"
    IL_006e:  ldloca.s   V_1
    IL_0070:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0076:  ldloca.s   V_5
    IL_0078:  initobj    "SM$T"
    IL_007e:  ldloc.s    V_5
    IL_0080:  box        "SM$T"
    IL_0085:  brtrue.s   IL_0093
    IL_0087:  ldarg.0
    IL_0088:  ldarg.0
    IL_0089:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_008e:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_0093:  ldarg.0
    IL_0094:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0099:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_009e:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_00a3:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a8:  stloc.s    V_4
    IL_00aa:  ldloca.s   V_4
    IL_00ac:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_00b1:  brtrue.s   IL_00f4
    IL_00b3:  ldarg.0
    IL_00b4:  ldc.i4.1
    IL_00b5:  dup
    IL_00b6:  stloc.0
    IL_00b7:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00bc:  ldarg.0
    IL_00bd:  ldloc.s    V_4
    IL_00bf:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00c4:  ldarg.0
    IL_00c5:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00ca:  ldloca.s   V_4
    IL_00cc:  ldarg.0
    IL_00cd:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_00d2:  leave      IL_01a5
    IL_00d7:  ldarg.0
    IL_00d8:  ldc.i4.m1
    IL_00d9:  dup
    IL_00da:  stloc.0
    IL_00db:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00e0:  ldarg.0
    IL_00e1:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00e6:  stloc.s    V_4
    IL_00e8:  ldarg.0
    IL_00e9:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00ee:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00f4:  ldloca.s   V_5
    IL_00f6:  initobj    "SM$T"
    IL_00fc:  ldloc.s    V_5
    IL_00fe:  box        "SM$T"
    IL_0103:  brtrue.s   IL_010d
    IL_0105:  ldarg.0
    IL_0106:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_010b:  br.s       IL_0113
    IL_010d:  ldarg.0
    IL_010e:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0113:  ldloca.s   V_4
    IL_0115:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_011a:  ldloca.s   V_4
    IL_011c:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0122:  dup
    IL_0123:  stloc.3
    IL_0124:  ldloca.s   V_5
    IL_0126:  initobj    "SM$T"
    IL_012c:  ldloc.s    V_5
    IL_012e:  box        "SM$T"
    IL_0133:  brtrue.s   IL_013d
    IL_0135:  ldarg.0
    IL_0136:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_013b:  br.s       IL_0143
    IL_013d:  ldarg.0
    IL_013e:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0143:  ldloc.3
    IL_0144:  constrained. "SM$T"
    IL_014a:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
    IL_014f:  ldc.i4.2
    IL_0150:  newarr     "Integer"
    IL_0155:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
    IL_015a:  castclass  "Integer()"
    IL_015f:  constrained. "SM$T"
    IL_0165:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
    IL_016a:  leave.s    IL_0190
  }
  catch System.Exception
  {
    IL_016c:  dup
    IL_016d:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0172:  stloc.s    V_6
    IL_0174:  ldarg.0
    IL_0175:  ldc.i4.s   -2
    IL_0177:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_017c:  ldarg.0
    IL_017d:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0182:  ldloc.s    V_6
    IL_0184:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0189:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_018e:  leave.s    IL_01a5
  }
  IL_0190:  ldarg.0
  IL_0191:  ldc.i4.s   -2
  IL_0193:  dup
  IL_0194:  stloc.0
  IL_0195:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_019a:  ldarg.0
  IL_019b:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_01a0:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_01a5:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Struct_Index_Async_02()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        await Task.Yield()
        Redim Preserve item.Position(await GetOffsetAsync(GetOffset(item)))(1)
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        await Task.Yield()
        Redim Preserve item.Position(await GetOffsetAsync(GetOffset(item)))(1)
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-1'
Position get for item '-2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      343 (0x157)
  .maxstack  4
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter V_1,
                System.Runtime.CompilerServices.YieldAwaitable V_2,
                Integer V_3,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004b
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_00ba
    IL_0011:  call       "Function System.Threading.Tasks.Task.Yield() As System.Runtime.CompilerServices.YieldAwaitable"
    IL_0016:  stloc.2
    IL_0017:  ldloca.s   V_2
    IL_0019:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.GetAwaiter() As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_001e:  stloc.1
    IL_001f:  ldloca.s   V_1
    IL_0021:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.get_IsCompleted() As Boolean"
    IL_0026:  brtrue.s   IL_0067
    IL_0028:  ldarg.0
    IL_0029:  ldc.i4.0
    IL_002a:  dup
    IL_002b:  stloc.0
    IL_002c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0031:  ldarg.0
    IL_0032:  ldloc.1
    IL_0033:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0038:  ldarg.0
    IL_0039:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003e:  ldloca.s   V_1
    IL_0040:  ldarg.0
    IL_0041:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0046:  leave      IL_0156
    IL_004b:  ldarg.0
    IL_004c:  ldc.i4.m1
    IL_004d:  dup
    IL_004e:  stloc.0
    IL_004f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0054:  ldarg.0
    IL_0055:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_005a:  stloc.1
    IL_005b:  ldarg.0
    IL_005c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0061:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0067:  ldloca.s   V_1
    IL_0069:  call       "Sub System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.GetResult()"
    IL_006e:  ldloca.s   V_1
    IL_0070:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0076:  ldarg.0
    IL_0077:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_007c:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0081:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0086:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008b:  stloc.s    V_4
    IL_008d:  ldloca.s   V_4
    IL_008f:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0094:  brtrue.s   IL_00d7
    IL_0096:  ldarg.0
    IL_0097:  ldc.i4.1
    IL_0098:  dup
    IL_0099:  stloc.0
    IL_009a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_009f:  ldarg.0
    IL_00a0:  ldloc.s    V_4
    IL_00a2:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a7:  ldarg.0
    IL_00a8:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00ad:  ldloca.s   V_4
    IL_00af:  ldarg.0
    IL_00b0:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_00b5:  leave      IL_0156
    IL_00ba:  ldarg.0
    IL_00bb:  ldc.i4.m1
    IL_00bc:  dup
    IL_00bd:  stloc.0
    IL_00be:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00c3:  ldarg.0
    IL_00c4:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00c9:  stloc.s    V_4
    IL_00cb:  ldarg.0
    IL_00cc:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d1:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d7:  ldarg.0
    IL_00d8:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_00dd:  ldloca.s   V_4
    IL_00df:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00e4:  ldloca.s   V_4
    IL_00e6:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00ec:  dup
    IL_00ed:  stloc.3
    IL_00ee:  ldarg.0
    IL_00ef:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_00f4:  ldloc.3
    IL_00f5:  constrained. "SM$T"
    IL_00fb:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
    IL_0100:  ldc.i4.2
    IL_0101:  newarr     "Integer"
    IL_0106:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
    IL_010b:  castclass  "Integer()"
    IL_0110:  constrained. "SM$T"
    IL_0116:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
    IL_011b:  leave.s    IL_0141
  }
  catch System.Exception
  {
    IL_011d:  dup
    IL_011e:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0123:  stloc.s    V_5
    IL_0125:  ldarg.0
    IL_0126:  ldc.i4.s   -2
    IL_0128:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_012d:  ldarg.0
    IL_012e:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0133:  ldloc.s    V_5
    IL_0135:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_013a:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_013f:  leave.s    IL_0156
  }
  IL_0141:  ldarg.0
  IL_0142:  ldc.i4.s   -2
  IL_0144:  dup
  IL_0145:  stloc.0
  IL_0146:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_014b:  ldarg.0
  IL_014c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0151:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0156:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Class_Value()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(item As T)
        Redim Preserve item.Position(1)(GetOffset(item))
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        Redim Preserve item.Position(1)(GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       53 (0x35)
  .maxstack  5
  .locals init (T V_0)
  IL_0000:  ldarg.0
  IL_0001:  dup
  IL_0002:  stloc.0
  IL_0003:  box        "T"
  IL_0008:  ldc.i4.1
  IL_0009:  ldloca.s   V_0
  IL_000b:  ldc.i4.1
  IL_000c:  constrained. "T"
  IL_0012:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
  IL_0017:  ldarga.s   V_0
  IL_0019:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_001e:  ldc.i4.1
  IL_001f:  add.ovf
  IL_0020:  newarr     "Integer"
  IL_0025:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
  IL_002a:  castclass  "Integer()"
  IL_002f:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
  IL_0034:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
      // Code size       77 (0x4d)
  .maxstack  5
  .locals init (T V_0)
  IL_0000:  ldarga.s   V_0
  IL_0002:  ldloca.s   V_0
  IL_0004:  initobj    "T"
  IL_000a:  ldloc.0
  IL_000b:  box        "T"
  IL_0010:  brtrue.s   IL_001a
  IL_0012:  ldobj      "T"
  IL_0017:  stloc.0
  IL_0018:  ldloca.s   V_0
  IL_001a:  ldc.i4.1
  IL_001b:  ldarga.s   V_0
  IL_001d:  ldc.i4.1
  IL_001e:  constrained. "T"
  IL_0024:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
  IL_0029:  ldarga.s   V_0
  IL_002b:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0030:  ldc.i4.1
  IL_0031:  add.ovf
  IL_0032:  newarr     "Integer"
  IL_0037:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
  IL_003c:  castclass  "Integer()"
  IL_0041:  constrained. "T"
  IL_0047:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
  IL_004c:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Struct_Value()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(item As T)
        Redim Preserve item.Position(1)(GetOffset(item))
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        Redim Preserve item.Position(1)(GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '-1'
Position get for item '2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       53 (0x35)
  .maxstack  5
  IL_0000:  ldarga.s   V_0
  IL_0002:  ldc.i4.1
  IL_0003:  ldarga.s   V_0
  IL_0005:  ldc.i4.1
  IL_0006:  constrained. "T"
  IL_000c:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
  IL_0011:  ldarga.s   V_0
  IL_0013:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0018:  ldc.i4.1
  IL_0019:  add.ovf
  IL_001a:  newarr     "Integer"
  IL_001f:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
  IL_0024:  castclass  "Integer()"
  IL_0029:  constrained. "T"
  IL_002f:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
  IL_0034:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Class_Value_Ref()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(ByRef item As T)
        Redim Preserve item.Position(1)(GetOffset(item))
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        Redim Preserve item.Position(1)(GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       57 (0x39)
  .maxstack  5
  .locals init (T V_0)
  IL_0000:  ldarg.0
  IL_0001:  ldobj      "T"
  IL_0006:  dup
  IL_0007:  stloc.0
  IL_0008:  box        "T"
  IL_000d:  ldc.i4.1
  IL_000e:  ldloca.s   V_0
  IL_0010:  ldc.i4.1
  IL_0011:  constrained. "T"
  IL_0017:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
  IL_001c:  ldarg.0
  IL_001d:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0022:  ldc.i4.1
  IL_0023:  add.ovf
  IL_0024:  newarr     "Integer"
  IL_0029:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
  IL_002e:  castclass  "Integer()"
  IL_0033:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
  IL_0038:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
      // Code size       74 (0x4a)
  .maxstack  5
  .locals init (T V_0)
  IL_0000:  ldarg.0
  IL_0001:  ldloca.s   V_0
  IL_0003:  initobj    "T"
  IL_0009:  ldloc.0
  IL_000a:  box        "T"
  IL_000f:  brtrue.s   IL_0019
  IL_0011:  ldobj      "T"
  IL_0016:  stloc.0
  IL_0017:  ldloca.s   V_0
  IL_0019:  ldc.i4.1
  IL_001a:  ldarg.0
  IL_001b:  ldc.i4.1
  IL_001c:  constrained. "T"
  IL_0022:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
  IL_0027:  ldarg.0
  IL_0028:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_002d:  ldc.i4.1
  IL_002e:  add.ovf
  IL_002f:  newarr     "Integer"
  IL_0034:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
  IL_0039:  castclass  "Integer()"
  IL_003e:  constrained. "T"
  IL_0044:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
  IL_0049:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Struct_Value_Ref()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(ByRef item As T)
        Redim Preserve item.Position(1)(GetOffset(item))
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        Redim Preserve item.Position(1)(GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '-1'
Position get for item '2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       50 (0x32)
  .maxstack  5
  IL_0000:  ldarg.0
  IL_0001:  ldc.i4.1
  IL_0002:  ldarg.0
  IL_0003:  ldc.i4.1
  IL_0004:  constrained. "T"
  IL_000a:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
  IL_000f:  ldarg.0
  IL_0010:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0015:  ldc.i4.1
  IL_0016:  add.ovf
  IL_0017:  newarr     "Integer"
  IL_001c:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
  IL_0021:  castclass  "Integer()"
  IL_0026:  constrained. "T"
  IL_002c:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
  IL_0031:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Class_Value_Async_01()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        Redim Preserve item.Position(1)(await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        Redim Preserve item.Position(1)(await GetOffsetAsync(GetOffset(item)))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      281 (0x119)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                SM$T V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0079
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldarg.0
    IL_000d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0012:  dup
    IL_0013:  stloc.2
    IL_0014:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0019:  ldloc.2
    IL_001a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_001f:  ldarg.0
    IL_0020:  ldarg.0
    IL_0021:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0026:  ldc.i4.1
    IL_0027:  constrained. "SM$T"
    IL_002d:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
    IL_0032:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As System.Array"
    IL_0037:  ldarg.0
    IL_0038:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_003d:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0042:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0047:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_004c:  stloc.1
    IL_004d:  ldloca.s   V_1
    IL_004f:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0054:  brtrue.s   IL_0095
    IL_0056:  ldarg.0
    IL_0057:  ldc.i4.0
    IL_0058:  dup
    IL_0059:  stloc.0
    IL_005a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_005f:  ldarg.0
    IL_0060:  ldloc.1
    IL_0061:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0066:  ldarg.0
    IL_0067:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_006c:  ldloca.s   V_1
    IL_006e:  ldarg.0
    IL_006f:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0074:  leave      IL_0118
    IL_0079:  ldarg.0
    IL_007a:  ldc.i4.m1
    IL_007b:  dup
    IL_007c:  stloc.0
    IL_007d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0082:  ldarg.0
    IL_0083:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0088:  stloc.1
    IL_0089:  ldarg.0
    IL_008a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008f:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0095:  ldarg.0
    IL_0096:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_009b:  box        "SM$T"
    IL_00a0:  ldc.i4.1
    IL_00a1:  ldarg.0
    IL_00a2:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As System.Array"
    IL_00a7:  ldloca.s   V_1
    IL_00a9:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00ae:  ldloca.s   V_1
    IL_00b0:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b6:  ldc.i4.1
    IL_00b7:  add.ovf
    IL_00b8:  newarr     "Integer"
    IL_00bd:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
    IL_00c2:  castclass  "Integer()"
    IL_00c7:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
    IL_00cc:  ldarg.0
    IL_00cd:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_00d2:  initobj    "SM$T"
    IL_00d8:  ldarg.0
    IL_00d9:  ldnull
    IL_00da:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As System.Array"
    IL_00df:  leave.s    IL_0103
  }
  catch System.Exception
  {
    IL_00e1:  dup
    IL_00e2:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00e7:  stloc.3
    IL_00e8:  ldarg.0
    IL_00e9:  ldc.i4.s   -2
    IL_00eb:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00f0:  ldarg.0
    IL_00f1:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00f6:  ldloc.3
    IL_00f7:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00fc:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0101:  leave.s    IL_0118
  }
  IL_0103:  ldarg.0
  IL_0104:  ldc.i4.s   -2
  IL_0106:  dup
  IL_0107:  stloc.0
  IL_0108:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_010d:  ldarg.0
  IL_010e:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0113:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0118:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      313 (0x139)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                SM$T V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0080
    IL_000a:  ldloca.s   V_2
    IL_000c:  initobj    "SM$T"
    IL_0012:  ldloc.2
    IL_0013:  box        "SM$T"
    IL_0018:  brtrue.s   IL_0026
    IL_001a:  ldarg.0
    IL_001b:  ldarg.0
    IL_001c:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0021:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T"
    IL_0026:  ldarg.0
    IL_0027:  ldarg.0
    IL_0028:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_002d:  ldc.i4.1
    IL_002e:  constrained. "SM$T"
    IL_0034:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
    IL_0039:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As System.Array"
    IL_003e:  ldarg.0
    IL_003f:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0044:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0049:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_004e:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0053:  stloc.1
    IL_0054:  ldloca.s   V_1
    IL_0056:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_005b:  brtrue.s   IL_009c
    IL_005d:  ldarg.0
    IL_005e:  ldc.i4.0
    IL_005f:  dup
    IL_0060:  stloc.0
    IL_0061:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0066:  ldarg.0
    IL_0067:  ldloc.1
    IL_0068:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_006d:  ldarg.0
    IL_006e:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0073:  ldloca.s   V_1
    IL_0075:  ldarg.0
    IL_0076:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_007b:  leave      IL_0138
    IL_0080:  ldarg.0
    IL_0081:  ldc.i4.m1
    IL_0082:  dup
    IL_0083:  stloc.0
    IL_0084:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0089:  ldarg.0
    IL_008a:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008f:  stloc.1
    IL_0090:  ldarg.0
    IL_0091:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0096:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009c:  ldloca.s   V_2
    IL_009e:  initobj    "SM$T"
    IL_00a4:  ldloc.2
    IL_00a5:  box        "SM$T"
    IL_00aa:  brtrue.s   IL_00b4
    IL_00ac:  ldarg.0
    IL_00ad:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T"
    IL_00b2:  br.s       IL_00ba
    IL_00b4:  ldarg.0
    IL_00b5:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00ba:  ldc.i4.1
    IL_00bb:  ldarg.0
    IL_00bc:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As System.Array"
    IL_00c1:  ldloca.s   V_1
    IL_00c3:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00c8:  ldloca.s   V_1
    IL_00ca:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d0:  ldc.i4.1
    IL_00d1:  add.ovf
    IL_00d2:  newarr     "Integer"
    IL_00d7:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
    IL_00dc:  castclass  "Integer()"
    IL_00e1:  constrained. "SM$T"
    IL_00e7:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
    IL_00ec:  ldarg.0
    IL_00ed:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As SM$T"
    IL_00f2:  initobj    "SM$T"
    IL_00f8:  ldarg.0
    IL_00f9:  ldnull
    IL_00fa:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As System.Array"
    IL_00ff:  leave.s    IL_0123
  }
  catch System.Exception
  {
    IL_0101:  dup
    IL_0102:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0107:  stloc.3
    IL_0108:  ldarg.0
    IL_0109:  ldc.i4.s   -2
    IL_010b:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0110:  ldarg.0
    IL_0111:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0116:  ldloc.3
    IL_0117:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_011c:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0121:  leave.s    IL_0138
  }
  IL_0123:  ldarg.0
  IL_0124:  ldc.i4.s   -2
  IL_0126:  dup
  IL_0127:  stloc.0
  IL_0128:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_012d:  ldarg.0
  IL_012e:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0133:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0138:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Struct_Value_Async_01()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        Redim Preserve item.Position(1)(await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        Redim Preserve item.Position(1)(await GetOffsetAsync(GetOffset(item)))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '-1'
Position get for item '2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      249 (0xf9)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                System.Exception V_2)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0064
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0011:  ldc.i4.1
    IL_0012:  constrained. "SM$T"
    IL_0018:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
    IL_001d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As System.Array"
    IL_0022:  ldarg.0
    IL_0023:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0028:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_002d:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0032:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0037:  stloc.1
    IL_0038:  ldloca.s   V_1
    IL_003a:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_003f:  brtrue.s   IL_0080
    IL_0041:  ldarg.0
    IL_0042:  ldc.i4.0
    IL_0043:  dup
    IL_0044:  stloc.0
    IL_0045:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_004a:  ldarg.0
    IL_004b:  ldloc.1
    IL_004c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0051:  ldarg.0
    IL_0052:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0057:  ldloca.s   V_1
    IL_0059:  ldarg.0
    IL_005a:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_005f:  leave      IL_00f8
    IL_0064:  ldarg.0
    IL_0065:  ldc.i4.m1
    IL_0066:  dup
    IL_0067:  stloc.0
    IL_0068:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_006d:  ldarg.0
    IL_006e:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0073:  stloc.1
    IL_0074:  ldarg.0
    IL_0075:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007a:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0080:  ldarg.0
    IL_0081:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0086:  ldc.i4.1
    IL_0087:  ldarg.0
    IL_0088:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As System.Array"
    IL_008d:  ldloca.s   V_1
    IL_008f:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0094:  ldloca.s   V_1
    IL_0096:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009c:  ldc.i4.1
    IL_009d:  add.ovf
    IL_009e:  newarr     "Integer"
    IL_00a3:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
    IL_00a8:  castclass  "Integer()"
    IL_00ad:  constrained. "SM$T"
    IL_00b3:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
    IL_00b8:  ldarg.0
    IL_00b9:  ldnull
    IL_00ba:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As System.Array"
    IL_00bf:  leave.s    IL_00e3
  }
  catch System.Exception
  {
    IL_00c1:  dup
    IL_00c2:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00c7:  stloc.2
    IL_00c8:  ldarg.0
    IL_00c9:  ldc.i4.s   -2
    IL_00cb:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00d0:  ldarg.0
    IL_00d1:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00d6:  ldloc.2
    IL_00d7:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00dc:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00e1:  leave.s    IL_00f8
  }
  IL_00e3:  ldarg.0
  IL_00e4:  ldc.i4.s   -2
  IL_00e6:  dup
  IL_00e7:  stloc.0
  IL_00e8:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00ed:  ldarg.0
  IL_00ee:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00f3:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_00f8:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Class_IndexAndValue()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(item As T)
        Redim Preserve item.Position(GetOffset(item))(GetOffset(item))
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        Redim Preserve item.Position(GetOffset(item))(GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       61 (0x3d)
  .maxstack  5
  .locals init (T V_0,
                Integer V_1)
  IL_0000:  ldarg.0
  IL_0001:  dup
  IL_0002:  stloc.0
  IL_0003:  box        "T"
  IL_0008:  ldarga.s   V_0
  IL_000a:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_000f:  dup
  IL_0010:  stloc.1
  IL_0011:  ldloca.s   V_0
  IL_0013:  ldloc.1
  IL_0014:  constrained. "T"
  IL_001a:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
  IL_001f:  ldarga.s   V_0
  IL_0021:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0026:  ldc.i4.1
  IL_0027:  add.ovf
  IL_0028:  newarr     "Integer"
  IL_002d:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
  IL_0032:  castclass  "Integer()"
  IL_0037:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
  IL_003c:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
      // Code size      103 (0x67)
  .maxstack  5
  .locals init (T V_0,
                Integer V_1,
                T V_2)
  IL_0000:  ldloca.s   V_2
  IL_0002:  initobj    "T"
  IL_0008:  ldloc.2
  IL_0009:  box        "T"
  IL_000e:  brtrue.s   IL_0016
  IL_0010:  ldarg.0
  IL_0011:  stloc.0
  IL_0012:  ldloca.s   V_0
  IL_0014:  br.s       IL_0018
  IL_0016:  ldarga.s   V_0
  IL_0018:  ldarga.s   V_0
  IL_001a:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_001f:  dup
  IL_0020:  stloc.1
  IL_0021:  ldloca.s   V_2
  IL_0023:  initobj    "T"
  IL_0029:  ldloc.2
  IL_002a:  box        "T"
  IL_002f:  brtrue.s   IL_0035
  IL_0031:  ldloca.s   V_0
  IL_0033:  br.s       IL_0037
  IL_0035:  ldarga.s   V_0
  IL_0037:  ldloc.1
  IL_0038:  constrained. "T"
  IL_003e:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
  IL_0043:  ldarga.s   V_0
  IL_0045:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_004a:  ldc.i4.1
  IL_004b:  add.ovf
  IL_004c:  newarr     "Integer"
  IL_0051:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
  IL_0056:  castclass  "Integer()"
  IL_005b:  constrained. "T"
  IL_0061:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
  IL_0066:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Struct_IndexAndValue()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(item As T)
        Redim Preserve item.Position(GetOffset(item))(GetOffset(item))
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        Redim Preserve item.Position(GetOffset(item))(GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-2'
Position get for item '-3'
Position set for item '-4'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       61 (0x3d)
  .maxstack  5
  .locals init (Integer V_0)
  IL_0000:  ldarga.s   V_0
  IL_0002:  ldarga.s   V_0
  IL_0004:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0009:  dup
  IL_000a:  stloc.0
  IL_000b:  ldarga.s   V_0
  IL_000d:  ldloc.0
  IL_000e:  constrained. "T"
  IL_0014:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
  IL_0019:  ldarga.s   V_0
  IL_001b:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0020:  ldc.i4.1
  IL_0021:  add.ovf
  IL_0022:  newarr     "Integer"
  IL_0027:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
  IL_002c:  castclass  "Integer()"
  IL_0031:  constrained. "T"
  IL_0037:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
  IL_003c:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Class_IndexAndValue_Ref()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(ByRef item As T)
        Redim Preserve item.Position(GetOffset(item))(GetOffset(item))
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        Redim Preserve item.Position(GetOffset(item))(GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       64 (0x40)
  .maxstack  5
  .locals init (T V_0,
                Integer V_1)
  IL_0000:  ldarg.0
  IL_0001:  ldobj      "T"
  IL_0006:  dup
  IL_0007:  stloc.0
  IL_0008:  box        "T"
  IL_000d:  ldarg.0
  IL_000e:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0013:  dup
  IL_0014:  stloc.1
  IL_0015:  ldloca.s   V_0
  IL_0017:  ldloc.1
  IL_0018:  constrained. "T"
  IL_001e:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
  IL_0023:  ldarg.0
  IL_0024:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0029:  ldc.i4.1
  IL_002a:  add.ovf
  IL_002b:  newarr     "Integer"
  IL_0030:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
  IL_0035:  castclass  "Integer()"
  IL_003a:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
  IL_003f:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
      // Code size      104 (0x68)
  .maxstack  5
  .locals init (T V_0,
                Integer V_1,
                T V_2)
  IL_0000:  ldloca.s   V_2
  IL_0002:  initobj    "T"
  IL_0008:  ldloc.2
  IL_0009:  box        "T"
  IL_000e:  brtrue.s   IL_001b
  IL_0010:  ldarg.0
  IL_0011:  ldobj      "T"
  IL_0016:  stloc.0
  IL_0017:  ldloca.s   V_0
  IL_0019:  br.s       IL_001c
  IL_001b:  ldarg.0
  IL_001c:  ldarg.0
  IL_001d:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0022:  dup
  IL_0023:  stloc.1
  IL_0024:  ldloca.s   V_2
  IL_0026:  initobj    "T"
  IL_002c:  ldloc.2
  IL_002d:  box        "T"
  IL_0032:  brtrue.s   IL_0038
  IL_0034:  ldloca.s   V_0
  IL_0036:  br.s       IL_0039
  IL_0038:  ldarg.0
  IL_0039:  ldloc.1
  IL_003a:  constrained. "T"
  IL_0040:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
  IL_0045:  ldarg.0
  IL_0046:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_004b:  ldc.i4.1
  IL_004c:  add.ovf
  IL_004d:  newarr     "Integer"
  IL_0052:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
  IL_0057:  castclass  "Integer()"
  IL_005c:  constrained. "T"
  IL_0062:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
  IL_0067:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Struct_IndexAndValue_Ref()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(ByRef item As T)
        Redim Preserve item.Position(GetOffset(item))(GetOffset(item))
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        Redim Preserve item.Position(GetOffset(item))(GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-2'
Position get for item '-3'
Position set for item '-4'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       57 (0x39)
  .maxstack  5
  .locals init (Integer V_0)
  IL_0000:  ldarg.0
  IL_0001:  ldarg.0
  IL_0002:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0007:  dup
  IL_0008:  stloc.0
  IL_0009:  ldarg.0
  IL_000a:  ldloc.0
  IL_000b:  constrained. "T"
  IL_0011:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
  IL_0016:  ldarg.0
  IL_0017:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_001c:  ldc.i4.1
  IL_001d:  add.ovf
  IL_001e:  newarr     "Integer"
  IL_0023:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
  IL_0028:  castclass  "Integer()"
  IL_002d:  constrained. "T"
  IL_0033:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
  IL_0038:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Class_IndexAndValue_Async_01()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        Redim Preserve item.Position(GetOffset(item))(await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        Redim Preserve item.Position(GetOffset(item))(await GetOffsetAsync(GetOffset(item)))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      321 (0x141)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                SM$T V_2,
                Integer V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse    IL_009a
    IL_000d:  ldarg.0
    IL_000e:  ldarg.0
    IL_000f:  ldarg.0
    IL_0010:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0015:  dup
    IL_0016:  stloc.2
    IL_0017:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_001c:  ldloc.2
    IL_001d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_0022:  ldarg.0
    IL_0023:  ldarg.0
    IL_0024:  ldarg.0
    IL_0025:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_002a:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_002f:  dup
    IL_0030:  stloc.3
    IL_0031:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S1 As Integer"
    IL_0036:  ldloc.3
    IL_0037:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U3 As Integer"
    IL_003c:  ldarg.0
    IL_003d:  ldarg.0
    IL_003e:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0043:  box        "SM$T"
    IL_0048:  ldarg.0
    IL_0049:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S1 As Integer"
    IL_004e:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
    IL_0053:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As System.Array"
    IL_0058:  ldarg.0
    IL_0059:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_005e:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0063:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0068:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_006d:  stloc.1
    IL_006e:  ldloca.s   V_1
    IL_0070:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0075:  brtrue.s   IL_00b6
    IL_0077:  ldarg.0
    IL_0078:  ldc.i4.0
    IL_0079:  dup
    IL_007a:  stloc.0
    IL_007b:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0080:  ldarg.0
    IL_0081:  ldloc.1
    IL_0082:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0087:  ldarg.0
    IL_0088:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_008d:  ldloca.s   V_1
    IL_008f:  ldarg.0
    IL_0090:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0095:  leave      IL_0140
    IL_009a:  ldarg.0
    IL_009b:  ldc.i4.m1
    IL_009c:  dup
    IL_009d:  stloc.0
    IL_009e:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00a3:  ldarg.0
    IL_00a4:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a9:  stloc.1
    IL_00aa:  ldarg.0
    IL_00ab:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b0:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b6:  ldarg.0
    IL_00b7:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_00bc:  box        "SM$T"
    IL_00c1:  ldarg.0
    IL_00c2:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U3 As Integer"
    IL_00c7:  ldarg.0
    IL_00c8:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As System.Array"
    IL_00cd:  ldloca.s   V_1
    IL_00cf:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00d4:  ldloca.s   V_1
    IL_00d6:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00dc:  ldc.i4.1
    IL_00dd:  add.ovf
    IL_00de:  newarr     "Integer"
    IL_00e3:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
    IL_00e8:  castclass  "Integer()"
    IL_00ed:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
    IL_00f2:  ldarg.0
    IL_00f3:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_00f8:  initobj    "SM$T"
    IL_00fe:  ldarg.0
    IL_00ff:  ldnull
    IL_0100:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As System.Array"
    IL_0105:  leave.s    IL_012b
  }
  catch System.Exception
  {
    IL_0107:  dup
    IL_0108:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_010d:  stloc.s    V_4
    IL_010f:  ldarg.0
    IL_0110:  ldc.i4.s   -2
    IL_0112:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0117:  ldarg.0
    IL_0118:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_011d:  ldloc.s    V_4
    IL_011f:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0124:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0129:  leave.s    IL_0140
  }
  IL_012b:  ldarg.0
  IL_012c:  ldc.i4.s   -2
  IL_012e:  dup
  IL_012f:  stloc.0
  IL_0130:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0135:  ldarg.0
  IL_0136:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_013b:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0140:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      366 (0x16e)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                SM$T V_2,
                Integer V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse    IL_00ba
    IL_000d:  ldloca.s   V_2
    IL_000f:  initobj    "SM$T"
    IL_0015:  ldloc.2
    IL_0016:  box        "SM$T"
    IL_001b:  brtrue.s   IL_0029
    IL_001d:  ldarg.0
    IL_001e:  ldarg.0
    IL_001f:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0024:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_0029:  ldarg.0
    IL_002a:  ldarg.0
    IL_002b:  ldarg.0
    IL_002c:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0031:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0036:  dup
    IL_0037:  stloc.3
    IL_0038:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S1 As Integer"
    IL_003d:  ldloc.3
    IL_003e:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As Integer"
    IL_0043:  ldarg.0
    IL_0044:  ldloca.s   V_2
    IL_0046:  initobj    "SM$T"
    IL_004c:  ldloc.2
    IL_004d:  box        "SM$T"
    IL_0052:  brtrue.s   IL_005c
    IL_0054:  ldarg.0
    IL_0055:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_005a:  br.s       IL_0062
    IL_005c:  ldarg.0
    IL_005d:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0062:  ldarg.0
    IL_0063:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S1 As Integer"
    IL_0068:  constrained. "SM$T"
    IL_006e:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
    IL_0073:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As System.Array"
    IL_0078:  ldarg.0
    IL_0079:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_007e:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0083:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0088:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008d:  stloc.1
    IL_008e:  ldloca.s   V_1
    IL_0090:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0095:  brtrue.s   IL_00d6
    IL_0097:  ldarg.0
    IL_0098:  ldc.i4.0
    IL_0099:  dup
    IL_009a:  stloc.0
    IL_009b:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00a0:  ldarg.0
    IL_00a1:  ldloc.1
    IL_00a2:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a7:  ldarg.0
    IL_00a8:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00ad:  ldloca.s   V_1
    IL_00af:  ldarg.0
    IL_00b0:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_00b5:  leave      IL_016d
    IL_00ba:  ldarg.0
    IL_00bb:  ldc.i4.m1
    IL_00bc:  dup
    IL_00bd:  stloc.0
    IL_00be:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00c3:  ldarg.0
    IL_00c4:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00c9:  stloc.1
    IL_00ca:  ldarg.0
    IL_00cb:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d0:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d6:  ldloca.s   V_2
    IL_00d8:  initobj    "SM$T"
    IL_00de:  ldloc.2
    IL_00df:  box        "SM$T"
    IL_00e4:  brtrue.s   IL_00ee
    IL_00e6:  ldarg.0
    IL_00e7:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_00ec:  br.s       IL_00f4
    IL_00ee:  ldarg.0
    IL_00ef:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00f4:  ldarg.0
    IL_00f5:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As Integer"
    IL_00fa:  ldarg.0
    IL_00fb:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As System.Array"
    IL_0100:  ldloca.s   V_1
    IL_0102:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0107:  ldloca.s   V_1
    IL_0109:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_010f:  ldc.i4.1
    IL_0110:  add.ovf
    IL_0111:  newarr     "Integer"
    IL_0116:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
    IL_011b:  castclass  "Integer()"
    IL_0120:  constrained. "SM$T"
    IL_0126:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
    IL_012b:  ldarg.0
    IL_012c:  ldnull
    IL_012d:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As System.Array"
    IL_0132:  leave.s    IL_0158
  }
  catch System.Exception
  {
    IL_0134:  dup
    IL_0135:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_013a:  stloc.s    V_4
    IL_013c:  ldarg.0
    IL_013d:  ldc.i4.s   -2
    IL_013f:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0144:  ldarg.0
    IL_0145:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_014a:  ldloc.s    V_4
    IL_014c:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0151:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0156:  leave.s    IL_016d
  }
  IL_0158:  ldarg.0
  IL_0159:  ldc.i4.s   -2
  IL_015b:  dup
  IL_015c:  stloc.0
  IL_015d:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0162:  ldarg.0
  IL_0163:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0168:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_016d:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Struct_IndexAndValue_Async_01()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        Redim Preserve item.Position(GetOffset(item))(await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        Redim Preserve item.Position(GetOffset(item))(await GetOffsetAsync(GetOffset(item)))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-2'
Position get for item '-3'
Position set for item '-4'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      285 (0x11d)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                Integer V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0083
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldarg.0
    IL_000d:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0012:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0017:  dup
    IL_0018:  stloc.2
    IL_0019:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As Integer"
    IL_001e:  ldloc.2
    IL_001f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As Integer"
    IL_0024:  ldarg.0
    IL_0025:  ldarg.0
    IL_0026:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_002b:  ldarg.0
    IL_002c:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As Integer"
    IL_0031:  constrained. "SM$T"
    IL_0037:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
    IL_003c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As System.Array"
    IL_0041:  ldarg.0
    IL_0042:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0047:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_004c:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0051:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0056:  stloc.1
    IL_0057:  ldloca.s   V_1
    IL_0059:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_005e:  brtrue.s   IL_009f
    IL_0060:  ldarg.0
    IL_0061:  ldc.i4.0
    IL_0062:  dup
    IL_0063:  stloc.0
    IL_0064:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0069:  ldarg.0
    IL_006a:  ldloc.1
    IL_006b:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0070:  ldarg.0
    IL_0071:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0076:  ldloca.s   V_1
    IL_0078:  ldarg.0
    IL_0079:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_007e:  leave      IL_011c
    IL_0083:  ldarg.0
    IL_0084:  ldc.i4.m1
    IL_0085:  dup
    IL_0086:  stloc.0
    IL_0087:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_008c:  ldarg.0
    IL_008d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0092:  stloc.1
    IL_0093:  ldarg.0
    IL_0094:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0099:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009f:  ldarg.0
    IL_00a0:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_00a5:  ldarg.0
    IL_00a6:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As Integer"
    IL_00ab:  ldarg.0
    IL_00ac:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As System.Array"
    IL_00b1:  ldloca.s   V_1
    IL_00b3:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00b8:  ldloca.s   V_1
    IL_00ba:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00c0:  ldc.i4.1
    IL_00c1:  add.ovf
    IL_00c2:  newarr     "Integer"
    IL_00c7:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
    IL_00cc:  castclass  "Integer()"
    IL_00d1:  constrained. "SM$T"
    IL_00d7:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
    IL_00dc:  ldarg.0
    IL_00dd:  ldnull
    IL_00de:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As System.Array"
    IL_00e3:  leave.s    IL_0107
  }
  catch System.Exception
  {
    IL_00e5:  dup
    IL_00e6:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00eb:  stloc.3
    IL_00ec:  ldarg.0
    IL_00ed:  ldc.i4.s   -2
    IL_00ef:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00f4:  ldarg.0
    IL_00f5:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00fa:  ldloc.3
    IL_00fb:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0100:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0105:  leave.s    IL_011c
  }
  IL_0107:  ldarg.0
  IL_0108:  ldc.i4.s   -2
  IL_010a:  dup
  IL_010b:  stloc.0
  IL_010c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0111:  ldarg.0
  IL_0112:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0117:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_011c:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Class_IndexAndValue_Async_03()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        Redim Preserve item.Position(await GetOffsetAsync(GetOffset(item)))(GetOffset(item))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        Redim Preserve item.Position(await GetOffsetAsync(GetOffset(item)))(GetOffset(item))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      276 (0x114)
  .maxstack  5
  .locals init (Integer V_0,
                Integer V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0061
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldarg.0
    IL_000d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0012:  dup
    IL_0013:  stloc.3
    IL_0014:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0019:  ldloc.3
    IL_001a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_001f:  ldarg.0
    IL_0020:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0025:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_002a:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_002f:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0034:  stloc.2
    IL_0035:  ldloca.s   V_2
    IL_0037:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_003c:  brtrue.s   IL_007d
    IL_003e:  ldarg.0
    IL_003f:  ldc.i4.0
    IL_0040:  dup
    IL_0041:  stloc.0
    IL_0042:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0047:  ldarg.0
    IL_0048:  ldloc.2
    IL_0049:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_004e:  ldarg.0
    IL_004f:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0054:  ldloca.s   V_2
    IL_0056:  ldarg.0
    IL_0057:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_005c:  leave      IL_0113
    IL_0061:  ldarg.0
    IL_0062:  ldc.i4.m1
    IL_0063:  dup
    IL_0064:  stloc.0
    IL_0065:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_006a:  ldarg.0
    IL_006b:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0070:  stloc.2
    IL_0071:  ldarg.0
    IL_0072:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0077:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007d:  ldarg.0
    IL_007e:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0083:  box        "SM$T"
    IL_0088:  ldloca.s   V_2
    IL_008a:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_008f:  ldloca.s   V_2
    IL_0091:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0097:  dup
    IL_0098:  stloc.1
    IL_0099:  ldarg.0
    IL_009a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_009f:  ldloc.1
    IL_00a0:  constrained. "SM$T"
    IL_00a6:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
    IL_00ab:  ldarg.0
    IL_00ac:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_00b1:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_00b6:  ldc.i4.1
    IL_00b7:  add.ovf
    IL_00b8:  newarr     "Integer"
    IL_00bd:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
    IL_00c2:  castclass  "Integer()"
    IL_00c7:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
    IL_00cc:  ldarg.0
    IL_00cd:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_00d2:  initobj    "SM$T"
    IL_00d8:  leave.s    IL_00fe
  }
  catch System.Exception
  {
    IL_00da:  dup
    IL_00db:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00e0:  stloc.s    V_4
    IL_00e2:  ldarg.0
    IL_00e3:  ldc.i4.s   -2
    IL_00e5:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00ea:  ldarg.0
    IL_00eb:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00f0:  ldloc.s    V_4
    IL_00f2:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00f7:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00fc:  leave.s    IL_0113
  }
  IL_00fe:  ldarg.0
  IL_00ff:  ldc.i4.s   -2
  IL_0101:  dup
  IL_0102:  stloc.0
  IL_0103:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0108:  ldarg.0
  IL_0109:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_010e:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0113:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      320 (0x140)
  .maxstack  5
  .locals init (Integer V_0,
                Integer V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0068
    IL_000a:  ldloca.s   V_3
    IL_000c:  initobj    "SM$T"
    IL_0012:  ldloc.3
    IL_0013:  box        "SM$T"
    IL_0018:  brtrue.s   IL_0026
    IL_001a:  ldarg.0
    IL_001b:  ldarg.0
    IL_001c:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0021:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_0026:  ldarg.0
    IL_0027:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_002c:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0031:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0036:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_003b:  stloc.2
    IL_003c:  ldloca.s   V_2
    IL_003e:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0043:  brtrue.s   IL_0084
    IL_0045:  ldarg.0
    IL_0046:  ldc.i4.0
    IL_0047:  dup
    IL_0048:  stloc.0
    IL_0049:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_004e:  ldarg.0
    IL_004f:  ldloc.2
    IL_0050:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0055:  ldarg.0
    IL_0056:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_005b:  ldloca.s   V_2
    IL_005d:  ldarg.0
    IL_005e:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_0063:  leave      IL_013f
    IL_0068:  ldarg.0
    IL_0069:  ldc.i4.m1
    IL_006a:  dup
    IL_006b:  stloc.0
    IL_006c:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0071:  ldarg.0
    IL_0072:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0077:  stloc.2
    IL_0078:  ldarg.0
    IL_0079:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007e:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0084:  ldloca.s   V_3
    IL_0086:  initobj    "SM$T"
    IL_008c:  ldloc.3
    IL_008d:  box        "SM$T"
    IL_0092:  brtrue.s   IL_009c
    IL_0094:  ldarg.0
    IL_0095:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_009a:  br.s       IL_00a2
    IL_009c:  ldarg.0
    IL_009d:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00a2:  ldloca.s   V_2
    IL_00a4:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00a9:  ldloca.s   V_2
    IL_00ab:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b1:  dup
    IL_00b2:  stloc.1
    IL_00b3:  ldloca.s   V_3
    IL_00b5:  initobj    "SM$T"
    IL_00bb:  ldloc.3
    IL_00bc:  box        "SM$T"
    IL_00c1:  brtrue.s   IL_00cb
    IL_00c3:  ldarg.0
    IL_00c4:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_00c9:  br.s       IL_00d1
    IL_00cb:  ldarg.0
    IL_00cc:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00d1:  ldloc.1
    IL_00d2:  constrained. "SM$T"
    IL_00d8:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
    IL_00dd:  ldarg.0
    IL_00de:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00e3:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_00e8:  ldc.i4.1
    IL_00e9:  add.ovf
    IL_00ea:  newarr     "Integer"
    IL_00ef:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
    IL_00f4:  castclass  "Integer()"
    IL_00f9:  constrained. "SM$T"
    IL_00ff:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
    IL_0104:  leave.s    IL_012a
  }
  catch System.Exception
  {
    IL_0106:  dup
    IL_0107:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_010c:  stloc.s    V_4
    IL_010e:  ldarg.0
    IL_010f:  ldc.i4.s   -2
    IL_0111:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0116:  ldarg.0
    IL_0117:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_011c:  ldloc.s    V_4
    IL_011e:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0123:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0128:  leave.s    IL_013f
  }
  IL_012a:  ldarg.0
  IL_012b:  ldc.i4.s   -2
  IL_012d:  dup
  IL_012e:  stloc.0
  IL_012f:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0134:  ldarg.0
  IL_0135:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_013a:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_013f:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Struct_IndexAndValue_Async_03()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        Redim Preserve item.Position(await GetOffsetAsync(GetOffset(item)))(GetOffset(item))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        Redim Preserve item.Position(await GetOffsetAsync(GetOffset(item)))(GetOffset(item))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-2'
Position get for item '-3'
Position set for item '-4'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
    // Code size      242 (0xf2)
    .maxstack  5
    .locals init (Integer V_0,
                Integer V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                System.Exception V_3)
    IL_0000:  ldarg.0
    IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0006:  stloc.0
    .try
    {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004c
    IL_000a:  ldarg.0
    IL_000b:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0010:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0015:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_001a:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_001f:  stloc.2
    IL_0020:  ldloca.s   V_2
    IL_0022:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0027:  brtrue.s   IL_0068
    IL_0029:  ldarg.0
    IL_002a:  ldc.i4.0
    IL_002b:  dup
    IL_002c:  stloc.0
    IL_002d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0032:  ldarg.0
    IL_0033:  ldloc.2
    IL_0034:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0039:  ldarg.0
    IL_003a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003f:  ldloca.s   V_2
    IL_0041:  ldarg.0
    IL_0042:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0047:  leave      IL_00f1
    IL_004c:  ldarg.0
    IL_004d:  ldc.i4.m1
    IL_004e:  dup
    IL_004f:  stloc.0
    IL_0050:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0055:  ldarg.0
    IL_0056:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_005b:  stloc.2
    IL_005c:  ldarg.0
    IL_005d:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0062:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0068:  ldarg.0
    IL_0069:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_006e:  ldloca.s   V_2
    IL_0070:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0075:  ldloca.s   V_2
    IL_0077:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007d:  dup
    IL_007e:  stloc.1
    IL_007f:  ldarg.0
    IL_0080:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0085:  ldloc.1
    IL_0086:  constrained. "SM$T"
    IL_008c:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
    IL_0091:  ldarg.0
    IL_0092:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0097:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_009c:  ldc.i4.1
    IL_009d:  add.ovf
    IL_009e:  newarr     "Integer"
    IL_00a3:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
    IL_00a8:  castclass  "Integer()"
    IL_00ad:  constrained. "SM$T"
    IL_00b3:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
    IL_00b8:  leave.s    IL_00dc
    }
    catch System.Exception
    {
    IL_00ba:  dup
    IL_00bb:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00c0:  stloc.3
    IL_00c1:  ldarg.0
    IL_00c2:  ldc.i4.s   -2
    IL_00c4:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00c9:  ldarg.0
    IL_00ca:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00cf:  ldloc.3
    IL_00d0:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00d5:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00da:  leave.s    IL_00f1
    }
    IL_00dc:  ldarg.0
    IL_00dd:  ldc.i4.s   -2
    IL_00df:  dup
    IL_00e0:  stloc.0
    IL_00e1:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00e6:  ldarg.0
    IL_00e7:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00ec:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
    IL_00f1:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Class_IndexAndValue_Async_04()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        Redim Preserve item.Position(await GetOffsetAsync(GetOffset(item)))(await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        Redim Preserve item.Position(await GetOffsetAsync(GetOffset(item)))(await GetOffsetAsync(GetOffset(item)))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      425 (0x1a9)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T V_3,
                Integer V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0068
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_0102
    IL_0011:  ldarg.0
    IL_0012:  ldarg.0
    IL_0013:  ldarg.0
    IL_0014:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0019:  dup
    IL_001a:  stloc.3
    IL_001b:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0020:  ldloc.3
    IL_0021:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_0026:  ldarg.0
    IL_0027:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_002c:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0031:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0036:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_003b:  stloc.1
    IL_003c:  ldloca.s   V_1
    IL_003e:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0043:  brtrue.s   IL_0084
    IL_0045:  ldarg.0
    IL_0046:  ldc.i4.0
    IL_0047:  dup
    IL_0048:  stloc.0
    IL_0049:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_004e:  ldarg.0
    IL_004f:  ldloc.1
    IL_0050:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0055:  ldarg.0
    IL_0056:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_005b:  ldloca.s   V_1
    IL_005d:  ldarg.0
    IL_005e:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0063:  leave      IL_01a8
    IL_0068:  ldarg.0
    IL_0069:  ldc.i4.m1
    IL_006a:  dup
    IL_006b:  stloc.0
    IL_006c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0071:  ldarg.0
    IL_0072:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0077:  stloc.1
    IL_0078:  ldarg.0
    IL_0079:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007e:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0084:  ldarg.0
    IL_0085:  ldarg.0
    IL_0086:  ldloca.s   V_1
    IL_0088:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_008d:  ldloca.s   V_1
    IL_008f:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0095:  dup
    IL_0096:  stloc.s    V_4
    IL_0098:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S1 As Integer"
    IL_009d:  ldloc.s    V_4
    IL_009f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U3 As Integer"
    IL_00a4:  ldarg.0
    IL_00a5:  ldarg.0
    IL_00a6:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_00ab:  box        "SM$T"
    IL_00b0:  ldarg.0
    IL_00b1:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S1 As Integer"
    IL_00b6:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
    IL_00bb:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As System.Array"
    IL_00c0:  ldarg.0
    IL_00c1:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_00c6:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_00cb:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_00d0:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d5:  stloc.2
    IL_00d6:  ldloca.s   V_2
    IL_00d8:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_00dd:  brtrue.s   IL_011e
    IL_00df:  ldarg.0
    IL_00e0:  ldc.i4.1
    IL_00e1:  dup
    IL_00e2:  stloc.0
    IL_00e3:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00e8:  ldarg.0
    IL_00e9:  ldloc.2
    IL_00ea:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00ef:  ldarg.0
    IL_00f0:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00f5:  ldloca.s   V_2
    IL_00f7:  ldarg.0
    IL_00f8:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_00fd:  leave      IL_01a8
    IL_0102:  ldarg.0
    IL_0103:  ldc.i4.m1
    IL_0104:  dup
    IL_0105:  stloc.0
    IL_0106:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_010b:  ldarg.0
    IL_010c:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0111:  stloc.2
    IL_0112:  ldarg.0
    IL_0113:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0118:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_011e:  ldarg.0
    IL_011f:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_0124:  box        "SM$T"
    IL_0129:  ldarg.0
    IL_012a:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U3 As Integer"
    IL_012f:  ldarg.0
    IL_0130:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As System.Array"
    IL_0135:  ldloca.s   V_2
    IL_0137:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_013c:  ldloca.s   V_2
    IL_013e:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0144:  ldc.i4.1
    IL_0145:  add.ovf
    IL_0146:  newarr     "Integer"
    IL_014b:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
    IL_0150:  castclass  "Integer()"
    IL_0155:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
    IL_015a:  ldarg.0
    IL_015b:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As SM$T"
    IL_0160:  initobj    "SM$T"
    IL_0166:  ldarg.0
    IL_0167:  ldnull
    IL_0168:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As System.Array"
    IL_016d:  leave.s    IL_0193
  }
  catch System.Exception
  {
    IL_016f:  dup
    IL_0170:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0175:  stloc.s    V_5
    IL_0177:  ldarg.0
    IL_0178:  ldc.i4.s   -2
    IL_017a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_017f:  ldarg.0
    IL_0180:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0185:  ldloc.s    V_5
    IL_0187:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_018c:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0191:  leave.s    IL_01a8
  }
  IL_0193:  ldarg.0
  IL_0194:  ldc.i4.s   -2
  IL_0196:  dup
  IL_0197:  stloc.0
  IL_0198:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_019d:  ldarg.0
  IL_019e:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_01a3:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_01a8:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      470 (0x1d6)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T V_3,
                Integer V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_006f
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_0122
    IL_0011:  ldloca.s   V_3
    IL_0013:  initobj    "SM$T"
    IL_0019:  ldloc.3
    IL_001a:  box        "SM$T"
    IL_001f:  brtrue.s   IL_002d
    IL_0021:  ldarg.0
    IL_0022:  ldarg.0
    IL_0023:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0028:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_002d:  ldarg.0
    IL_002e:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0033:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0038:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_003d:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0042:  stloc.1
    IL_0043:  ldloca.s   V_1
    IL_0045:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_004a:  brtrue.s   IL_008b
    IL_004c:  ldarg.0
    IL_004d:  ldc.i4.0
    IL_004e:  dup
    IL_004f:  stloc.0
    IL_0050:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0055:  ldarg.0
    IL_0056:  ldloc.1
    IL_0057:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_005c:  ldarg.0
    IL_005d:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0062:  ldloca.s   V_1
    IL_0064:  ldarg.0
    IL_0065:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_006a:  leave      IL_01d5
    IL_006f:  ldarg.0
    IL_0070:  ldc.i4.m1
    IL_0071:  dup
    IL_0072:  stloc.0
    IL_0073:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0078:  ldarg.0
    IL_0079:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007e:  stloc.1
    IL_007f:  ldarg.0
    IL_0080:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0085:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008b:  ldarg.0
    IL_008c:  ldarg.0
    IL_008d:  ldloca.s   V_1
    IL_008f:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0094:  ldloca.s   V_1
    IL_0096:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009c:  dup
    IL_009d:  stloc.s    V_4
    IL_009f:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S1 As Integer"
    IL_00a4:  ldloc.s    V_4
    IL_00a6:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As Integer"
    IL_00ab:  ldarg.0
    IL_00ac:  ldloca.s   V_3
    IL_00ae:  initobj    "SM$T"
    IL_00b4:  ldloc.3
    IL_00b5:  box        "SM$T"
    IL_00ba:  brtrue.s   IL_00c4
    IL_00bc:  ldarg.0
    IL_00bd:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_00c2:  br.s       IL_00ca
    IL_00c4:  ldarg.0
    IL_00c5:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00ca:  ldarg.0
    IL_00cb:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S1 As Integer"
    IL_00d0:  constrained. "SM$T"
    IL_00d6:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
    IL_00db:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As System.Array"
    IL_00e0:  ldarg.0
    IL_00e1:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00e6:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_00eb:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_00f0:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00f5:  stloc.2
    IL_00f6:  ldloca.s   V_2
    IL_00f8:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_00fd:  brtrue.s   IL_013e
    IL_00ff:  ldarg.0
    IL_0100:  ldc.i4.1
    IL_0101:  dup
    IL_0102:  stloc.0
    IL_0103:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0108:  ldarg.0
    IL_0109:  ldloc.2
    IL_010a:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_010f:  ldarg.0
    IL_0110:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0115:  ldloca.s   V_2
    IL_0117:  ldarg.0
    IL_0118:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_011d:  leave      IL_01d5
    IL_0122:  ldarg.0
    IL_0123:  ldc.i4.m1
    IL_0124:  dup
    IL_0125:  stloc.0
    IL_0126:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_012b:  ldarg.0
    IL_012c:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0131:  stloc.2
    IL_0132:  ldarg.0
    IL_0133:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0138:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_013e:  ldloca.s   V_3
    IL_0140:  initobj    "SM$T"
    IL_0146:  ldloc.3
    IL_0147:  box        "SM$T"
    IL_014c:  brtrue.s   IL_0156
    IL_014e:  ldarg.0
    IL_014f:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_0154:  br.s       IL_015c
    IL_0156:  ldarg.0
    IL_0157:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_015c:  ldarg.0
    IL_015d:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U2 As Integer"
    IL_0162:  ldarg.0
    IL_0163:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As System.Array"
    IL_0168:  ldloca.s   V_2
    IL_016a:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_016f:  ldloca.s   V_2
    IL_0171:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0177:  ldc.i4.1
    IL_0178:  add.ovf
    IL_0179:  newarr     "Integer"
    IL_017e:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
    IL_0183:  castclass  "Integer()"
    IL_0188:  constrained. "SM$T"
    IL_018e:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
    IL_0193:  ldarg.0
    IL_0194:  ldnull
    IL_0195:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$U1 As System.Array"
    IL_019a:  leave.s    IL_01c0
  }
  catch System.Exception
  {
    IL_019c:  dup
    IL_019d:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_01a2:  stloc.s    V_5
    IL_01a4:  ldarg.0
    IL_01a5:  ldc.i4.s   -2
    IL_01a7:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_01ac:  ldarg.0
    IL_01ad:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_01b2:  ldloc.s    V_5
    IL_01b4:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_01b9:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_01be:  leave.s    IL_01d5
  }
  IL_01c0:  ldarg.0
  IL_01c1:  ldc.i4.s   -2
  IL_01c3:  dup
  IL_01c4:  stloc.0
  IL_01c5:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_01ca:  ldarg.0
  IL_01cb:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_01d0:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_01d5:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_Redim_Indexer_Struct_IndexAndValue_Async_04()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer()
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer() Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return Nothing
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        Redim Preserve item.Position(await GetOffsetAsync(GetOffset(item)))(await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        Redim Preserve item.Position(await GetOffsetAsync(GetOffset(item)))(await GetOffsetAsync(GetOffset(item)))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return Nothing
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-2'
Position get for item '-3'
Position set for item '-4'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      392 (0x188)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                Integer V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0053
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_00ec
    IL_0011:  ldarg.0
    IL_0012:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0017:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_001c:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0021:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0026:  stloc.1
    IL_0027:  ldloca.s   V_1
    IL_0029:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_002e:  brtrue.s   IL_006f
    IL_0030:  ldarg.0
    IL_0031:  ldc.i4.0
    IL_0032:  dup
    IL_0033:  stloc.0
    IL_0034:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0039:  ldarg.0
    IL_003a:  ldloc.1
    IL_003b:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0040:  ldarg.0
    IL_0041:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0046:  ldloca.s   V_1
    IL_0048:  ldarg.0
    IL_0049:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_004e:  leave      IL_0187
    IL_0053:  ldarg.0
    IL_0054:  ldc.i4.m1
    IL_0055:  dup
    IL_0056:  stloc.0
    IL_0057:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_005c:  ldarg.0
    IL_005d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0062:  stloc.1
    IL_0063:  ldarg.0
    IL_0064:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0069:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_006f:  ldarg.0
    IL_0070:  ldarg.0
    IL_0071:  ldloca.s   V_1
    IL_0073:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0078:  ldloca.s   V_1
    IL_007a:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0080:  dup
    IL_0081:  stloc.3
    IL_0082:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As Integer"
    IL_0087:  ldloc.3
    IL_0088:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As Integer"
    IL_008d:  ldarg.0
    IL_008e:  ldarg.0
    IL_008f:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0094:  ldarg.0
    IL_0095:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As Integer"
    IL_009a:  constrained. "SM$T"
    IL_00a0:  callvirt   "Function IMoveable.get_Position(Integer) As Integer()"
    IL_00a5:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As System.Array"
    IL_00aa:  ldarg.0
    IL_00ab:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_00b0:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_00b5:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_00ba:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00bf:  stloc.2
    IL_00c0:  ldloca.s   V_2
    IL_00c2:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_00c7:  brtrue.s   IL_0108
    IL_00c9:  ldarg.0
    IL_00ca:  ldc.i4.1
    IL_00cb:  dup
    IL_00cc:  stloc.0
    IL_00cd:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00d2:  ldarg.0
    IL_00d3:  ldloc.2
    IL_00d4:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d9:  ldarg.0
    IL_00da:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00df:  ldloca.s   V_2
    IL_00e1:  ldarg.0
    IL_00e2:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_00e7:  leave      IL_0187
    IL_00ec:  ldarg.0
    IL_00ed:  ldc.i4.m1
    IL_00ee:  dup
    IL_00ef:  stloc.0
    IL_00f0:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00f5:  ldarg.0
    IL_00f6:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00fb:  stloc.2
    IL_00fc:  ldarg.0
    IL_00fd:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0102:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0108:  ldarg.0
    IL_0109:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_010e:  ldarg.0
    IL_010f:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U2 As Integer"
    IL_0114:  ldarg.0
    IL_0115:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As System.Array"
    IL_011a:  ldloca.s   V_2
    IL_011c:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0121:  ldloca.s   V_2
    IL_0123:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0129:  ldc.i4.1
    IL_012a:  add.ovf
    IL_012b:  newarr     "Integer"
    IL_0130:  call       "Function Microsoft.VisualBasic.CompilerServices.Utils.CopyArray(System.Array, System.Array) As System.Array"
    IL_0135:  castclass  "Integer()"
    IL_013a:  constrained. "SM$T"
    IL_0140:  callvirt   "Sub IMoveable.set_Position(Integer, Integer())"
    IL_0145:  ldarg.0
    IL_0146:  ldnull
    IL_0147:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As System.Array"
    IL_014c:  leave.s    IL_0172
  }
  catch System.Exception
  {
    IL_014e:  dup
    IL_014f:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0154:  stloc.s    V_4
    IL_0156:  ldarg.0
    IL_0157:  ldc.i4.s   -2
    IL_0159:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_015e:  ldarg.0
    IL_015f:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0164:  ldloc.s    V_4
    IL_0166:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_016b:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0170:  leave.s    IL_0187
  }
  IL_0172:  ldarg.0
  IL_0173:  ldc.i4.s   -2
  IL_0175:  dup
  IL_0176:  stloc.0
  IL_0177:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_017c:  ldarg.0
  IL_017d:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0182:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0187:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Theory>
        <CombinatorialData>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Property_Class_Ref(asRValue As Boolean)

            Dim leftParen As String = ""
            Dim rightParen As String = ""

            If asRValue Then
                leftParen = "("
                rightParen = ")"
            End If

            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Item As Item

    Shared Sub Main()
        Item = New Item With {.Name = "1"}
        Call1(Item)

        Item = New Item With {.Name = "2"}
        Call2(Item)

        Item = New Item With {.Name = "3"}
        Call3(Item)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(ByRef item As T)
        M(<%= leftParen %>item<%= rightParen %>.Position)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        M(<%= leftParen %>item<%= rightParen %>.Position)
    End Sub

    Private Shared Sub Call3(ByRef item As Item)
        M(<%= leftParen %>item<%= rightParen %>.Position)
    End Sub

    Shared value as Integer

    Shared Sub M(ByRef x As Integer)
        value -= 1
        Item = New Item With {.Name = value.ToString()}
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
Position get for item '3'
Position set for item '3'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       37 (0x25)
  .maxstack  2
  .locals init (Integer V_0)
  IL_0000:  ldarg.0
  IL_0001:  ldobj      "T"
  IL_0006:  dup
  IL_0007:  box        "T"
  IL_000c:  callvirt   "Function IMoveable.get_Position() As Integer"
  IL_0011:  stloc.0
  IL_0012:  ldloca.s   V_0
  IL_0014:  call       "Sub Program.M(ByRef Integer)"
  IL_0019:  box        "T"
  IL_001e:  ldloc.0
  IL_001f:  callvirt   "Sub IMoveable.set_Position(Integer)"
  IL_0024:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
If(asRValue,
            <![CDATA[
{
      // Code size       43 (0x2b)
  .maxstack  2
  .locals init (T V_0,
                Integer V_1)
  IL_0000:  ldarg.0
  IL_0001:  ldobj      "T"
  IL_0006:  stloc.0
  IL_0007:  ldloca.s   V_0
  IL_0009:  constrained. "T"
  IL_000f:  callvirt   "Function IMoveable.get_Position() As Integer"
  IL_0014:  stloc.1
  IL_0015:  ldloca.s   V_1
  IL_0017:  call       "Sub Program.M(ByRef Integer)"
  IL_001c:  ldloca.s   V_0
  IL_001e:  ldloc.1
  IL_001f:  constrained. "T"
  IL_0025:  callvirt   "Sub IMoveable.set_Position(Integer)"
  IL_002a:  ret
}
]]>,
            <![CDATA[
{
      // Code size       81 (0x51)
  .maxstack  2
  .locals init (T V_0,
                Integer V_1,
                T V_2)
  IL_0000:  ldloca.s   V_2
  IL_0002:  initobj    "T"
  IL_0008:  ldloc.2
  IL_0009:  box        "T"
  IL_000e:  brtrue.s   IL_001b
  IL_0010:  ldarg.0
  IL_0011:  ldobj      "T"
  IL_0016:  stloc.0
  IL_0017:  ldloca.s   V_0
  IL_0019:  br.s       IL_001c
  IL_001b:  ldarg.0
  IL_001c:  constrained. "T"
  IL_0022:  callvirt   "Function IMoveable.get_Position() As Integer"
  IL_0027:  stloc.1
  IL_0028:  ldloca.s   V_1
  IL_002a:  call       "Sub Program.M(ByRef Integer)"
  IL_002f:  ldloca.s   V_2
  IL_0031:  initobj    "T"
  IL_0037:  ldloc.2
  IL_0038:  box        "T"
  IL_003d:  brtrue.s   IL_0043
  IL_003f:  ldloca.s   V_0
  IL_0041:  br.s       IL_0044
  IL_0043:  ldarg.0
  IL_0044:  ldloc.1
  IL_0045:  constrained. "T"
  IL_004b:  callvirt   "Sub IMoveable.set_Position(Integer)"
  IL_0050:  ret
}
]]>))
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Property_Struct_Ref()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Item As Item

    Shared Sub Main()
        Item = New Item With {.Name = "1"}
        Call1(Item)

        Item = New Item With {.Name = "2"}
        Call2(Item)

        Item = New Item With {.Name = "3"}
        Call3(Item)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(ByRef item As T)
        M(item.Position)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        M(item.Position)
    End Sub

    Private Shared Sub Call3(ByRef item As Item)
        M(item.Position)
    End Sub

    Shared value as Integer

    Shared Sub M(ByRef x As Integer)
        value -= 1
        Item = New Item With {.Name = value.ToString()}
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '-1'
Position get for item '2'
Position set for item '-2'
Position get for item '3'
Position set for item '-3'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       34 (0x22)
  .maxstack  2
  .locals init (Integer V_0)
  IL_0000:  ldarg.0
  IL_0001:  constrained. "T"
  IL_0007:  callvirt   "Function IMoveable.get_Position() As Integer"
  IL_000c:  stloc.0
  IL_000d:  ldloca.s   V_0
  IL_000f:  call       "Sub Program.M(ByRef Integer)"
  IL_0014:  ldarg.0
  IL_0015:  ldloc.0
  IL_0016:  constrained. "T"
  IL_001c:  callvirt   "Sub IMoveable.set_Position(Integer)"
  IL_0021:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Property_Struct_Ref_RValue()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
            Name = ""
        End Set
    End Property
End Structure

Class Program
    Shared Item As Item

    Shared Sub Main()
        Item = New Item With {.Name = "1"}
        Call1(Item)
        Console.WriteLine(Item.Name)

        Item = New Item With {.Name = "2"}
        Call2(Item)
        Console.WriteLine(Item.Name)

        Item = New Item With {.Name = "3"}
        Call3(Item)
        Console.WriteLine(Item.Name)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(ByRef item As T)
        M((item).Position)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        M((item).Position)
    End Sub

    Private Shared Sub Call3(ByRef item As Item)
        M((item).Position)
    End Sub

    Shared value as Integer

    Shared Sub M(ByRef x As Integer)
        value -= 1
        Item = New Item With {.Name = value.ToString()}
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
-1
Position get for item '2'
Position set for item '2'
-2
Position get for item '3'
Position set for item '3'
-3
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       45 (0x2d)
  .maxstack  2
  .locals init (T V_0,
                Integer V_1,
                T V_2)
  IL_0000:  ldarg.0
  IL_0001:  ldobj      "T"
  IL_0006:  dup
  IL_0007:  stloc.0
  IL_0008:  stloc.2
  IL_0009:  ldloca.s   V_2
  IL_000b:  constrained. "T"
  IL_0011:  callvirt   "Function IMoveable.get_Position() As Integer"
  IL_0016:  stloc.1
  IL_0017:  ldloca.s   V_1
  IL_0019:  call       "Sub Program.M(ByRef Integer)"
  IL_001e:  ldloca.s   V_0
  IL_0020:  ldloc.1
  IL_0021:  constrained. "T"
  IL_0027:  callvirt   "Sub IMoveable.set_Position(Integer)"
  IL_002c:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Theory>
        <CombinatorialData>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Property_Class_InIterator(asRValue As Boolean)

            Dim leftParen As String = ""
            Dim rightParen As String = ""

            If asRValue Then
                leftParen = "("
                rightParen = ")"
            End If

            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Collections

Interface IMoveable
    Property Position As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Item As Item

    Shared Sub Main()
        Item = New Item With {.Name = "1"}
        for each x in Call1(Item)
        next

        Item = New Item With {.Name = "2"}
        for each x in Call2(Item)
        next

        Item = New Item With {.Name = "3"}
        for each x in Call3(Item)
        next
    End Sub

    Private Shared Iterator Function Call1(Of T As {Class, IMoveable})(item As T) As IEnumerable
        Yield Nothing
        M(<%= leftParen %>item<%= rightParen %>.Position)
        Yield Nothing
    End Function

    Private Shared Iterator Function Call2(Of T As {IMoveable})(item As T) As IEnumerable
        Yield Nothing
        M(<%= leftParen %>item<%= rightParen %>.Position)
        Yield Nothing
    End Function

    Private Shared Iterator Function Call3(item As Item) As IEnumerable
        Yield Nothing
        M(<%= leftParen %>item<%= rightParen %>.Position)
        Yield Nothing
    End Function

    Shared value as Integer

    Shared Sub M(ByRef x As Integer)
        value -= 1
        Item = New Item With {.Name = value.ToString()}
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
Position get for item '3'
Position set for item '3'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Class_Index()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(item As T)
        M(item.Position(GetOffset(item)), 1)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        M(item.Position(GetOffset(item)), 1)
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       43 (0x2b)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1)
  IL_0000:  ldarg.0
  IL_0001:  dup
  IL_0002:  box        "T"
  IL_0007:  ldarga.s   V_0
  IL_0009:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_000e:  dup
  IL_000f:  stloc.0
  IL_0010:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0015:  stloc.1
  IL_0016:  ldloca.s   V_1
  IL_0018:  ldc.i4.1
  IL_0019:  call       "Sub Program.M(ByRef Integer, Integer)"
  IL_001e:  box        "T"
  IL_0023:  ldloc.0
  IL_0024:  ldloc.1
  IL_0025:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_002a:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
      // Code size       89 (0x59)
  .maxstack  3
  .locals init (T V_0,
                Integer V_1,
                Integer V_2,
                T V_3)
  IL_0000:  ldloca.s   V_3
  IL_0002:  initobj    "T"
  IL_0008:  ldloc.3
  IL_0009:  box        "T"
  IL_000e:  brtrue.s   IL_0016
  IL_0010:  ldarg.0
  IL_0011:  stloc.0
  IL_0012:  ldloca.s   V_0
  IL_0014:  br.s       IL_0018
  IL_0016:  ldarga.s   V_0
  IL_0018:  ldarga.s   V_0
  IL_001a:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_001f:  dup
  IL_0020:  stloc.1
  IL_0021:  constrained. "T"
  IL_0027:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_002c:  stloc.2
  IL_002d:  ldloca.s   V_2
  IL_002f:  ldc.i4.1
  IL_0030:  call       "Sub Program.M(ByRef Integer, Integer)"
  IL_0035:  ldloca.s   V_3
  IL_0037:  initobj    "T"
  IL_003d:  ldloc.3
  IL_003e:  box        "T"
  IL_0043:  brtrue.s   IL_0049
  IL_0045:  ldloca.s   V_0
  IL_0047:  br.s       IL_004b
  IL_0049:  ldarga.s   V_0
  IL_004b:  ldloc.1
  IL_004c:  ldloc.2
  IL_004d:  constrained. "T"
  IL_0053:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0058:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Struct_Index()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(item As T)
        M(item.Position(GetOffset(item)), 1)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        M(item.Position(GetOffset(item)), 1)
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-1'
Position get for item '-2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       47 (0x2f)
  .maxstack  3
  .locals init (Integer V_0,
                Integer V_1)
  IL_0000:  ldarga.s   V_0
  IL_0002:  ldarga.s   V_0
  IL_0004:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0009:  dup
  IL_000a:  stloc.0
  IL_000b:  constrained. "T"
  IL_0011:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0016:  stloc.1
  IL_0017:  ldloca.s   V_1
  IL_0019:  ldc.i4.1
  IL_001a:  call       "Sub Program.M(ByRef Integer, Integer)"
  IL_001f:  ldarga.s   V_0
  IL_0021:  ldloc.0
  IL_0022:  ldloc.1
  IL_0023:  constrained. "T"
  IL_0029:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_002e:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Class_Index_Ref()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(ByRef item As T)
        M(item.Position(GetOffset(item)), 1)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        M(item.Position(GetOffset(item)), 1)
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       47 (0x2f)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1)
  IL_0000:  ldarg.0
  IL_0001:  ldobj      "T"
  IL_0006:  dup
  IL_0007:  box        "T"
  IL_000c:  ldarg.0
  IL_000d:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0012:  dup
  IL_0013:  stloc.0
  IL_0014:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0019:  stloc.1
  IL_001a:  ldloca.s   V_1
  IL_001c:  ldc.i4.1
  IL_001d:  call       "Sub Program.M(ByRef Integer, Integer)"
  IL_0022:  box        "T"
  IL_0027:  ldloc.0
  IL_0028:  ldloc.1
  IL_0029:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_002e:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
      // Code size       91 (0x5b)
  .maxstack  3
  .locals init (T V_0,
                Integer V_1,
                Integer V_2,
                T V_3)
  IL_0000:  ldloca.s   V_3
  IL_0002:  initobj    "T"
  IL_0008:  ldloc.3
  IL_0009:  box        "T"
  IL_000e:  brtrue.s   IL_001b
  IL_0010:  ldarg.0
  IL_0011:  ldobj      "T"
  IL_0016:  stloc.0
  IL_0017:  ldloca.s   V_0
  IL_0019:  br.s       IL_001c
  IL_001b:  ldarg.0
  IL_001c:  ldarg.0
  IL_001d:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0022:  dup
  IL_0023:  stloc.1
  IL_0024:  constrained. "T"
  IL_002a:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_002f:  stloc.2
  IL_0030:  ldloca.s   V_2
  IL_0032:  ldc.i4.1
  IL_0033:  call       "Sub Program.M(ByRef Integer, Integer)"
  IL_0038:  ldloca.s   V_3
  IL_003a:  initobj    "T"
  IL_0040:  ldloc.3
  IL_0041:  box        "T"
  IL_0046:  brtrue.s   IL_004c
  IL_0048:  ldloca.s   V_0
  IL_004a:  br.s       IL_004d
  IL_004c:  ldarg.0
  IL_004d:  ldloc.1
  IL_004e:  ldloc.2
  IL_004f:  constrained. "T"
  IL_0055:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_005a:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Struct_Index_Ref()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(ByRef item As T)
        M(item.Position(GetOffset(item)), 1)
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        M(item.Position(GetOffset(item)), 1)
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-1'
Position get for item '-2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       44 (0x2c)
  .maxstack  3
  .locals init (Integer V_0,
                Integer V_1)
  IL_0000:  ldarg.0
  IL_0001:  ldarg.0
  IL_0002:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0007:  dup
  IL_0008:  stloc.0
  IL_0009:  constrained. "T"
  IL_000f:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0014:  stloc.1
  IL_0015:  ldloca.s   V_1
  IL_0017:  ldc.i4.1
  IL_0018:  call       "Sub Program.M(ByRef Integer, Integer)"
  IL_001d:  ldarg.0
  IL_001e:  ldloc.0
  IL_001f:  ldloc.1
  IL_0020:  constrained. "T"
  IL_0026:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_002b:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Class_Index_Async_01()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        M(item.Position(await GetOffsetAsync(GetOffset(item))), 1)
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        M(item.Position(await GetOffsetAsync(GetOffset(item))), 1)
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      260 (0x104)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1,
                Integer V_2,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_3,
                SM$T V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0063
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldarg.0
    IL_000d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0012:  dup
    IL_0013:  stloc.s    V_4
    IL_0015:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_001a:  ldloc.s    V_4
    IL_001c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0021:  ldarg.0
    IL_0022:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0027:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_002c:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0031:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0036:  stloc.3
    IL_0037:  ldloca.s   V_3
    IL_0039:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_003e:  brtrue.s   IL_007f
    IL_0040:  ldarg.0
    IL_0041:  ldc.i4.0
    IL_0042:  dup
    IL_0043:  stloc.0
    IL_0044:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0049:  ldarg.0
    IL_004a:  ldloc.3
    IL_004b:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0050:  ldarg.0
    IL_0051:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0056:  ldloca.s   V_3
    IL_0058:  ldarg.0
    IL_0059:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_005e:  leave      IL_0103
    IL_0063:  ldarg.0
    IL_0064:  ldc.i4.m1
    IL_0065:  dup
    IL_0066:  stloc.0
    IL_0067:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_006c:  ldarg.0
    IL_006d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0072:  stloc.3
    IL_0073:  ldarg.0
    IL_0074:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0079:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007f:  ldarg.0
    IL_0080:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0085:  box        "SM$T"
    IL_008a:  ldloca.s   V_3
    IL_008c:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0091:  ldloca.s   V_3
    IL_0093:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0099:  dup
    IL_009a:  stloc.1
    IL_009b:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_00a0:  stloc.2
    IL_00a1:  ldloca.s   V_2
    IL_00a3:  ldc.i4.1
    IL_00a4:  call       "Sub Program.M(ByRef Integer, Integer)"
    IL_00a9:  ldarg.0
    IL_00aa:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_00af:  ldloc.1
    IL_00b0:  ldloc.2
    IL_00b1:  constrained. "SM$T"
    IL_00b7:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00bc:  ldarg.0
    IL_00bd:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_00c2:  initobj    "SM$T"
    IL_00c8:  leave.s    IL_00ee
  }
  catch System.Exception
  {
    IL_00ca:  dup
    IL_00cb:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00d0:  stloc.s    V_5
    IL_00d2:  ldarg.0
    IL_00d3:  ldc.i4.s   -2
    IL_00d5:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00da:  ldarg.0
    IL_00db:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00e0:  ldloc.s    V_5
    IL_00e2:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00e7:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00ec:  leave.s    IL_0103
  }
  IL_00ee:  ldarg.0
  IL_00ef:  ldc.i4.s   -2
  IL_00f1:  dup
  IL_00f2:  stloc.0
  IL_00f3:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00f8:  ldarg.0
  IL_00f9:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00fe:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0103:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      305 (0x131)
  .maxstack  3
  .locals init (Integer V_0,
                Integer V_1,
                Integer V_2,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_3,
                SM$T V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0069
    IL_000a:  ldloca.s   V_4
    IL_000c:  initobj    "SM$T"
    IL_0012:  ldloc.s    V_4
    IL_0014:  box        "SM$T"
    IL_0019:  brtrue.s   IL_0027
    IL_001b:  ldarg.0
    IL_001c:  ldarg.0
    IL_001d:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0022:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_0027:  ldarg.0
    IL_0028:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_002d:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0032:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0037:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_003c:  stloc.3
    IL_003d:  ldloca.s   V_3
    IL_003f:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0044:  brtrue.s   IL_0085
    IL_0046:  ldarg.0
    IL_0047:  ldc.i4.0
    IL_0048:  dup
    IL_0049:  stloc.0
    IL_004a:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_004f:  ldarg.0
    IL_0050:  ldloc.3
    IL_0051:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0056:  ldarg.0
    IL_0057:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_005c:  ldloca.s   V_3
    IL_005e:  ldarg.0
    IL_005f:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_0064:  leave      IL_0130
    IL_0069:  ldarg.0
    IL_006a:  ldc.i4.m1
    IL_006b:  dup
    IL_006c:  stloc.0
    IL_006d:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0072:  ldarg.0
    IL_0073:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0078:  stloc.3
    IL_0079:  ldarg.0
    IL_007a:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007f:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0085:  ldloca.s   V_4
    IL_0087:  initobj    "SM$T"
    IL_008d:  ldloc.s    V_4
    IL_008f:  box        "SM$T"
    IL_0094:  brtrue.s   IL_009e
    IL_0096:  ldarg.0
    IL_0097:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_009c:  br.s       IL_00a4
    IL_009e:  ldarg.0
    IL_009f:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00a4:  ldloca.s   V_3
    IL_00a6:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00ab:  ldloca.s   V_3
    IL_00ad:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b3:  dup
    IL_00b4:  stloc.1
    IL_00b5:  constrained. "SM$T"
    IL_00bb:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_00c0:  stloc.2
    IL_00c1:  ldloca.s   V_2
    IL_00c3:  ldc.i4.1
    IL_00c4:  call       "Sub Program.M(ByRef Integer, Integer)"
    IL_00c9:  ldloca.s   V_4
    IL_00cb:  initobj    "SM$T"
    IL_00d1:  ldloc.s    V_4
    IL_00d3:  box        "SM$T"
    IL_00d8:  brtrue.s   IL_00e2
    IL_00da:  ldarg.0
    IL_00db:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_00e0:  br.s       IL_00e8
    IL_00e2:  ldarg.0
    IL_00e3:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00e8:  ldloc.1
    IL_00e9:  ldloc.2
    IL_00ea:  constrained. "SM$T"
    IL_00f0:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00f5:  leave.s    IL_011b
  }
  catch System.Exception
  {
    IL_00f7:  dup
    IL_00f8:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00fd:  stloc.s    V_5
    IL_00ff:  ldarg.0
    IL_0100:  ldc.i4.s   -2
    IL_0102:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0107:  ldarg.0
    IL_0108:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_010d:  ldloc.s    V_5
    IL_010f:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0114:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0119:  leave.s    IL_0130
  }
  IL_011b:  ldarg.0
  IL_011c:  ldc.i4.s   -2
  IL_011e:  dup
  IL_011f:  stloc.0
  IL_0120:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0125:  ldarg.0
  IL_0126:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_012b:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0130:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Struct_Index_Async_01()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        M(item.Position(await GetOffsetAsync(GetOffset(item))), 1)
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        M(item.Position(await GetOffsetAsync(GetOffset(item))), 1)
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-1'
Position get for item '-2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      226 (0xe2)
  .maxstack  3
  .locals init (Integer V_0,
                Integer V_1,
                Integer V_2,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004c
    IL_000a:  ldarg.0
    IL_000b:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0010:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0015:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_001a:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_001f:  stloc.3
    IL_0020:  ldloca.s   V_3
    IL_0022:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0027:  brtrue.s   IL_0068
    IL_0029:  ldarg.0
    IL_002a:  ldc.i4.0
    IL_002b:  dup
    IL_002c:  stloc.0
    IL_002d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0032:  ldarg.0
    IL_0033:  ldloc.3
    IL_0034:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0039:  ldarg.0
    IL_003a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003f:  ldloca.s   V_3
    IL_0041:  ldarg.0
    IL_0042:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0047:  leave      IL_00e1
    IL_004c:  ldarg.0
    IL_004d:  ldc.i4.m1
    IL_004e:  dup
    IL_004f:  stloc.0
    IL_0050:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0055:  ldarg.0
    IL_0056:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_005b:  stloc.3
    IL_005c:  ldarg.0
    IL_005d:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0062:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0068:  ldarg.0
    IL_0069:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_006e:  ldloca.s   V_3
    IL_0070:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0075:  ldloca.s   V_3
    IL_0077:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007d:  dup
    IL_007e:  stloc.1
    IL_007f:  constrained. "SM$T"
    IL_0085:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_008a:  stloc.2
    IL_008b:  ldloca.s   V_2
    IL_008d:  ldc.i4.1
    IL_008e:  call       "Sub Program.M(ByRef Integer, Integer)"
    IL_0093:  ldarg.0
    IL_0094:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0099:  ldloc.1
    IL_009a:  ldloc.2
    IL_009b:  constrained. "SM$T"
    IL_00a1:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00a6:  leave.s    IL_00cc
  }
  catch System.Exception
  {
    IL_00a8:  dup
    IL_00a9:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00ae:  stloc.s    V_4
    IL_00b0:  ldarg.0
    IL_00b1:  ldc.i4.s   -2
    IL_00b3:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00b8:  ldarg.0
    IL_00b9:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00be:  ldloc.s    V_4
    IL_00c0:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00c5:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00ca:  leave.s    IL_00e1
  }
  IL_00cc:  ldarg.0
  IL_00cd:  ldc.i4.s   -2
  IL_00cf:  dup
  IL_00d0:  stloc.0
  IL_00d1:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00d6:  ldarg.0
  IL_00d7:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00dc:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_00e1:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Class_Index_Async_02()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x as Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x as Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        await Task.Yield()
        M(item.Position(await GetOffsetAsync(GetOffset(item))), 1)
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        await Task.Yield()
        M(item.Position(await GetOffsetAsync(GetOffset(item))), 1)
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      373 (0x175)
  .maxstack  4
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter V_1,
                System.Runtime.CompilerServices.YieldAwaitable V_2,
                Integer V_3,
                Integer V_4,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_5,
                SM$T V_6,
                System.Exception V_7)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004b
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_00d1
    IL_0011:  call       "Function System.Threading.Tasks.Task.Yield() As System.Runtime.CompilerServices.YieldAwaitable"
    IL_0016:  stloc.2
    IL_0017:  ldloca.s   V_2
    IL_0019:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.GetAwaiter() As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_001e:  stloc.1
    IL_001f:  ldloca.s   V_1
    IL_0021:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.get_IsCompleted() As Boolean"
    IL_0026:  brtrue.s   IL_0067
    IL_0028:  ldarg.0
    IL_0029:  ldc.i4.0
    IL_002a:  dup
    IL_002b:  stloc.0
    IL_002c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0031:  ldarg.0
    IL_0032:  ldloc.1
    IL_0033:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0038:  ldarg.0
    IL_0039:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003e:  ldloca.s   V_1
    IL_0040:  ldarg.0
    IL_0041:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0046:  leave      IL_0174
    IL_004b:  ldarg.0
    IL_004c:  ldc.i4.m1
    IL_004d:  dup
    IL_004e:  stloc.0
    IL_004f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0054:  ldarg.0
    IL_0055:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_005a:  stloc.1
    IL_005b:  ldarg.0
    IL_005c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0061:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0067:  ldloca.s   V_1
    IL_0069:  call       "Sub System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.GetResult()"
    IL_006e:  ldloca.s   V_1
    IL_0070:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0076:  ldarg.0
    IL_0077:  ldarg.0
    IL_0078:  ldarg.0
    IL_0079:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_007e:  dup
    IL_007f:  stloc.s    V_6
    IL_0081:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0086:  ldloc.s    V_6
    IL_0088:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_008d:  ldarg.0
    IL_008e:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0093:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0098:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_009d:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a2:  stloc.s    V_5
    IL_00a4:  ldloca.s   V_5
    IL_00a6:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_00ab:  brtrue.s   IL_00ee
    IL_00ad:  ldarg.0
    IL_00ae:  ldc.i4.1
    IL_00af:  dup
    IL_00b0:  stloc.0
    IL_00b1:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00b6:  ldarg.0
    IL_00b7:  ldloc.s    V_5
    IL_00b9:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00be:  ldarg.0
    IL_00bf:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00c4:  ldloca.s   V_5
    IL_00c6:  ldarg.0
    IL_00c7:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_00cc:  leave      IL_0174
    IL_00d1:  ldarg.0
    IL_00d2:  ldc.i4.m1
    IL_00d3:  dup
    IL_00d4:  stloc.0
    IL_00d5:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00da:  ldarg.0
    IL_00db:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00e0:  stloc.s    V_5
    IL_00e2:  ldarg.0
    IL_00e3:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00e8:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00ee:  ldarg.0
    IL_00ef:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_00f4:  box        "SM$T"
    IL_00f9:  ldloca.s   V_5
    IL_00fb:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0100:  ldloca.s   V_5
    IL_0102:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0108:  dup
    IL_0109:  stloc.3
    IL_010a:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_010f:  stloc.s    V_4
    IL_0111:  ldloca.s   V_4
    IL_0113:  ldc.i4.1
    IL_0114:  call       "Sub Program.M(ByRef Integer, Integer)"
    IL_0119:  ldarg.0
    IL_011a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_011f:  ldloc.3
    IL_0120:  ldloc.s    V_4
    IL_0122:  constrained. "SM$T"
    IL_0128:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_012d:  ldarg.0
    IL_012e:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0133:  initobj    "SM$T"
    IL_0139:  leave.s    IL_015f
  }
  catch System.Exception
  {
    IL_013b:  dup
    IL_013c:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0141:  stloc.s    V_7
    IL_0143:  ldarg.0
    IL_0144:  ldc.i4.s   -2
    IL_0146:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_014b:  ldarg.0
    IL_014c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0151:  ldloc.s    V_7
    IL_0153:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0158:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_015d:  leave.s    IL_0174
  }
  IL_015f:  ldarg.0
  IL_0160:  ldc.i4.s   -2
  IL_0162:  dup
  IL_0163:  stloc.0
  IL_0164:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0169:  ldarg.0
  IL_016a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_016f:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0174:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      418 (0x1a2)
  .maxstack  3
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter V_1,
                System.Runtime.CompilerServices.YieldAwaitable V_2,
                Integer V_3,
                Integer V_4,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_5,
                SM$T V_6,
                System.Exception V_7)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004b
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_00d7
    IL_0011:  call       "Function System.Threading.Tasks.Task.Yield() As System.Runtime.CompilerServices.YieldAwaitable"
    IL_0016:  stloc.2
    IL_0017:  ldloca.s   V_2
    IL_0019:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.GetAwaiter() As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_001e:  stloc.1
    IL_001f:  ldloca.s   V_1
    IL_0021:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.get_IsCompleted() As Boolean"
    IL_0026:  brtrue.s   IL_0067
    IL_0028:  ldarg.0
    IL_0029:  ldc.i4.0
    IL_002a:  dup
    IL_002b:  stloc.0
    IL_002c:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0031:  ldarg.0
    IL_0032:  ldloc.1
    IL_0033:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0038:  ldarg.0
    IL_0039:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003e:  ldloca.s   V_1
    IL_0040:  ldarg.0
    IL_0041:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_0046:  leave      IL_01a1
    IL_004b:  ldarg.0
    IL_004c:  ldc.i4.m1
    IL_004d:  dup
    IL_004e:  stloc.0
    IL_004f:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0054:  ldarg.0
    IL_0055:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_005a:  stloc.1
    IL_005b:  ldarg.0
    IL_005c:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0061:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0067:  ldloca.s   V_1
    IL_0069:  call       "Sub System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.GetResult()"
    IL_006e:  ldloca.s   V_1
    IL_0070:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0076:  ldloca.s   V_6
    IL_0078:  initobj    "SM$T"
    IL_007e:  ldloc.s    V_6
    IL_0080:  box        "SM$T"
    IL_0085:  brtrue.s   IL_0093
    IL_0087:  ldarg.0
    IL_0088:  ldarg.0
    IL_0089:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_008e:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_0093:  ldarg.0
    IL_0094:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0099:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_009e:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_00a3:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a8:  stloc.s    V_5
    IL_00aa:  ldloca.s   V_5
    IL_00ac:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_00b1:  brtrue.s   IL_00f4
    IL_00b3:  ldarg.0
    IL_00b4:  ldc.i4.1
    IL_00b5:  dup
    IL_00b6:  stloc.0
    IL_00b7:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00bc:  ldarg.0
    IL_00bd:  ldloc.s    V_5
    IL_00bf:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00c4:  ldarg.0
    IL_00c5:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00ca:  ldloca.s   V_5
    IL_00cc:  ldarg.0
    IL_00cd:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_00d2:  leave      IL_01a1
    IL_00d7:  ldarg.0
    IL_00d8:  ldc.i4.m1
    IL_00d9:  dup
    IL_00da:  stloc.0
    IL_00db:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00e0:  ldarg.0
    IL_00e1:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00e6:  stloc.s    V_5
    IL_00e8:  ldarg.0
    IL_00e9:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00ee:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00f4:  ldloca.s   V_6
    IL_00f6:  initobj    "SM$T"
    IL_00fc:  ldloc.s    V_6
    IL_00fe:  box        "SM$T"
    IL_0103:  brtrue.s   IL_010d
    IL_0105:  ldarg.0
    IL_0106:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_010b:  br.s       IL_0113
    IL_010d:  ldarg.0
    IL_010e:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0113:  ldloca.s   V_5
    IL_0115:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_011a:  ldloca.s   V_5
    IL_011c:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0122:  dup
    IL_0123:  stloc.3
    IL_0124:  constrained. "SM$T"
    IL_012a:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_012f:  stloc.s    V_4
    IL_0131:  ldloca.s   V_4
    IL_0133:  ldc.i4.1
    IL_0134:  call       "Sub Program.M(ByRef Integer, Integer)"
    IL_0139:  ldloca.s   V_6
    IL_013b:  initobj    "SM$T"
    IL_0141:  ldloc.s    V_6
    IL_0143:  box        "SM$T"
    IL_0148:  brtrue.s   IL_0152
    IL_014a:  ldarg.0
    IL_014b:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_0150:  br.s       IL_0158
    IL_0152:  ldarg.0
    IL_0153:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0158:  ldloc.3
    IL_0159:  ldloc.s    V_4
    IL_015b:  constrained. "SM$T"
    IL_0161:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_0166:  leave.s    IL_018c
  }
  catch System.Exception
  {
    IL_0168:  dup
    IL_0169:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_016e:  stloc.s    V_7
    IL_0170:  ldarg.0
    IL_0171:  ldc.i4.s   -2
    IL_0173:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0178:  ldarg.0
    IL_0179:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_017e:  ldloc.s    V_7
    IL_0180:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0185:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_018a:  leave.s    IL_01a1
  }
  IL_018c:  ldarg.0
  IL_018d:  ldc.i4.s   -2
  IL_018f:  dup
  IL_0190:  stloc.0
  IL_0191:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0196:  ldarg.0
  IL_0197:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_019c:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_01a1:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Struct_Index_Async_02()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x as Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x as Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        await Task.Yield()
        M(item.Position(await GetOffsetAsync(GetOffset(item))), 1)
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        await Task.Yield()
        M(item.Position(await GetOffsetAsync(GetOffset(item))), 1)
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-1'
Position get for item '-2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      339 (0x153)
  .maxstack  3
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter V_1,
                System.Runtime.CompilerServices.YieldAwaitable V_2,
                Integer V_3,
                Integer V_4,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_5,
                System.Exception V_6)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004b
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_00ba
    IL_0011:  call       "Function System.Threading.Tasks.Task.Yield() As System.Runtime.CompilerServices.YieldAwaitable"
    IL_0016:  stloc.2
    IL_0017:  ldloca.s   V_2
    IL_0019:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.GetAwaiter() As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_001e:  stloc.1
    IL_001f:  ldloca.s   V_1
    IL_0021:  call       "Function System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.get_IsCompleted() As Boolean"
    IL_0026:  brtrue.s   IL_0067
    IL_0028:  ldarg.0
    IL_0029:  ldc.i4.0
    IL_002a:  dup
    IL_002b:  stloc.0
    IL_002c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0031:  ldarg.0
    IL_0032:  ldloc.1
    IL_0033:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0038:  ldarg.0
    IL_0039:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003e:  ldloca.s   V_1
    IL_0040:  ldarg.0
    IL_0041:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter, ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0046:  leave      IL_0152
    IL_004b:  ldarg.0
    IL_004c:  ldc.i4.m1
    IL_004d:  dup
    IL_004e:  stloc.0
    IL_004f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0054:  ldarg.0
    IL_0055:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_005a:  stloc.1
    IL_005b:  ldarg.0
    IL_005c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0061:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0067:  ldloca.s   V_1
    IL_0069:  call       "Sub System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter.GetResult()"
    IL_006e:  ldloca.s   V_1
    IL_0070:  initobj    "System.Runtime.CompilerServices.YieldAwaitable.YieldAwaiter"
    IL_0076:  ldarg.0
    IL_0077:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_007c:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0081:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0086:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008b:  stloc.s    V_5
    IL_008d:  ldloca.s   V_5
    IL_008f:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0094:  brtrue.s   IL_00d7
    IL_0096:  ldarg.0
    IL_0097:  ldc.i4.1
    IL_0098:  dup
    IL_0099:  stloc.0
    IL_009a:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_009f:  ldarg.0
    IL_00a0:  ldloc.s    V_5
    IL_00a2:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a7:  ldarg.0
    IL_00a8:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00ad:  ldloca.s   V_5
    IL_00af:  ldarg.0
    IL_00b0:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_00b5:  leave      IL_0152
    IL_00ba:  ldarg.0
    IL_00bb:  ldc.i4.m1
    IL_00bc:  dup
    IL_00bd:  stloc.0
    IL_00be:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00c3:  ldarg.0
    IL_00c4:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00c9:  stloc.s    V_5
    IL_00cb:  ldarg.0
    IL_00cc:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A1 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d1:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00d7:  ldarg.0
    IL_00d8:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_00dd:  ldloca.s   V_5
    IL_00df:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00e4:  ldloca.s   V_5
    IL_00e6:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00ec:  dup
    IL_00ed:  stloc.3
    IL_00ee:  constrained. "SM$T"
    IL_00f4:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_00f9:  stloc.s    V_4
    IL_00fb:  ldloca.s   V_4
    IL_00fd:  ldc.i4.1
    IL_00fe:  call       "Sub Program.M(ByRef Integer, Integer)"
    IL_0103:  ldarg.0
    IL_0104:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0109:  ldloc.3
    IL_010a:  ldloc.s    V_4
    IL_010c:  constrained. "SM$T"
    IL_0112:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_0117:  leave.s    IL_013d
  }
  catch System.Exception
  {
    IL_0119:  dup
    IL_011a:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_011f:  stloc.s    V_6
    IL_0121:  ldarg.0
    IL_0122:  ldc.i4.s   -2
    IL_0124:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0129:  ldarg.0
    IL_012a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_012f:  ldloc.s    V_6
    IL_0131:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0136:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_013b:  leave.s    IL_0152
  }
  IL_013d:  ldarg.0
  IL_013e:  ldc.i4.s   -2
  IL_0140:  dup
  IL_0141:  stloc.0
  IL_0142:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0147:  ldarg.0
  IL_0148:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_014d:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0152:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Class_Value()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(item As T)
        M(item.Position(1), GetOffset(item))
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        M(item.Position(1), GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       41 (0x29)
  .maxstack  3
  .locals init (Integer V_0)
  IL_0000:  ldarg.0
  IL_0001:  dup
  IL_0002:  box        "T"
  IL_0007:  ldc.i4.1
  IL_0008:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_000d:  stloc.0
  IL_000e:  ldloca.s   V_0
  IL_0010:  ldarga.s   V_0
  IL_0012:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0017:  call       "Sub Program.M(ByRef Integer, Integer)"
  IL_001c:  box        "T"
  IL_0021:  ldc.i4.1
  IL_0022:  ldloc.0
  IL_0023:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0028:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
      // Code size       87 (0x57)
  .maxstack  3
  .locals init (T V_0,
                Integer V_1,
                T V_2)
  IL_0000:  ldloca.s   V_2
  IL_0002:  initobj    "T"
  IL_0008:  ldloc.2
  IL_0009:  box        "T"
  IL_000e:  brtrue.s   IL_0016
  IL_0010:  ldarg.0
  IL_0011:  stloc.0
  IL_0012:  ldloca.s   V_0
  IL_0014:  br.s       IL_0018
  IL_0016:  ldarga.s   V_0
  IL_0018:  ldc.i4.1
  IL_0019:  constrained. "T"
  IL_001f:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0024:  stloc.1
  IL_0025:  ldloca.s   V_1
  IL_0027:  ldarga.s   V_0
  IL_0029:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_002e:  call       "Sub Program.M(ByRef Integer, Integer)"
  IL_0033:  ldloca.s   V_2
  IL_0035:  initobj    "T"
  IL_003b:  ldloc.2
  IL_003c:  box        "T"
  IL_0041:  brtrue.s   IL_0047
  IL_0043:  ldloca.s   V_0
  IL_0045:  br.s       IL_0049
  IL_0047:  ldarga.s   V_0
  IL_0049:  ldc.i4.1
  IL_004a:  ldloc.1
  IL_004b:  constrained. "T"
  IL_0051:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0056:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Struct_Value()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(item As T)
        M(item.Position(1), GetOffset(item))
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        M(item.Position(1), GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '-1'
Position get for item '2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       45 (0x2d)
  .maxstack  3
  .locals init (Integer V_0)
  IL_0000:  ldarga.s   V_0
  IL_0002:  ldc.i4.1
  IL_0003:  constrained. "T"
  IL_0009:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_000e:  stloc.0
  IL_000f:  ldloca.s   V_0
  IL_0011:  ldarga.s   V_0
  IL_0013:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0018:  call       "Sub Program.M(ByRef Integer, Integer)"
  IL_001d:  ldarga.s   V_0
  IL_001f:  ldc.i4.1
  IL_0020:  ldloc.0
  IL_0021:  constrained. "T"
  IL_0027:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_002c:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Class_Value_Ref()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(ByRef item As T)
        M(item.Position(1), GetOffset(item))
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        M(item.Position(1), GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       45 (0x2d)
  .maxstack  3
  .locals init (Integer V_0)
  IL_0000:  ldarg.0
  IL_0001:  ldobj      "T"
  IL_0006:  dup
  IL_0007:  box        "T"
  IL_000c:  ldc.i4.1
  IL_000d:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0012:  stloc.0
  IL_0013:  ldloca.s   V_0
  IL_0015:  ldarg.0
  IL_0016:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_001b:  call       "Sub Program.M(ByRef Integer, Integer)"
  IL_0020:  box        "T"
  IL_0025:  ldc.i4.1
  IL_0026:  ldloc.0
  IL_0027:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_002c:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
      // Code size       89 (0x59)
  .maxstack  3
  .locals init (T V_0,
                Integer V_1,
                T V_2)
  IL_0000:  ldloca.s   V_2
  IL_0002:  initobj    "T"
  IL_0008:  ldloc.2
  IL_0009:  box        "T"
  IL_000e:  brtrue.s   IL_001b
  IL_0010:  ldarg.0
  IL_0011:  ldobj      "T"
  IL_0016:  stloc.0
  IL_0017:  ldloca.s   V_0
  IL_0019:  br.s       IL_001c
  IL_001b:  ldarg.0
  IL_001c:  ldc.i4.1
  IL_001d:  constrained. "T"
  IL_0023:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0028:  stloc.1
  IL_0029:  ldloca.s   V_1
  IL_002b:  ldarg.0
  IL_002c:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0031:  call       "Sub Program.M(ByRef Integer, Integer)"
  IL_0036:  ldloca.s   V_2
  IL_0038:  initobj    "T"
  IL_003e:  ldloc.2
  IL_003f:  box        "T"
  IL_0044:  brtrue.s   IL_004a
  IL_0046:  ldloca.s   V_0
  IL_0048:  br.s       IL_004b
  IL_004a:  ldarg.0
  IL_004b:  ldc.i4.1
  IL_004c:  ldloc.1
  IL_004d:  constrained. "T"
  IL_0053:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0058:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Struct_Value_Ref()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(ByRef item As T)
        M(item.Position(1), GetOffset(item))
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        M(item.Position(1), GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '-1'
Position get for item '2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       42 (0x2a)
  .maxstack  3
  .locals init (Integer V_0)
  IL_0000:  ldarg.0
  IL_0001:  ldc.i4.1
  IL_0002:  constrained. "T"
  IL_0008:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_000d:  stloc.0
  IL_000e:  ldloca.s   V_0
  IL_0010:  ldarg.0
  IL_0011:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0016:  call       "Sub Program.M(ByRef Integer, Integer)"
  IL_001b:  ldarg.0
  IL_001c:  ldc.i4.1
  IL_001d:  ldloc.0
  IL_001e:  constrained. "T"
  IL_0024:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0029:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Class_Value_Async_01()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        M(item.Position(1), await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        M(item.Position(1), await GetOffsetAsync(GetOffset(item)))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      243 (0xf3)
  .maxstack  4
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                SM$T V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_006c
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldarg.0
    IL_000d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0012:  dup
    IL_0013:  stloc.2
    IL_0014:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0019:  ldloc.2
    IL_001a:  box        "SM$T"
    IL_001f:  ldc.i4.1
    IL_0020:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_0025:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S1 As Integer"
    IL_002a:  ldarg.0
    IL_002b:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0030:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0035:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_003a:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_003f:  stloc.1
    IL_0040:  ldloca.s   V_1
    IL_0042:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0047:  brtrue.s   IL_0088
    IL_0049:  ldarg.0
    IL_004a:  ldc.i4.0
    IL_004b:  dup
    IL_004c:  stloc.0
    IL_004d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0052:  ldarg.0
    IL_0053:  ldloc.1
    IL_0054:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0059:  ldarg.0
    IL_005a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_005f:  ldloca.s   V_1
    IL_0061:  ldarg.0
    IL_0062:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0067:  leave      IL_00f2
    IL_006c:  ldarg.0
    IL_006d:  ldc.i4.m1
    IL_006e:  dup
    IL_006f:  stloc.0
    IL_0070:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0075:  ldarg.0
    IL_0076:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007b:  stloc.1
    IL_007c:  ldarg.0
    IL_007d:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0082:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0088:  ldarg.0
    IL_0089:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S1 As Integer"
    IL_008e:  ldloca.s   V_1
    IL_0090:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0095:  ldloca.s   V_1
    IL_0097:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009d:  call       "Sub Program.M(ByRef Integer, Integer)"
    IL_00a2:  ldarg.0
    IL_00a3:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_00a8:  box        "SM$T"
    IL_00ad:  ldc.i4.1
    IL_00ae:  ldarg.0
    IL_00af:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S1 As Integer"
    IL_00b4:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00b9:  leave.s    IL_00dd
  }
  catch System.Exception
  {
    IL_00bb:  dup
    IL_00bc:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00c1:  stloc.3
    IL_00c2:  ldarg.0
    IL_00c3:  ldc.i4.s   -2
    IL_00c5:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00ca:  ldarg.0
    IL_00cb:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00d0:  ldloc.3
    IL_00d1:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00d6:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00db:  leave.s    IL_00f2
  }
  IL_00dd:  ldarg.0
  IL_00de:  ldc.i4.s   -2
  IL_00e0:  dup
  IL_00e1:  stloc.0
  IL_00e2:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00e7:  ldarg.0
  IL_00e8:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00ed:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_00f2:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      296 (0x128)
  .maxstack  3
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                SM$T V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0088
    IL_000a:  ldarg.0
    IL_000b:  ldloca.s   V_2
    IL_000d:  initobj    "SM$T"
    IL_0013:  ldloc.2
    IL_0014:  box        "SM$T"
    IL_0019:  brtrue.s   IL_002f
    IL_001b:  ldarg.0
    IL_001c:  ldarg.0
    IL_001d:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0022:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_0027:  ldarg.0
    IL_0028:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_002d:  br.s       IL_0035
    IL_002f:  ldarg.0
    IL_0030:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0035:  ldc.i4.1
    IL_0036:  constrained. "SM$T"
    IL_003c:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_0041:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S1 As Integer"
    IL_0046:  ldarg.0
    IL_0047:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_004c:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0051:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0056:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_005b:  stloc.1
    IL_005c:  ldloca.s   V_1
    IL_005e:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0063:  brtrue.s   IL_00a4
    IL_0065:  ldarg.0
    IL_0066:  ldc.i4.0
    IL_0067:  dup
    IL_0068:  stloc.0
    IL_0069:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_006e:  ldarg.0
    IL_006f:  ldloc.1
    IL_0070:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0075:  ldarg.0
    IL_0076:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_007b:  ldloca.s   V_1
    IL_007d:  ldarg.0
    IL_007e:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_0083:  leave      IL_0127
    IL_0088:  ldarg.0
    IL_0089:  ldc.i4.m1
    IL_008a:  dup
    IL_008b:  stloc.0
    IL_008c:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0091:  ldarg.0
    IL_0092:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0097:  stloc.1
    IL_0098:  ldarg.0
    IL_0099:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009e:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a4:  ldarg.0
    IL_00a5:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S1 As Integer"
    IL_00aa:  ldloca.s   V_1
    IL_00ac:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00b1:  ldloca.s   V_1
    IL_00b3:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b9:  call       "Sub Program.M(ByRef Integer, Integer)"
    IL_00be:  ldloca.s   V_2
    IL_00c0:  initobj    "SM$T"
    IL_00c6:  ldloc.2
    IL_00c7:  box        "SM$T"
    IL_00cc:  brtrue.s   IL_00d6
    IL_00ce:  ldarg.0
    IL_00cf:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_00d4:  br.s       IL_00dc
    IL_00d6:  ldarg.0
    IL_00d7:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00dc:  ldc.i4.1
    IL_00dd:  ldarg.0
    IL_00de:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S1 As Integer"
    IL_00e3:  constrained. "SM$T"
    IL_00e9:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00ee:  leave.s    IL_0112
  }
  catch System.Exception
  {
    IL_00f0:  dup
    IL_00f1:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00f6:  stloc.3
    IL_00f7:  ldarg.0
    IL_00f8:  ldc.i4.s   -2
    IL_00fa:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00ff:  ldarg.0
    IL_0100:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0105:  ldloc.3
    IL_0106:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_010b:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0110:  leave.s    IL_0127
  }
  IL_0112:  ldarg.0
  IL_0113:  ldc.i4.s   -2
  IL_0115:  dup
  IL_0116:  stloc.0
  IL_0117:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_011c:  ldarg.0
  IL_011d:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0122:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0127:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Struct_Value_Async_01()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        M(item.Position(1), await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        M(item.Position(1), await GetOffsetAsync(GetOffset(item)))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '-1'
Position get for item '2'
Position set for item '-2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      236 (0xec)
  .maxstack  3
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                System.Exception V_2)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0064
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0011:  ldc.i4.1
    IL_0012:  constrained. "SM$T"
    IL_0018:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_001d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As Integer"
    IL_0022:  ldarg.0
    IL_0023:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0028:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_002d:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0032:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0037:  stloc.1
    IL_0038:  ldloca.s   V_1
    IL_003a:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_003f:  brtrue.s   IL_0080
    IL_0041:  ldarg.0
    IL_0042:  ldc.i4.0
    IL_0043:  dup
    IL_0044:  stloc.0
    IL_0045:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_004a:  ldarg.0
    IL_004b:  ldloc.1
    IL_004c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0051:  ldarg.0
    IL_0052:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0057:  ldloca.s   V_1
    IL_0059:  ldarg.0
    IL_005a:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_005f:  leave      IL_00eb
    IL_0064:  ldarg.0
    IL_0065:  ldc.i4.m1
    IL_0066:  dup
    IL_0067:  stloc.0
    IL_0068:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_006d:  ldarg.0
    IL_006e:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0073:  stloc.1
    IL_0074:  ldarg.0
    IL_0075:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007a:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0080:  ldarg.0
    IL_0081:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As Integer"
    IL_0086:  ldloca.s   V_1
    IL_0088:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_008d:  ldloca.s   V_1
    IL_008f:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0095:  call       "Sub Program.M(ByRef Integer, Integer)"
    IL_009a:  ldarg.0
    IL_009b:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_00a0:  ldc.i4.1
    IL_00a1:  ldarg.0
    IL_00a2:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As Integer"
    IL_00a7:  constrained. "SM$T"
    IL_00ad:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00b2:  leave.s    IL_00d6
  }
  catch System.Exception
  {
    IL_00b4:  dup
    IL_00b5:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00ba:  stloc.2
    IL_00bb:  ldarg.0
    IL_00bc:  ldc.i4.s   -2
    IL_00be:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00c3:  ldarg.0
    IL_00c4:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00c9:  ldloc.2
    IL_00ca:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00cf:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00d4:  leave.s    IL_00eb
  }
  IL_00d6:  ldarg.0
  IL_00d7:  ldc.i4.s   -2
  IL_00d9:  dup
  IL_00da:  stloc.0
  IL_00db:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00e0:  ldarg.0
  IL_00e1:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00e6:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_00eb:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Class_IndexAndValue()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(item As T)
        M(item.Position(GetOffset(item)), GetOffset(item))
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        M(item.Position(GetOffset(item)), GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       49 (0x31)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1)
  IL_0000:  ldarg.0
  IL_0001:  dup
  IL_0002:  box        "T"
  IL_0007:  ldarga.s   V_0
  IL_0009:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_000e:  dup
  IL_000f:  stloc.0
  IL_0010:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0015:  stloc.1
  IL_0016:  ldloca.s   V_1
  IL_0018:  ldarga.s   V_0
  IL_001a:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_001f:  call       "Sub Program.M(ByRef Integer, Integer)"
  IL_0024:  box        "T"
  IL_0029:  ldloc.0
  IL_002a:  ldloc.1
  IL_002b:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0030:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
      // Code size       95 (0x5f)
  .maxstack  3
  .locals init (T V_0,
                Integer V_1,
                Integer V_2,
                T V_3)
  IL_0000:  ldloca.s   V_3
  IL_0002:  initobj    "T"
  IL_0008:  ldloc.3
  IL_0009:  box        "T"
  IL_000e:  brtrue.s   IL_0016
  IL_0010:  ldarg.0
  IL_0011:  stloc.0
  IL_0012:  ldloca.s   V_0
  IL_0014:  br.s       IL_0018
  IL_0016:  ldarga.s   V_0
  IL_0018:  ldarga.s   V_0
  IL_001a:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_001f:  dup
  IL_0020:  stloc.1
  IL_0021:  constrained. "T"
  IL_0027:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_002c:  stloc.2
  IL_002d:  ldloca.s   V_2
  IL_002f:  ldarga.s   V_0
  IL_0031:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0036:  call       "Sub Program.M(ByRef Integer, Integer)"
  IL_003b:  ldloca.s   V_3
  IL_003d:  initobj    "T"
  IL_0043:  ldloc.3
  IL_0044:  box        "T"
  IL_0049:  brtrue.s   IL_004f
  IL_004b:  ldloca.s   V_0
  IL_004d:  br.s       IL_0051
  IL_004f:  ldarga.s   V_0
  IL_0051:  ldloc.1
  IL_0052:  ldloc.2
  IL_0053:  constrained. "T"
  IL_0059:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_005e:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Struct_IndexAndValue()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(item As T)
        M(item.Position(GetOffset(item)), GetOffset(item))
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(item As T)
        M(item.Position(GetOffset(item)), GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-2'
Position get for item '-3'
Position set for item '-4'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       53 (0x35)
  .maxstack  3
  .locals init (Integer V_0,
                Integer V_1)
  IL_0000:  ldarga.s   V_0
  IL_0002:  ldarga.s   V_0
  IL_0004:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0009:  dup
  IL_000a:  stloc.0
  IL_000b:  constrained. "T"
  IL_0011:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0016:  stloc.1
  IL_0017:  ldloca.s   V_1
  IL_0019:  ldarga.s   V_0
  IL_001b:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0020:  call       "Sub Program.M(ByRef Integer, Integer)"
  IL_0025:  ldarga.s   V_0
  IL_0027:  ldloc.0
  IL_0028:  ldloc.1
  IL_0029:  constrained. "T"
  IL_002f:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0034:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Class_IndexAndValue_Ref()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Class, IMoveable})(ByRef item As T)
        M(item.Position(GetOffset(item)), GetOffset(item))
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        M(item.Position(GetOffset(item)), GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
      // Code size       52 (0x34)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1)
  IL_0000:  ldarg.0
  IL_0001:  ldobj      "T"
  IL_0006:  dup
  IL_0007:  box        "T"
  IL_000c:  ldarg.0
  IL_000d:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0012:  dup
  IL_0013:  stloc.0
  IL_0014:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0019:  stloc.1
  IL_001a:  ldloca.s   V_1
  IL_001c:  ldarg.0
  IL_001d:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0022:  call       "Sub Program.M(ByRef Integer, Integer)"
  IL_0027:  box        "T"
  IL_002c:  ldloc.0
  IL_002d:  ldloc.1
  IL_002e:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0033:  ret
}
]]>)

            verifier.VerifyIL("Program.Call2(Of T)",
            <![CDATA[
{
      // Code size       96 (0x60)
  .maxstack  3
  .locals init (T V_0,
                Integer V_1,
                Integer V_2,
                T V_3)
  IL_0000:  ldloca.s   V_3
  IL_0002:  initobj    "T"
  IL_0008:  ldloc.3
  IL_0009:  box        "T"
  IL_000e:  brtrue.s   IL_001b
  IL_0010:  ldarg.0
  IL_0011:  ldobj      "T"
  IL_0016:  stloc.0
  IL_0017:  ldloca.s   V_0
  IL_0019:  br.s       IL_001c
  IL_001b:  ldarg.0
  IL_001c:  ldarg.0
  IL_001d:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0022:  dup
  IL_0023:  stloc.1
  IL_0024:  constrained. "T"
  IL_002a:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_002f:  stloc.2
  IL_0030:  ldloca.s   V_2
  IL_0032:  ldarg.0
  IL_0033:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0038:  call       "Sub Program.M(ByRef Integer, Integer)"
  IL_003d:  ldloca.s   V_3
  IL_003f:  initobj    "T"
  IL_0045:  ldloc.3
  IL_0046:  box        "T"
  IL_004b:  brtrue.s   IL_0051
  IL_004d:  ldloca.s   V_0
  IL_004f:  br.s       IL_0052
  IL_0051:  ldarg.0
  IL_0052:  ldloc.1
  IL_0053:  ldloc.2
  IL_0054:  constrained. "T"
  IL_005a:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_005f:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Struct_IndexAndValue_Ref()
            Dim comp =
<compilation>
    <file>
Imports System

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1)

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2)
    End Sub

    Private Shared Sub Call1(Of T As {Structure, IMoveable})(ByRef item As T)
        M(item.Position(GetOffset(item)), GetOffset(item))
    End Sub

    Private Shared Sub Call2(Of T As {IMoveable})(ByRef item As T)
        M(item.Position(GetOffset(item)), GetOffset(item))
    End Sub

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-2'
Position get for item '-3'
Position set for item '-4'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.Call1(Of T)",
            <![CDATA[
{
  // Code size       49 (0x31)
  .maxstack  3
  .locals init (Integer V_0,
            Integer V_1)
  IL_0000:  ldarg.0
  IL_0001:  ldarg.0
  IL_0002:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_0007:  dup
  IL_0008:  stloc.0
  IL_0009:  constrained. "T"
  IL_000f:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
  IL_0014:  stloc.1
  IL_0015:  ldloca.s   V_1
  IL_0017:  ldarg.0
  IL_0018:  call       "Function Program.GetOffset(Of T)(ByRef T) As Integer"
  IL_001d:  call       "Sub Program.M(ByRef Integer, Integer)"
  IL_0022:  ldarg.0
  IL_0023:  ldloc.0
  IL_0024:  ldloc.1
  IL_0025:  constrained. "T"
  IL_002b:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
  IL_0030:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Class_IndexAndValue_Async_01()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        M(item.Position(GetOffset(item)), await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        M(item.Position(GetOffset(item)), await GetOffsetAsync(GetOffset(item)))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      269 (0x10d)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                SM$T V_2,
                Integer V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_007f
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldarg.0
    IL_000d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0012:  dup
    IL_0013:  stloc.2
    IL_0014:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0019:  ldloc.2
    IL_001a:  box        "SM$T"
    IL_001f:  ldarg.0
    IL_0020:  ldarg.0
    IL_0021:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0026:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_002b:  dup
    IL_002c:  stloc.3
    IL_002d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S1 As Integer"
    IL_0032:  ldloc.3
    IL_0033:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_0038:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S2 As Integer"
    IL_003d:  ldarg.0
    IL_003e:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0043:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0048:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_004d:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0052:  stloc.1
    IL_0053:  ldloca.s   V_1
    IL_0055:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_005a:  brtrue.s   IL_009b
    IL_005c:  ldarg.0
    IL_005d:  ldc.i4.0
    IL_005e:  dup
    IL_005f:  stloc.0
    IL_0060:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0065:  ldarg.0
    IL_0066:  ldloc.1
    IL_0067:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_006c:  ldarg.0
    IL_006d:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0072:  ldloca.s   V_1
    IL_0074:  ldarg.0
    IL_0075:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_007a:  leave      IL_010c
    IL_007f:  ldarg.0
    IL_0080:  ldc.i4.m1
    IL_0081:  dup
    IL_0082:  stloc.0
    IL_0083:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0088:  ldarg.0
    IL_0089:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008e:  stloc.1
    IL_008f:  ldarg.0
    IL_0090:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0095:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_009b:  ldarg.0
    IL_009c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S2 As Integer"
    IL_00a1:  ldloca.s   V_1
    IL_00a3:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00a8:  ldloca.s   V_1
    IL_00aa:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b0:  call       "Sub Program.M(ByRef Integer, Integer)"
    IL_00b5:  ldarg.0
    IL_00b6:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_00bb:  box        "SM$T"
    IL_00c0:  ldarg.0
    IL_00c1:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S1 As Integer"
    IL_00c6:  ldarg.0
    IL_00c7:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S2 As Integer"
    IL_00cc:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00d1:  leave.s    IL_00f7
  }
  catch System.Exception
  {
    IL_00d3:  dup
    IL_00d4:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00d9:  stloc.s    V_4
    IL_00db:  ldarg.0
    IL_00dc:  ldc.i4.s   -2
    IL_00de:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00e3:  ldarg.0
    IL_00e4:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00e9:  ldloc.s    V_4
    IL_00eb:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00f0:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00f5:  leave.s    IL_010c
  }
  IL_00f7:  ldarg.0
  IL_00f8:  ldc.i4.s   -2
  IL_00fa:  dup
  IL_00fb:  stloc.0
  IL_00fc:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0101:  ldarg.0
  IL_0102:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0107:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_010c:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      325 (0x145)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                SM$T V_2,
                Integer V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse    IL_009e
    IL_000d:  ldarg.0
    IL_000e:  ldloca.s   V_2
    IL_0010:  initobj    "SM$T"
    IL_0016:  ldloc.2
    IL_0017:  box        "SM$T"
    IL_001c:  brtrue.s   IL_0032
    IL_001e:  ldarg.0
    IL_001f:  ldarg.0
    IL_0020:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0025:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_002a:  ldarg.0
    IL_002b:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_0030:  br.s       IL_0038
    IL_0032:  ldarg.0
    IL_0033:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0038:  ldarg.0
    IL_0039:  ldarg.0
    IL_003a:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_003f:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0044:  dup
    IL_0045:  stloc.3
    IL_0046:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S1 As Integer"
    IL_004b:  ldloc.3
    IL_004c:  constrained. "SM$T"
    IL_0052:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_0057:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S2 As Integer"
    IL_005c:  ldarg.0
    IL_005d:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0062:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0067:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_006c:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0071:  stloc.1
    IL_0072:  ldloca.s   V_1
    IL_0074:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0079:  brtrue.s   IL_00ba
    IL_007b:  ldarg.0
    IL_007c:  ldc.i4.0
    IL_007d:  dup
    IL_007e:  stloc.0
    IL_007f:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0084:  ldarg.0
    IL_0085:  ldloc.1
    IL_0086:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008b:  ldarg.0
    IL_008c:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0091:  ldloca.s   V_1
    IL_0093:  ldarg.0
    IL_0094:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_0099:  leave      IL_0144
    IL_009e:  ldarg.0
    IL_009f:  ldc.i4.m1
    IL_00a0:  dup
    IL_00a1:  stloc.0
    IL_00a2:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_00a7:  ldarg.0
    IL_00a8:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00ad:  stloc.1
    IL_00ae:  ldarg.0
    IL_00af:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b4:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00ba:  ldarg.0
    IL_00bb:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S2 As Integer"
    IL_00c0:  ldloca.s   V_1
    IL_00c2:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00c7:  ldloca.s   V_1
    IL_00c9:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00cf:  call       "Sub Program.M(ByRef Integer, Integer)"
    IL_00d4:  ldloca.s   V_2
    IL_00d6:  initobj    "SM$T"
    IL_00dc:  ldloc.2
    IL_00dd:  box        "SM$T"
    IL_00e2:  brtrue.s   IL_00ec
    IL_00e4:  ldarg.0
    IL_00e5:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_00ea:  br.s       IL_00f2
    IL_00ec:  ldarg.0
    IL_00ed:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00f2:  ldarg.0
    IL_00f3:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S1 As Integer"
    IL_00f8:  ldarg.0
    IL_00f9:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S2 As Integer"
    IL_00fe:  constrained. "SM$T"
    IL_0104:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_0109:  leave.s    IL_012f
  }
  catch System.Exception
  {
    IL_010b:  dup
    IL_010c:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0111:  stloc.s    V_4
    IL_0113:  ldarg.0
    IL_0114:  ldc.i4.s   -2
    IL_0116:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_011b:  ldarg.0
    IL_011c:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0121:  ldloc.s    V_4
    IL_0123:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0128:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_012d:  leave.s    IL_0144
  }
  IL_012f:  ldarg.0
  IL_0130:  ldc.i4.s   -2
  IL_0132:  dup
  IL_0133:  stloc.0
  IL_0134:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0139:  ldarg.0
  IL_013a:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_013f:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0144:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Struct_IndexAndValue_Async_01()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        M(item.Position(GetOffset(item)), await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        M(item.Position(GetOffset(item)), await GetOffsetAsync(GetOffset(item)))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-2'
Position get for item '-3'
Position set for item '-4'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      260 (0x104)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                Integer V_2,
                System.Exception V_3)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0077
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0011:  ldarg.0
    IL_0012:  ldarg.0
    IL_0013:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0018:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_001d:  dup
    IL_001e:  stloc.2
    IL_001f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As Integer"
    IL_0024:  ldloc.2
    IL_0025:  constrained. "SM$T"
    IL_002b:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_0030:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S1 As Integer"
    IL_0035:  ldarg.0
    IL_0036:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_003b:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0040:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0045:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_004a:  stloc.1
    IL_004b:  ldloca.s   V_1
    IL_004d:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0052:  brtrue.s   IL_0093
    IL_0054:  ldarg.0
    IL_0055:  ldc.i4.0
    IL_0056:  dup
    IL_0057:  stloc.0
    IL_0058:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_005d:  ldarg.0
    IL_005e:  ldloc.1
    IL_005f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0064:  ldarg.0
    IL_0065:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_006a:  ldloca.s   V_1
    IL_006c:  ldarg.0
    IL_006d:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0072:  leave      IL_0103
    IL_0077:  ldarg.0
    IL_0078:  ldc.i4.m1
    IL_0079:  dup
    IL_007a:  stloc.0
    IL_007b:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0080:  ldarg.0
    IL_0081:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0086:  stloc.1
    IL_0087:  ldarg.0
    IL_0088:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_008d:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0093:  ldarg.0
    IL_0094:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S1 As Integer"
    IL_0099:  ldloca.s   V_1
    IL_009b:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00a0:  ldloca.s   V_1
    IL_00a2:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a8:  call       "Sub Program.M(ByRef Integer, Integer)"
    IL_00ad:  ldarg.0
    IL_00ae:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_00b3:  ldarg.0
    IL_00b4:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As Integer"
    IL_00b9:  ldarg.0
    IL_00ba:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S1 As Integer"
    IL_00bf:  constrained. "SM$T"
    IL_00c5:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00ca:  leave.s    IL_00ee
  }
  catch System.Exception
  {
    IL_00cc:  dup
    IL_00cd:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00d2:  stloc.3
    IL_00d3:  ldarg.0
    IL_00d4:  ldc.i4.s   -2
    IL_00d6:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00db:  ldarg.0
    IL_00dc:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00e1:  ldloc.3
    IL_00e2:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00e7:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00ec:  leave.s    IL_0103
  }
  IL_00ee:  ldarg.0
  IL_00ef:  ldc.i4.s   -2
  IL_00f1:  dup
  IL_00f2:  stloc.0
  IL_00f3:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00f8:  ldarg.0
  IL_00f9:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00fe:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_0103:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Class_IndexAndValue_Async_03()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        M(item.Position(await GetOffsetAsync(GetOffset(item))), GetOffset(item))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        M(item.Position(await GetOffsetAsync(GetOffset(item))), GetOffset(item))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      270 (0x10e)
  .maxstack  4
  .locals init (Integer V_0,
                Integer V_1,
                Integer V_2,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_3,
                SM$T V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0063
    IL_000a:  ldarg.0
    IL_000b:  ldarg.0
    IL_000c:  ldarg.0
    IL_000d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0012:  dup
    IL_0013:  stloc.s    V_4
    IL_0015:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_001a:  ldloc.s    V_4
    IL_001c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0021:  ldarg.0
    IL_0022:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0027:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_002c:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0031:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0036:  stloc.3
    IL_0037:  ldloca.s   V_3
    IL_0039:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_003e:  brtrue.s   IL_007f
    IL_0040:  ldarg.0
    IL_0041:  ldc.i4.0
    IL_0042:  dup
    IL_0043:  stloc.0
    IL_0044:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0049:  ldarg.0
    IL_004a:  ldloc.3
    IL_004b:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0050:  ldarg.0
    IL_0051:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0056:  ldloca.s   V_3
    IL_0058:  ldarg.0
    IL_0059:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_005e:  leave      IL_010d
    IL_0063:  ldarg.0
    IL_0064:  ldc.i4.m1
    IL_0065:  dup
    IL_0066:  stloc.0
    IL_0067:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_006c:  ldarg.0
    IL_006d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0072:  stloc.3
    IL_0073:  ldarg.0
    IL_0074:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0079:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007f:  ldarg.0
    IL_0080:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0085:  box        "SM$T"
    IL_008a:  ldloca.s   V_3
    IL_008c:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0091:  ldloca.s   V_3
    IL_0093:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0099:  dup
    IL_009a:  stloc.1
    IL_009b:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_00a0:  stloc.2
    IL_00a1:  ldloca.s   V_2
    IL_00a3:  ldarg.0
    IL_00a4:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_00a9:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_00ae:  call       "Sub Program.M(ByRef Integer, Integer)"
    IL_00b3:  ldarg.0
    IL_00b4:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_00b9:  ldloc.1
    IL_00ba:  ldloc.2
    IL_00bb:  constrained. "SM$T"
    IL_00c1:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00c6:  ldarg.0
    IL_00c7:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_00cc:  initobj    "SM$T"
    IL_00d2:  leave.s    IL_00f8
  }
  catch System.Exception
  {
    IL_00d4:  dup
    IL_00d5:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00da:  stloc.s    V_5
    IL_00dc:  ldarg.0
    IL_00dd:  ldc.i4.s   -2
    IL_00df:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00e4:  ldarg.0
    IL_00e5:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00ea:  ldloc.s    V_5
    IL_00ec:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00f1:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00f6:  leave.s    IL_010d
  }
  IL_00f8:  ldarg.0
  IL_00f9:  ldc.i4.s   -2
  IL_00fb:  dup
  IL_00fc:  stloc.0
  IL_00fd:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0102:  ldarg.0
  IL_0103:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0108:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_010d:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      315 (0x13b)
  .maxstack  3
  .locals init (Integer V_0,
                Integer V_1,
                Integer V_2,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_3,
                SM$T V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0069
    IL_000a:  ldloca.s   V_4
    IL_000c:  initobj    "SM$T"
    IL_0012:  ldloc.s    V_4
    IL_0014:  box        "SM$T"
    IL_0019:  brtrue.s   IL_0027
    IL_001b:  ldarg.0
    IL_001c:  ldarg.0
    IL_001d:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0022:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_0027:  ldarg.0
    IL_0028:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_002d:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0032:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0037:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_003c:  stloc.3
    IL_003d:  ldloca.s   V_3
    IL_003f:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0044:  brtrue.s   IL_0085
    IL_0046:  ldarg.0
    IL_0047:  ldc.i4.0
    IL_0048:  dup
    IL_0049:  stloc.0
    IL_004a:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_004f:  ldarg.0
    IL_0050:  ldloc.3
    IL_0051:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0056:  ldarg.0
    IL_0057:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_005c:  ldloca.s   V_3
    IL_005e:  ldarg.0
    IL_005f:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_0064:  leave      IL_013a
    IL_0069:  ldarg.0
    IL_006a:  ldc.i4.m1
    IL_006b:  dup
    IL_006c:  stloc.0
    IL_006d:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0072:  ldarg.0
    IL_0073:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0078:  stloc.3
    IL_0079:  ldarg.0
    IL_007a:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007f:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0085:  ldloca.s   V_4
    IL_0087:  initobj    "SM$T"
    IL_008d:  ldloc.s    V_4
    IL_008f:  box        "SM$T"
    IL_0094:  brtrue.s   IL_009e
    IL_0096:  ldarg.0
    IL_0097:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_009c:  br.s       IL_00a4
    IL_009e:  ldarg.0
    IL_009f:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00a4:  ldloca.s   V_3
    IL_00a6:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_00ab:  ldloca.s   V_3
    IL_00ad:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00b3:  dup
    IL_00b4:  stloc.1
    IL_00b5:  constrained. "SM$T"
    IL_00bb:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_00c0:  stloc.2
    IL_00c1:  ldloca.s   V_2
    IL_00c3:  ldarg.0
    IL_00c4:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00c9:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_00ce:  call       "Sub Program.M(ByRef Integer, Integer)"
    IL_00d3:  ldloca.s   V_4
    IL_00d5:  initobj    "SM$T"
    IL_00db:  ldloc.s    V_4
    IL_00dd:  box        "SM$T"
    IL_00e2:  brtrue.s   IL_00ec
    IL_00e4:  ldarg.0
    IL_00e5:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_00ea:  br.s       IL_00f2
    IL_00ec:  ldarg.0
    IL_00ed:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_00f2:  ldloc.1
    IL_00f3:  ldloc.2
    IL_00f4:  constrained. "SM$T"
    IL_00fa:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00ff:  leave.s    IL_0125
  }
  catch System.Exception
  {
    IL_0101:  dup
    IL_0102:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_0107:  stloc.s    V_5
    IL_0109:  ldarg.0
    IL_010a:  ldc.i4.s   -2
    IL_010c:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0111:  ldarg.0
    IL_0112:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0117:  ldloc.s    V_5
    IL_0119:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_011e:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0123:  leave.s    IL_013a
  }
  IL_0125:  ldarg.0
  IL_0126:  ldc.i4.s   -2
  IL_0128:  dup
  IL_0129:  stloc.0
  IL_012a:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_012f:  ldarg.0
  IL_0130:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_0135:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_013a:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Struct_IndexAndValue_Async_03()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Structure Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Structure

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Structure, IMoveable})(item As T) As Task
        M(item.Position(await GetOffsetAsync(GetOffset(item))), GetOffset(item))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        M(item.Position(await GetOffsetAsync(GetOffset(item))), GetOffset(item))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '-1'
Position set for item '-2'
Position get for item '-3'
Position set for item '-4'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
  // Code size      236 (0xec)
  .maxstack  3
  .locals init (Integer V_0,
                Integer V_1,
                Integer V_2,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_3,
                System.Exception V_4)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_004c
    IL_000a:  ldarg.0
    IL_000b:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0010:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0015:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_001a:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_001f:  stloc.3
    IL_0020:  ldloca.s   V_3
    IL_0022:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0027:  brtrue.s   IL_0068
    IL_0029:  ldarg.0
    IL_002a:  ldc.i4.0
    IL_002b:  dup
    IL_002c:  stloc.0
    IL_002d:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0032:  ldarg.0
    IL_0033:  ldloc.3
    IL_0034:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0039:  ldarg.0
    IL_003a:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_003f:  ldloca.s   V_3
    IL_0041:  ldarg.0
    IL_0042:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0047:  leave      IL_00eb
    IL_004c:  ldarg.0
    IL_004d:  ldc.i4.m1
    IL_004e:  dup
    IL_004f:  stloc.0
    IL_0050:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0055:  ldarg.0
    IL_0056:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_005b:  stloc.3
    IL_005c:  ldarg.0
    IL_005d:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0062:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0068:  ldarg.0
    IL_0069:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_006e:  ldloca.s   V_3
    IL_0070:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0075:  ldloca.s   V_3
    IL_0077:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007d:  dup
    IL_007e:  stloc.1
    IL_007f:  constrained. "SM$T"
    IL_0085:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_008a:  stloc.2
    IL_008b:  ldloca.s   V_2
    IL_008d:  ldarg.0
    IL_008e:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0093:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0098:  call       "Sub Program.M(ByRef Integer, Integer)"
    IL_009d:  ldarg.0
    IL_009e:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_00a3:  ldloc.1
    IL_00a4:  ldloc.2
    IL_00a5:  constrained. "SM$T"
    IL_00ab:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_00b0:  leave.s    IL_00d6
  }
  catch System.Exception
  {
    IL_00b2:  dup
    IL_00b3:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_00b8:  stloc.s    V_4
    IL_00ba:  ldarg.0
    IL_00bb:  ldc.i4.s   -2
    IL_00bd:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00c2:  ldarg.0
    IL_00c3:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00c8:  ldloc.s    V_4
    IL_00ca:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_00cf:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_00d4:  leave.s    IL_00eb
  }
  IL_00d6:  ldarg.0
  IL_00d7:  ldc.i4.s   -2
  IL_00d9:  dup
  IL_00da:  stloc.0
  IL_00db:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_00e0:  ldarg.0
  IL_00e1:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_00e6:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_00eb:  ret
}
]]>)
            CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.DebugExe, expectedOutput:=expectedOutput).VerifyDiagnostics()
        End Sub

        <Fact>
        <WorkItem(63221, "https://github.com/dotnet/roslyn/issues/63221")>
        Public Sub GenericTypeParameterAsReceiver_CopyBack_Indexer_Class_IndexAndValue_Async_04()
            Dim comp =
<compilation>
    <file>
Imports System
Imports System.Threading.Tasks

Interface IMoveable
    Property Position(x As Integer) As Integer
End Interface

Class Item
    Implements IMoveable

    Public Property Name As String

    Public Property Position(x As Integer) As Integer Implements IMoveable.Position
        Get
            Console.WriteLine("Position get for item '{0}'", Me.Name)
            Return 0
        End Get
        Set
            Console.WriteLine("Position set for item '{0}'", Me.Name)
        End Set
    End Property
End Class

Class Program
    Shared Sub Main()
        Dim item1 = New Item With {.Name = "1"}
        Call1(item1).Wait()

        Dim item2 = New Item With {.Name = "2"}
        Call2(item2).Wait()
    End Sub

    Private Shared Async Function Call1(Of T As {Class, IMoveable})(item As T) As Task
        M(item.Position(await GetOffsetAsync(GetOffset(item))), await GetOffsetAsync(GetOffset(item)))
    End Function

    Private Shared Async Function Call2(Of T As {IMoveable})(item As T) As Task
        M(item.Position(await GetOffsetAsync(GetOffset(item))), await GetOffsetAsync(GetOffset(item)))
    End Function

    Shared value as Integer

    Shared Function GetOffset(Of T)(ByRef item As T) As Integer
        value -= 1
        item = DirectCast(DirectCast(New Item With {.Name = value.ToString()}, IMoveable), T)
        Return 0
    End Function

    Shared Async Function GetOffsetAsync(i As Integer) As Task(Of Integer)
        await Task.Yield()
        Return i
    End Function

    Shared Sub M(ByRef x As Integer, y As Integer)
        x += y
    End Sub
End Class
    </file>
</compilation>

            Dim expectedOutput =
"
Position get for item '1'
Position set for item '1'
Position get for item '2'
Position set for item '2'
"
            Dim verifier = CompileAndVerifyEx(comp, targetFramework:=TargetFramework.StandardAndVBRuntime, options:=TestOptions.ReleaseExe, expectedOutput:=expectedOutput).VerifyDiagnostics()

            verifier.VerifyIL("Program.VB$StateMachine_2_Call1(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      400 (0x190)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T V_3,
                Integer V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_0068
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_00f6
    IL_0011:  ldarg.0
    IL_0012:  ldarg.0
    IL_0013:  ldarg.0
    IL_0014:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_0019:  dup
    IL_001a:  stloc.3
    IL_001b:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0020:  ldloc.3
    IL_0021:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_0026:  ldarg.0
    IL_0027:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_002c:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0031:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_0036:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_003b:  stloc.2
    IL_003c:  ldloca.s   V_2
    IL_003e:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_0043:  brtrue.s   IL_0084
    IL_0045:  ldarg.0
    IL_0046:  ldc.i4.0
    IL_0047:  dup
    IL_0048:  stloc.0
    IL_0049:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_004e:  ldarg.0
    IL_004f:  ldloc.2
    IL_0050:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0055:  ldarg.0
    IL_0056:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_005b:  ldloca.s   V_2
    IL_005d:  ldarg.0
    IL_005e:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_0063:  leave      IL_018f
    IL_0068:  ldarg.0
    IL_0069:  ldc.i4.m1
    IL_006a:  dup
    IL_006b:  stloc.0
    IL_006c:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0071:  ldarg.0
    IL_0072:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0077:  stloc.2
    IL_0078:  ldarg.0
    IL_0079:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007e:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0084:  ldarg.0
    IL_0085:  ldarg.0
    IL_0086:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_008b:  box        "SM$T"
    IL_0090:  ldarg.0
    IL_0091:  ldloca.s   V_2
    IL_0093:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_0098:  ldloca.s   V_2
    IL_009a:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00a0:  dup
    IL_00a1:  stloc.s    V_4
    IL_00a3:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S1 As Integer"
    IL_00a8:  ldloc.s    V_4
    IL_00aa:  callvirt   "Function IMoveable.get_Position(Integer) As Integer"
    IL_00af:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S2 As Integer"
    IL_00b4:  ldarg.0
    IL_00b5:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$VB$Local_item As SM$T"
    IL_00ba:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_00bf:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_00c4:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00c9:  stloc.1
    IL_00ca:  ldloca.s   V_1
    IL_00cc:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_00d1:  brtrue.s   IL_0112
    IL_00d3:  ldarg.0
    IL_00d4:  ldc.i4.1
    IL_00d5:  dup
    IL_00d6:  stloc.0
    IL_00d7:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00dc:  ldarg.0
    IL_00dd:  ldloc.1
    IL_00de:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_00e3:  ldarg.0
    IL_00e4:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_00e9:  ldloca.s   V_1
    IL_00eb:  ldarg.0
    IL_00ec:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_2_Call1(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_2_Call1(Of SM$T))"
    IL_00f1:  leave      IL_018f
    IL_00f6:  ldarg.0
    IL_00f7:  ldc.i4.m1
    IL_00f8:  dup
    IL_00f9:  stloc.0
    IL_00fa:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_00ff:  ldarg.0
    IL_0100:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0105:  stloc.1
    IL_0106:  ldarg.0
    IL_0107:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_010c:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0112:  ldarg.0
    IL_0113:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$S2 As Integer"
    IL_0118:  ldloca.s   V_1
    IL_011a:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).GetResult() As Integer"
    IL_011f:  ldloca.s   V_1
    IL_0121:  initobj    "System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0127:  call       "Sub Program.M(ByRef Integer, Integer)"
    IL_012c:  ldarg.0
    IL_012d:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S0 As SM$T"
    IL_0132:  box        "SM$T"
    IL_0137:  ldarg.0
    IL_0138:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S1 As Integer"
    IL_013d:  ldarg.0
    IL_013e:  ldfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$S2 As Integer"
    IL_0143:  callvirt   "Sub IMoveable.set_Position(Integer, Integer)"
    IL_0148:  ldarg.0
    IL_0149:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$U1 As SM$T"
    IL_014e:  initobj    "SM$T"
    IL_0154:  leave.s    IL_017a
  }
  catch System.Exception
  {
    IL_0156:  dup
    IL_0157:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.SetProjectError(System.Exception)"
    IL_015c:  stloc.s    V_5
    IL_015e:  ldarg.0
    IL_015f:  ldc.i4.s   -2
    IL_0161:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
    IL_0166:  ldarg.0
    IL_0167:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_016c:  ldloc.s    V_5
    IL_016e:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetException(System.Exception)"
    IL_0173:  call       "Sub Microsoft.VisualBasic.CompilerServices.ProjectData.ClearProjectError()"
    IL_0178:  leave.s    IL_018f
  }
  IL_017a:  ldarg.0
  IL_017b:  ldc.i4.s   -2
  IL_017d:  dup
  IL_017e:  stloc.0
  IL_017f:  stfld      "Program.VB$StateMachine_2_Call1(Of SM$T).$State As Integer"
  IL_0184:  ldarg.0
  IL_0185:  ldflda     "Program.VB$StateMachine_2_Call1(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
  IL_018a:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.SetResult()"
  IL_018f:  ret
}
]]>)

            verifier.VerifyIL("Program.VB$StateMachine_3_Call2(Of SM$T).MoveNext()",
            <![CDATA[
{
      // Code size      445 (0x1bd)
  .maxstack  5
  .locals init (Integer V_0,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_1,
                System.Runtime.CompilerServices.TaskAwaiter(Of Integer) V_2,
                SM$T V_3,
                Integer V_4,
                System.Exception V_5)
  IL_0000:  ldarg.0
  IL_0001:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
  IL_0006:  stloc.0
  .try
  {
    IL_0007:  ldloc.0
    IL_0008:  brfalse.s  IL_006f
    IL_000a:  ldloc.0
    IL_000b:  ldc.i4.1
    IL_000c:  beq        IL_0116
    IL_0011:  ldloca.s   V_3
    IL_0013:  initobj    "SM$T"
    IL_0019:  ldloc.3
    IL_001a:  box        "SM$T"
    IL_001f:  brtrue.s   IL_002d
    IL_0021:  ldarg.0
    IL_0022:  ldarg.0
    IL_0023:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0028:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$S0 As SM$T"
    IL_002d:  ldarg.0
    IL_002e:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$VB$Local_item As SM$T"
    IL_0033:  call       "Function Program.GetOffset(Of SM$T)(ByRef SM$T) As Integer"
    IL_0038:  call       "Function Program.GetOffsetAsync(Integer) As System.Threading.Tasks.Task(Of Integer)"
    IL_003d:  callvirt   "Function System.Threading.Tasks.Task(Of Integer).GetAwaiter() As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_0042:  stloc.2
    IL_0043:  ldloca.s   V_2
    IL_0045:  call       "Function System.Runtime.CompilerServices.TaskAwaiter(Of Integer).get_IsCompleted() As Boolean"
    IL_004a:  brtrue.s   IL_008b
    IL_004c:  ldarg.0
    IL_004d:  ldc.i4.0
    IL_004e:  dup
    IL_004f:  stloc.0
    IL_0050:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0055:  ldarg.0
    IL_0056:  ldloc.2
    IL_0057:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_005c:  ldarg.0
    IL_005d:  ldflda     "Program.VB$StateMachine_3_Call2(Of SM$T).$Builder As System.Runtime.CompilerServices.AsyncTaskMethodBuilder"
    IL_0062:  ldloca.s   V_2
    IL_0064:  ldarg.0
    IL_0065:  call       "Sub System.Runtime.CompilerServices.AsyncTaskMethodBuilder.AwaitUnsafeOnCompleted(Of System.Runtime.CompilerServices.TaskAwaiter(Of Integer), Program.VB$StateMachine_3_Call2(Of SM$T))(ByRef System.Runtime.CompilerServices.TaskAwaiter(Of Integer), ByRef Program.VB$StateMachine_3_Call2(Of SM$T))"
    IL_006a:  leave      IL_01bc
    IL_006f:  ldarg.0
    IL_0070:  ldc.i4.m1
    IL_0071:  dup
    IL_0072:  stloc.0
    IL_0073:  stfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$State As Integer"
    IL_0078:  ldarg.0
    IL_0079:  ldfld      "Program.VB$StateMachine_3_Call2(Of SM$T).$A0 As System.Runtime.CompilerServices.TaskAwaiter(Of Integer)"
    IL_007e:  stloc.